/* global window document */
import { h } from './element.js';
import { cssPrefix } from '../config.js';
import Button from './button.js';
import { Draw } from '../canvas/draw.js';
import { renderCell } from './table.js';
import { t } from '../locale/locale.js';

// resolution: 72 => 595 x 842
// 150 => 1240 x 1754
// 200 => 1654 x 2339
// 300 => 2479 x 3508
// 96 * cm / 2.54 , 96 * cm / 2.54

const PAGER_SIZES = [
  ['A3', 11.69, 16.54],
  ['A4', 8.27, 11.69],
  ['A5', 5.83, 8.27],
  ['B4', 9.84, 13.90],
  ['B5', 6.93, 9.84],
];

const PAGER_ORIENTATIONS = ['portrait', 'landscape'];

function inches2px(inc) {
  return parseInt(96 * inc, 10);
}

function btnClick(type) {
  if (type === 'cancel') {
    this.el.hide();
    top.cancelPrint();
  } else {
    this.toPrint();
  }
}

function pagerSizeChange(evt) {
  const { paper } = this;
  const { value } = evt.target;
  const ps = PAGER_SIZES[value];
  paper.w = inches2px(ps[1]);
  paper.h = inches2px(ps[2]);
  // console.log('paper:', ps, paper);
  this.preview();
}
function pagerOrientationChange(evt) {
  const { paper } = this;
  const { value } = evt.target;
  const v = PAGER_ORIENTATIONS[value];
  paper.orientation = v;
  this.preview();
}

export default class Print {
  constructor(data) {
    this.paper = {
      w: inches2px(PAGER_SIZES[1][1]),
      h: inches2px(PAGER_SIZES[1][2]),
      padding: 50,
      orientation: PAGER_ORIENTATIONS[0],
      get width() {
        return this.orientation === 'landscape' ? this.h : this.w;
      },
      get height() {
        return this.orientation === 'landscape' ? this.w : this.h;
      }
    };
    this.data = data;
    this.el = h('div', `${cssPrefix}-print`)
      .children(
        h('div', `${cssPrefix}-print-bar`)
          .children(
            h('div', '-title').child('打印预览'),
            h('div', '-right').children(
              h('div', `${cssPrefix}-buttons`).children(
                new Button('cancel').on('click', btnClick.bind(this, 'cancel')),
                new Button('next', 'primary').on('click', btnClick.bind(this, 'next')),
              ),
            ),
          ),
        h('div', `${cssPrefix}-print-content`)
          .children(
            this.contentEl = h('div', '-content'),
            /*h('div', '-sider').child(
              h('form', '').children(
                h('fieldset', '').children(
                  h('label', '').child(`${t('print.size')}`),
                  h('select', '').children(
                    ...PAGER_SIZES.map((it, index) => h('option', '').attr('value', index).child(`${it[0]} ( ${it[1]}''x${it[2]}'' )`)),
                  ).on('change', pagerSizeChange.bind(this)),
                ),
                h('fieldset', '').children(
                  h('label', '').child(`${t('print.orientation')}`),
                  h('select', '').children(
                    ...PAGER_ORIENTATIONS.map((it, index) => h('option', '').attr('value', index).child(`${t('print.orientations')[index]}`)),
                  ).on('change', pagerOrientationChange.bind(this)),
                ),
              ),
            ),*/
          ),
      ).hide();
  }

  resetData(data) {
    this.data = data;
  }

  preview() {
    if(this.pageSetting.bgImage) {
      var that = this;
      var imgBg = new Image();
      imgBg.src = this.pageSetting.bgImage;
      imgBg.onload = function() {
        that.render(imgBg);
      }
    } else {
      this.render();
    }
  }

  getPageCount(pageHeight, start, end) {
    let ph = 0;
    let count = 0;
    for(let ri=start;ri<end;ri++) {
      let rh = this.data.rows.getHeight(ri);
      rh = parseInt(rh*window.devicePixelRatio);
      if ((ph+rh) < pageHeight) {
        ph += rh;
      } else {
        ph = rh;
        count++;
      }
    }
    return count+1;
  }

  render(imgBg) {
    const { data, paper, pageSetting } = this;
    if(pageSetting && (pageSetting.pageSize || pageSetting.pageSize==0) && pageSetting.pageSize!=1) {
      if((pageSetting.pageSize+"")=="-1") {
        paper.w = inches2px(pageSetting.pageWidth);
        paper.h = inches2px(pageSetting.pageHeight);
      } else {
        paper.w = inches2px(PAGER_SIZES[pageSetting.pageSize][1]);
        paper.h = inches2px(PAGER_SIZES[pageSetting.pageSize][2]);
      }
    }
    if(pageSetting && pageSetting.orientation=="landscape") {
      paper.orientation = "landscape";
    }
    const { width, height, padding } = paper;
    let left = padding;
    let top = padding;
    if(pageSetting && (pageSetting.paddingLeft || pageSetting.paddingLeft==0)) left = pageSetting.paddingLeft;
    if(pageSetting && (pageSetting.paddingTop || pageSetting.paddingTop==0)) top = pageSetting.paddingTop;

    this.paddingLeft = left;
    this.paddingTop = top;

    const iwidth = (width - left * 2) * window.devicePixelRatio;
    const iheight = (height - top * 2) * window.devicePixelRatio;
    const cr = data.contentRange();
    
    //let pageCount = parseInt((cr.h-19)*window.devicePixelRatio / (iheight), 10);
    //if((cr.h - 19 - iheight*pageCount)>0) pageCount += 1;

    let scale = iwidth / cr.w;
    scale = 1;
    if (scale > 1) {
      //left += (iwidth - cr.w) / 2;
    }
    let ri = 0;
    let yoffset = 0;
    this.contentEl.html('');
    this.canvases = [];
    const mViewRange = {
      sri: 0,
      sci: 0,
      eri: 0,
      eci: 0,
    };

    let pageCount = 0;
    if(this.pagesInfo && this.pagesInfo.length>0) {
      pageCount = this.pagesInfo.length;
    } else {
      pageCount = this.getPageCount(iheight, cr.sri, cr.eri);
    }

    for (let i = 0; i < pageCount; i += 1) {
      let th = 0;
      let th2 = 0;
      let yo = 0;
      
      const wrap = h('div', `${cssPrefix}-canvas-card`);
      wrap.css("width", width+"px");
      const canvas = h('canvas', `${cssPrefix}-canvas`);
      this.canvases.push(canvas.el);
      const draw = new Draw(canvas.el, width, height);
      
      if(this.pageSetting.bgImage) draw.ctx.drawImage(imgBg, left, top, imgBg.width, imgBg.height);
      
      draw.save();
      draw.translate(left, top);

      let ri2 = 0;
      let curPageStartRi = ri;
      let curPageEndRi = cr.eri;
      let pageInfo = null;
      if(this.pagesInfo && this.pagesInfo.length>0) {
        pageInfo = this.pagesInfo[i];
        ri=pageInfo[0];
        curPageEndRi = pageInfo[1];
      }
      for(;ri<=curPageEndRi;ri++) {
        let rh = data.rows.getHeight(ri);
        let rh2 = rh;
        rh = parseInt(rh*window.devicePixelRatio);
        th += rh;
        th2 += rh2;
        if(pageInfo) {
          for (let ci = 0; ci <= cr.eci; ci += 1) {
            renderCell(draw, data, ri, ci, yoffset);
            mViewRange.eci = ci;
          }
          yo = -(th2);
          ri2 = ri - 1;
        } else {
          if (th < iheight) {
            for (let ci = 0; ci <= cr.eci; ci += 1) {
              renderCell(draw, data, ri, ci, yoffset);
              mViewRange.eci = ci;
            }
            ri2 = ri;
          } else {
            yo = -(th2 - rh2);
            ri2 = ri - 1;
            break;
          }
        }
      }
      mViewRange.eri = ri2;
      draw.restore();
      draw.save();
      draw.translate(left, top);
      //if (scale < 1) draw.scale(scale, scale);
      let yof = yoffset;
      data.eachMergesInView(mViewRange, ({ sri, sci }) => {
        //renderCell(draw, data, sri, sci, yof);
      });
      draw.restore();

      //Draw Image
      if(this.images) {
        for(let j=0; j<this.images.length;j++) {
          let img = this.images[j];
          let imgTop = img.top;
          if(pageInfo) {
            if(img.ri<pageInfo[0] || img.ri>pageInfo[1]) continue;
            for(let riImg=pageInfo[0];riImg<img.ri;riImg++) {
              imgTop += data.rows.getHeight(riImg);
            }
          } else {
            if(img.ri<curPageStartRi || img.ri>ri2) continue;
            imgTop += this.getRowOffsetTop(img.ri, curPageStartRi);
          }
          let imgId = "imgCell_"+img.ri+"_"+img.ci;
          draw.ctx.drawImage($("#"+imgId).get(0), 
            parseInt((img.left+left-60)*window.devicePixelRatio),
            parseInt((imgTop+top)*window.devicePixelRatio),
            parseInt(img.width*window.devicePixelRatio),
            parseInt(img.height*window.devicePixelRatio)
          );
          //$("#"+imgId).hide();
        }
      }

      if(this.qrCodes) {
        for(let j=0; j<this.qrCodes.length;j++) {
          let qrCode = this.qrCodes[j];
          let codeTop = qrCode.top;
          if(pageInfo) {
            if(qrCode.ri<pageInfo[0] || qrCode.ri>pageInfo[1]) continue;
            for(let riCode=pageInfo[0];riCode<qrCode.ri;riCode++) {
              codeTop += data.rows.getHeight(riCode);
            }
          } else {
            if(qrCode.ri<curPageStartRi || qrCode.ri>ri2) continue;
            codeTop += this.getRowOffsetTop(qrCode.ri, curPageStartRi);
          }
          let codeId = "qrCell_"+qrCode.ri+"_"+qrCode.ci;
          draw.ctx.drawImage($("#"+codeId).find("img").get(0), 
            parseInt((qrCode.left+left-60)*window.devicePixelRatio),
            parseInt((codeTop+top)*window.devicePixelRatio),
            parseInt(qrCode.width*window.devicePixelRatio),
            parseInt(qrCode.height*window.devicePixelRatio)
          );
        }
      }

      mViewRange.sri = mViewRange.eri;
      mViewRange.sci = 0;
      yoffset += yo;
      this.contentEl.child(h('div', `${cssPrefix}-canvas-card-wraper`).child(wrap.child(canvas)));
    }
    this.el.show();
  }

  getRowOffsetTop(ri, curPageStartRi) {
    let top = 0;
    for(var i=curPageStartRi;i<ri;i++) {
      top += this.data.rows.getHeight(i);
    }
    return top;
  }

  //${paper.width}px ${paper.height}px;

  toPrint() {
    this.el.hide();
    const { paper } = this;
    const iframe = h('iframe', '').hide();
    const { el } = iframe;
    window.document.body.appendChild(el);
    const { contentWindow } = el;
    const idoc = contentWindow.document;
    const style = document.createElement('style');
    style.innerHTML = `
      @page { 
        size: ${this.pageSetting.pageWidth}in ${this.pageSetting.pageHeight}in ${this.pageSetting.orientation}; 
        margin: 0;
      };
      canvas {
        image-rendering: pixelated;
      };
      body {
        padding: 0;
        margin: 0;
        overflow: hidden;
      }
    `;
    idoc.head.appendChild(style);
    let that = this;
    let index = 0;
    this.canvases.forEach((it) => {
      const cn = it.cloneNode(false);
      //只有第一页之后的，需要添加分页符
      if(index>0) {
        $(cn).css("page-break-before", "always");
      }
      const ctx = cn.getContext('2d');
      // ctx.imageSmoothingEnabled = true;
      //ctx.translate(that.paddingLeft, that.paddingTop);
      //ctx.translate(0, 0);
      ctx.drawImage(it, 0, 0);
      idoc.body.appendChild(cn);
      index++;
    });
    contentWindow.print();
  }
}
