import { FormInput } from "../controller.js";

/**
 * 进度条组件
 * @param {[type]} dataModel [description]
 */
export default class Progress extends FormInput {
  constructor(__tfp, dataModel, parent, inputType) {
    if (inputType) {
      super(__tfp, inputType, dataModel, parent);
    } else {
      super(__tfp, "Progress", dataModel, parent);
    }
  }

  get value() {
    return this.dataModel.value || 0;
  }
  set value(value) {
    let val = value;
    if (isNull(value)) {
      val = 0;
    }
    this.dataModel.value = val;

    if (this._jqObj) {
      this.reRender();
    }

    if (!this._tfp.isDesigning) {
      this.valueOnChange();
      this.exeEventHandler("onChange", value);
    }
  }

  get dataType() {
    if (!this.dataModel.dataType) this.dataModel.dataType = "text";
    return this.dataModel.dataType;
  }
  set dataType(value) {
    this.dataModel.dataType = value;

    if (this._jqObj) {
      this.reRender();
    }

    if (!this._tfp.isDesigning) {
      this.exeEventHandler("onDatatypeChange", value);
    }
  }

  get circle() { return this.dataModel.circle }
  set circle(value) {
    this.dataModel.circle = value ? true : false;
    if (this._jqObj) {
      this.reRender();
    }
  }

  get diam() { return this.dataModel.diam }
  set diam(value) {
    this.dataModel.diam = value;
    if (this._jqObj) {
      this.reRender();
      if (this._tfp.isDesigning) {
        uiDesigner.setSelectedCptStyle("width", value + "px");
        uiDesigner.setSelectedCptStyle("height", value + "px");
        window.parent.$("input[style-setting-id=width]").val(value);
        window.parent.$("input[style-setting-id=height]").val(value);
      }
    }
  }

  reRender() {
    this._jqObj.empty();
    this._jqObj.append(this.getBodyHtml());
    if (this.dataModel.circle) this.circleStyle();
  }

  getBodyHtml(getComponentsHtml) {
    let bodyHtml = "";
    let indent = this.getHtmlIndent();
    let val = 0;
    let bgColor = "#6666FF";
    let diam = 200;
    if (this.dataModel.value) {
      val = this.dataModel.value;
    }
    if (this.dataModel.dataType) {
      switch (this.dataModel.dataType) {
        case 'success':
          bgColor = '#67c23a';
          break;
        case 'warning':
          bgColor = '#e6a23c';
          break;
        case 'exception':
          bgColor = '#f56c6c';
          break;
        default:
          break;
      }
    }
    if (this.dataModel.diam) {
      diam = this.dataModel.diam;
    }
    if (this.dataModel.circle) {
      bodyHtml += indent + `<canvas id="canvas" width="${diam}" height="${diam}"></canvas>`;
    } else {
      bodyHtml += indent + `<div class="tfp-progress-bar">
                     <div class="tfp-progress-bar__outer">
                       <div class="tfp-progress-bar__inner" style="width: ${val}%;background-color:${bgColor};"></div>
                     </div>
                  </div>`;
      bodyHtml += indent + `<div class="tfp-progress__text" style="font-size: 14.4px;">${val}%</div>`
    }

    return bodyHtml;
  }

  circleStyle() {
    let that = this;
    let c = that._jqObj.find('#canvas')[0];
    let ctx = c.getContext('2d');
    let precent = that.dataModel.value;
    let dif = this.dataModel.diam / 300;
    drawTrack();
    drawProgress(precent);
    drawText(precent);

    //绘制轨道
    function drawTrack() {
      ctx.save();
      ctx.beginPath();
      ctx.lineCap = 'round';
      ctx.lineWidth = 12 * dif;
      ctx.strokeStyle = '#ebeef5';
      ctx.arc(150 * dif, 150 * dif, 140 * dif, 0, 2 * Math.PI);
      ctx.stroke();
      ctx.closePath();
      ctx.restore();
    }

    //绘制进度环
    function drawProgress(num) {
      ctx.save();
      ctx.beginPath();
      ctx.lineCap = 'round';
      ctx.lineWidth = 12 * dif;
      ctx.strokeStyle = '#6666FF';
      switch (that.dataModel.dataType) {
        case 'success':
          ctx.strokeStyle = '#67c23a';
          break;
        case 'warning':
          ctx.strokeStyle = '#e6a23c';
          break;
        case 'exception':
          ctx.strokeStyle = '#f56c6c';
          break;
        default:
          break;
      }
      ctx.arc(150 * dif, 150 * dif, 140 * dif, -Math.PI / 2, -Math.PI / 2 + 2 * num / 100 * Math.PI);
      ctx.stroke();
      ctx.closePath();
      ctx.restore();
    }

    //绘制文字
    function drawText(num) {
      let textSize = 40 * dif;
      ctx.save();
      ctx.fillStyle = '#606266';
      ctx.font = '' + textSize + 'px Helvetica';
      ctx.textAlign = 'center';
      ctx.fillText(num + '%', 150 * dif, 160 * dif);
      ctx.restore();
    }
  }

  initRuntime() {
    if(!this.isRendered) this.reRender();
  }

  initDesigning() {
    if (this.dataModel.circle) this.circleStyle();
  }
}
