import { ContainerComponent } from "../controller.js";

/**
 * 打印报表页组件
 * @param {[type]} dataModel [description]
 */
export default class PrintReportPage extends ContainerComponent {

  constructor(__tfp, dataModel, parent) {
    super(__tfp, "PrintReportPage", dataModel, parent);
  }

  resetPageSize() {
    let _pageWidth = 794;
    let _pageHeight = 1123;
    if(this.dataModel.pageSize=="Custom") {
      if(this.dataModel.pageWidth) _pageWidth = this.mmToPx(this.dataModel.pageWidth);
      if(this.dataModel.pageHeight) _pageHeight = this.mmToPx(this.dataModel.pageHeight);
    } else {
      const PAGER_SIZES = {
        "A3": {
            "width": 297, 
            "height": 420
        },
        "A4": { 
            "width": 210, 
            "height": 297
        },
        "A5": {
            "width": 148, 
            "height": 210
        },
        "B4": {
            "width": 250,
            "height": 352
        },
        "B5": {
            "width": 176, 
            "height": 250
        }
      };
      let _pageSize = PAGER_SIZES[this.dataModel.pageSize];
      if(this.dataModel.orientation == "portrait") {
        _pageWidth = _pageSize.width;
        _pageHeight = _pageSize.height;
      } else {
        _pageWidth = _pageSize.height;
        _pageHeight = _pageSize.width;
      }
      _pageWidth = this.mmToPx(_pageWidth);
      _pageHeight = this.mmToPx(_pageHeight);
    }
    this.dataModel.styles["width"] = _pageWidth+"px";
    this.dataModel.styles["height"] = _pageHeight+"px";
    if(this._jqObj) {
      this._jqObj.css("width", _pageWidth+"px");
      this._jqObj.css("height", _pageHeight+"px");
    }
  }

  get pageSize() { return this.dataModel.pageSize }
  set pageSize(value) {
    this.dataModel.pageSize = value;
    if(!value) this.dataModel.pageSize = "A4";
    if(this.dataModel.pageSize=="Custom") {
      this.dataModel.pageWidth = this.pxToMm(this._jqObj.get(0).offsetWidth);
      this.dataModel.pageHeight = this.pxToMm(this._jqObj.get(0).offsetHeight);
      if(this._tfp.isDesigning) {
        window.parent.$(".cpt-attr[attr-name=pageWidth]").find("input").val(this.dataModel.pageWidth);
        window.parent.$(".cpt-attr[attr-name=pageHeight]").find("input").val(this.dataModel.pageHeight);
      }
    } else {
      delete this.dataModel.pageWidth;
      delete this.dataModel.pageHeight;
      if(this._tfp.isDesigning) {
        window.parent.$(".cpt-attr[attr-name=pageWidth]").find("input").val("");
        window.parent.$(".cpt-attr[attr-name=pageHeight]").find("input").val("");
      }
    }
    this.resetPageSize();
  }

  get orientation() { return this.dataModel.orientation }
  set orientation(value) {
    this.dataModel.orientation = value;
    if(!value) this.dataModel.orientation = "portrait";
    this.resetPageSize();
  }

  get pageWidth() { return this.dataModel.pageWidth }
  set pageWidth(value) {
    this.dataModel.pageWidth = value;
    this.resetPageSize();
  }

  get pageHeight() { return this.dataModel.pageHeight }
  set pageHeight(value) {
    this.dataModel.pageHeight = value;
    this.resetPageSize();
  }

  get paddingTop() { return this.dataModel.paddingTop }
  set paddingTop(value) {
    this.dataModel.paddingTop = value;
    if(!value) value = 5;
    this.dataModel.styles["padding-top"] = this.mmToPx(value)+"px";
    if(this._jqObj) this._jqObj.css("padding-top", this.mmToPx(value)+"px");
  }

  get paddingBottom() { return this.dataModel.paddingBottom }
  set paddingBottom(value) {
    this.dataModel.paddingBottom = value;
    if(!value) value = 5;
    this.dataModel.styles["padding-bottom"] = this.mmToPx(value)+"px";
    if(this._jqObj) this._jqObj.css("padding-bottom", this.mmToPx(value)+"px");
  }

  get paddingLeft() { return this.dataModel.paddingLeft }
  set paddingLeft(value) {
    this.dataModel.paddingLeft = value;
    if(!value) value = 5;
    this.dataModel.styles["padding-left"] = this.mmToPx(value)+"px";
    if(this._jqObj) this._jqObj.css("padding-left", this.mmToPx(value)+"px");
  }

  get paddingRight() { return this.dataModel.paddingRight }
  set paddingRight(value) {
    this.dataModel.paddingRight = value;
    if(!value) value = 5;
    this.dataModel.styles["padding-right"] = this.mmToPx(value)+"px";
    if(this._jqObj) this._jqObj.css("padding-right", this.mmToPx(value)+"px");
  }

  get pageHeaderPanel() { return this._tfp.components[this.id+"_page_header"] }
  set pageHeaderPanel(value) {}

  get pageHeaderHeight() { return this.dataModel.pageHeaderHeight }
  set pageHeaderHeight(value) {
    this.dataModel.pageHeaderHeight = parseInt(value);
    let _height = this.mmToPx(value)+"px";
    this.dataModel.components[0].components.forEach((cdm) => {
      cdm.styles["height"] = _height;
    });
    if (this._jqObj) {
      this._jqObj.find(".tfp-printreport-page-header").css("flex-basis", _height);
      this._jqObj.find(".tfp-printreport-page-header").children("div").css("height", _height);
    }
  }

  get pageHeaderHidden() { return this.dataModel.pageHeaderHidden ? true : false }
  set pageHeaderHidden(value) {
    this.dataModel.pageHeaderHidden = value ? true : false;
    if(value) {
      this.dataModel.components[0].styles["display"] = "none";
    } else {
      this.dataModel.components[0].styles["display"] = "flex";
    }
    if (this._jqObj) {
      if(!this.dataModel.pageHeaderHidden) {
        this._jqObj.find(".tfp-printreport-page-header").css("display", "flex");
      } else {
        this._jqObj.find(".tfp-printreport-page-header").hide();
      }
    }
  }

  get reportHeaderPanel() { return this._tfp.components[this.id+"_report_header"] }
  set reportHeaderPanel(value) {}

  get reportHeaderHeight() { return this.dataModel.reportHeaderHeight }
  set reportHeaderHeight(value) {
    this.dataModel.reportHeaderHeight = parseInt(value);
    let _height = this.mmToPx(value)+"px";
    this.dataModel.components[1].styles["height"] = _height;
    if (this._jqObj) {
      this._jqObj.find(".tfp-printreport-report-header").css("flex-basis", _height);
    }
  }

  get reportHeaderDisplay() { return this.dataModel.reportHeaderDisplay }
  set reportHeaderDisplay(value) {
    this.dataModel.reportHeaderDisplay = value;
    if (this._jqObj) {
      if(this.dataModel.reportHeaderDisplay != "none") {
        this._jqObj.find(".tfp-printreport-report-header").show();
      } else {
        this._jqObj.find(".tfp-printreport-report-header").hide();
      }
    }
  }

  get reportBodyPanel() { return this._tfp.components[this.id+"_report_body"] }
  set reportBodyPanel(value) {}

  get reportFooterPanel() { return this._tfp.components[this.id+"_report_footer"] }
  set reportFooterPanel(value) {}

  get reportFooterHeight() { return this.dataModel.reportFooterHeight }
  set reportFooterHeight(value) {
    this.dataModel.reportFooterHeight =  parseInt(value);
    let _height = this.mmToPx(value)+"px";
    this.dataModel.components[3].styles["height"] = _height;
    if (this._jqObj) {
      this._jqObj.find(".tfp-printreport-report-footer").css("flex-basis", _height);
    }
  }

  get reportFooterDisplay() { return this.dataModel.reportFooterDisplay }
  set reportFooterDisplay(value) {
    this.dataModel.reportFooterDisplay = value;
    if (this._jqObj) {
      if(this.dataModel.reportFooterDisplay != "none") {
        this._jqObj.find(".tfp-printreport-report-footer").show();
      } else {
        this._jqObj.find(".tfp-printreport-report-footer").hide();
      }
    }
  }

  get pageFooterPanel() { return this._tfp.components[this.id+"_page_footer"] }
  set pageFooterPanel(value) {}

  get pageFooterHeight() { return this.dataModel.pageFooterHeight }
  set pageFooterHeight(value) {
    this.dataModel.pageFooterHeight = parseInt(value);
    let _height = this.mmToPx(value)+"px";
    this.dataModel.components[4].components.forEach((cdm) => {
      cdm.styles["height"] = _height;
    });
    if (this._jqObj) {
      this._jqObj.find(".tfp-printreport-page-footer").css("flex-basis", _height);
      this._jqObj.find(".tfp-printreport-page-footer").children("div").css("height", _height);
    }
  }

  get pageFooterHidden() { return this.dataModel.pageFooterHidden ? true : false }
  set pageFooterHidden(value) {
    this.dataModel.pageFooterHidden = value ? true : false;
    if(value) {
      this.dataModel.components[4].styles["display"] = "none";
    } else {
      this.dataModel.components[4].styles["display"] = "flex";
    }
    if (this._jqObj) {
      if(this.dataModel.pageFooterHidden) {
        this._jqObj.find(".tfp-printreport-page-footer").hide();
      } else {
        this._jqObj.find(".tfp-printreport-page-footer").css("display", "flex");
      }
    }
  }

  inToPx(inc) {
    return parseInt(96 * inc, 10);
  }

  mmToPx(mm, dpi = 96) {
    // 1 inch = 25.4 millimeters
    const inchPerMillimeter = 1 / 25.4;
    // 1 inch = dpi pixels
    const pixelsPerInch = dpi;
    // 转换为英寸
    const inches = mm * inchPerMillimeter;
    // 转换为像素
    const pixels = inches * pixelsPerInch;
    return parseInt(pixels);
  }

  pxToMm(px, dpi = 96) {
    // 转换为英寸
    const inches = px / dpi;
    // 转换为像素
    const mm = inches * 25.4;
    return parseInt(mm);
  }

  /**
   * 设置指定属性的数据绑定信息
   * @param {*} attr 
   * @param {*} settingData 
   * @param {*} attrSetting 
   */
  setDataBindSetting(attr, settingData, attrSetting) {
    if (attr.name == "reportHeaderDataQuerySetting") {
      settingData.type = "query";
      settingData.serviceNameSuffix = "query_report_header";
      settingData.serviceComment = "表头数据查询";
      settingData.count = 1;
      attrSetting.dataMember = "data";
    } else if (attr.name == "reportBodyDataQuerySetting") {
      settingData.type = "query";
      settingData.serviceNameSuffix = "query_report_body";
      settingData.serviceComment = "表体数据查询";
      settingData.count = 1;
      attrSetting.dataMember = "data";
    } else if (attr.name == "reportFooterDataQuerySetting") {
      settingData.type = "query";
      settingData.serviceNameSuffix = "query_report_footer";
      settingData.serviceComment = "表尾数据查询";
      settingData.count = 1;
      attrSetting.dataMember = "data";
    } 
  }

  bindChildCptData(parentCdm, data) {
    if (!parentCdm.components || parentCdm.components.length==0 || !data) return;
    let bindData = data;
    //如果绑定数据是数组，则取数组的第一个元素作为绑定数据对象
    if (Array.isArray(bindData)) {
      if (bindData.length == 0) return;
      bindData = data[0];
    }
    for (var i = 0; i < parentCdm.components.length; i++) {
      let childCdm = parentCdm.components[i];
      if(["Table", "GridView", "DataCard"].includes(childCdm.type)) continue;
      this._tfp.bindCptData(childCdm, bindData);
    }
  }

  loadData() {
    let that = this;

    //加载表头数据
    if(this.dataModel.reportHeaderData) {
      this.bindChildCptData(this.dataModel.components[1], this.dataModel.reportHeaderData);
    } else if (this.dataModel.reportHeaderDataQuerySetting && this.dataModel.reportHeaderDataQuerySetting.servicePath) {
      let querySetting  = this.dataModel.reportHeaderDataQuerySetting;
      let options = {
        servicePath: querySetting.servicePath,
        autoShowError: true,
        showLoading: true
      };
      if (querySetting.requestArgs && querySetting.requestArgs.length > 0) {
        options.argSettings = querySetting.requestArgs;
      }
      if (querySetting.encryptRequestArgs) options.encryptRequestArgs = querySetting.encryptRequestArgs;
      if (querySetting.encryptResponseArgs) options.encryptResponseArgs = querySetting.encryptResponseArgs;

      this._tfp.request(options, function (req, res) {
        that.bindChildCptData(that.dataModel.components[1], res.data);
      });
    }

    //加载表体数据
    if (this.dataModel.reportBodyDataQuerySetting && this.dataModel.reportBodyDataQuerySetting.servicePath) {
      let querySetting  = this.dataModel.reportBodyDataQuerySetting;
      let options = {
        servicePath: querySetting.servicePath,
        autoShowError: true,
        showLoading: true
      };
      if (querySetting.requestArgs && querySetting.requestArgs.length > 0) {
        options.argSettings = querySetting.requestArgs;
      }
      if (querySetting.encryptRequestArgs) options.encryptRequestArgs = querySetting.encryptRequestArgs;
      if (querySetting.encryptResponseArgs) options.encryptResponseArgs = querySetting.encryptResponseArgs;

      this._tfp.request(options, function (req, res) {
        that.bindChildCptData(that.dataModel.components[2], res.data);
      });
    }

    //加载表头数据
    if(this.dataModel.reportFooterData) {
      this.bindChildCptData(this.dataModel.components[3], this.dataModel.reportFooterData);
    } else if (this.dataModel.reportFooterDataQuerySetting && this.dataModel.reportFooterDataQuerySetting.servicePath) {
      let querySetting  = this.dataModel.reportFooterDataQuerySetting;
      let options = {
        servicePath: querySetting.servicePath,
        autoShowError: true,
        showLoading: true
      };
      if (querySetting.requestArgs && querySetting.requestArgs.length > 0) {
        options.argSettings = querySetting.requestArgs;
      }
      if (querySetting.encryptRequestArgs) options.encryptRequestArgs = querySetting.encryptRequestArgs;
      if (querySetting.encryptResponseArgs) options.encryptResponseArgs = querySetting.encryptResponseArgs;

      this._tfp.request(options, function (req, res) {
        that.bindChildCptData(that.dataModel.components[3], res.data);
      });
    }
  }

  initRuntime() {
    var that = this;
    this.loadData();
  }
}