const _parent = Symbol("parent");

/**
 * TaskUI组件基类
 */
export class Component {
  constructor(__tfp, typeName, dataModel, parent) {
    if (!__tfp) {
      throw new Error("请提供tfp对象！");
      return;
    }
    if (!typeName) {
      console.log(dataModel);
      throw new Error("请提供组件类型！");
      return;
    }

    this._tfp = __tfp;
    this.level = 0;  //组件相对页面组件的层次
    this.index = 0;  //组件在父组件中的索引，主要用来排序
    //目前如果某个兄弟组件删除了，不会重置所有剩余的兄弟组件的索引

    //如果没有提供数据模型，则进行模型初始化
    if (!dataModel) {
      let metadata = this._tfp.type(typeName);
      if (!metadata) {
        throw new Error("请提供正确的组件类型！");
        return;
      }
      this.dataModel = {
        type: typeName
      };
      this.dataModel.id = this.dataModel.type.substr(0, 1).toLowerCase()
        + this.dataModel.type.substr(1) + this._tfp.getNewCptIndex(this.dataModel.type);
      //如果是用户自定义组件里的新创建的子组件，则在组件编号前面加上用户自定义组件的编号，以免跟其他页面里的组件id重复
      if(this.dataModel.type!="Page" && this._tfp.curPage.pageType=="component") 
        this.dataModel.id = this._tfp.curPage.id+"_"+this.dataModel.id;
      let attrs = this.attrTypes;
      //设置组件默认属性
      for (let i = 0; i < attrs.length; i++) {
        let attr = attrs[i];
        if (attr.default || attr.default == 0 || attr.default == false) {
          if (attr.type.toLowerCase() == "string") {
            this.dataModel[attr.name] = attr.default.replace("{id}", this.id);
          } else {
            this.dataModel[attr.name] = attr.default;
          }
        }
      }
      //设置可视组件数据模型的默认样式
      if (metadata.defaultStyles) {
        this.dataModel.styles = {};
        for (let style in metadata.defaultStyles) {
          this.dataModel.styles[style] = metadata.defaultStyles[style];
        }
      }
    } else {  //否则用传入的数据模型
      this.dataModel = dataModel;
      if (!this.dataModel.type) this.dataModel.type = typeName;
    }

    //设置组件数据模型的默认ID
    if (!this.dataModel.id) {
      this.dataModel.id = this.dataModel.type.substr(0, 1).toLowerCase()
        + this.dataModel.type.substr(1) + this._tfp.getNewCptIndex(this.dataModel.type);
    }
    this._tfp.components[this.dataModel.id] = this;
    if (this._tfp.isRuntime) window[this.dataModel.id] = this;

    //设置组件的父组件
    if (parent) {
      this[_parent] = parent;
      this.level = parent.level + 1;
      if (!this._tfp.isLoadingPage) {
        if (!parent.dataModel.components) parent.dataModel.components = [];
        let cptExists = false;
        for (var i = 0; i < parent.dataModel.components.length; i++) {
          let cdmTmp = parent.dataModel.components[i];
          if (cdmTmp.id == this.dataModel.id) {
            cptExists = true;
            continue;
          }
          if (cdmTmp.index >= this.index) this.index = cdmTmp.index + 1;
        }
        if (!cptExists) parent.dataModel.components.push(this.dataModel);
      }
    } else {
      //TODO 如果创建组件时没有提供父组件，则表示是临时组件，后续可以设置父组件
    }
  }

  //组件ID，每个组件的ID是唯一的，不能重复
  get id() { return this.dataModel.id }
  set id(value) {
    if (!value) return;
    let oldId = this.dataModel.id;
    if (oldId == value) return;
    for (let cptId in this._tfp.components) {
      if (cptId == value) {
        throw new Error("ID为的" + value + "组件已存在！");
        return;
      }
    }
    this.dataModel.id = value;
    delete this._tfp.components[oldId];
    this._tfp.components[this.dataModel.id] = this;
    if (this._tfp.isRuntime) {
      delete window[oldId];
      window[this.dataModel.id] = this;
    }
    if ($("#" + oldId).length > 0) {
      $("#" + oldId).attr("id", this.dataModel.id);
    }
  }

  //组件类型，组件一旦创建，不允许再修改类型
  get type() { return this.dataModel.type }
  set type(value) { }

  //组件类型元数据，组件一旦创建，不允许再修改
  get metadata() { return this._tfp.type(this.dataModel.type) }
  set metadata(value) { }

  //父组件
  get parent() { return this[_parent] }
  set parent(value) {
    //如果没有设置父组件
    if (!this[_parent]) {
      this[_parent] = value;
      if (!this[_parent].dataModel.components) this[_parent].dataModel.components = [];
      this[_parent].dataModel.components.push(this.dataModel);
    } else {
      if (this[_parent].id != value.id) {  //如果已经设置了父组件，但新设置的父组件和原来的父组件不一样
        this[_parent].dataModel.components.remove(this.dataModel);
        this[_parent] = value;
        if (!this[_parent].dataModel.components) this[_parent].dataModel.components = [];
        this[_parent].dataModel.components.push(this.dataModel);
      } else {
        //如果父组件没有变化，则不需要执行后续操作
        return;
      }
    }
    //设置当前组件的层级和索引
    this.level = this[_parent].level + 1;
    for (var i = 0; i < this[_parent].dataModel.components.length; i++) {
      let cdmTmp = this[_parent].dataModel.components[i];
      if (cdmTmp.index >= this.index) this.index = cdmTmp.index + 1;
    }
  }

  get attrTypes() {
    let attrs = [];
    let metadata = this._tfp.type(this.type);
    if (metadata.attrs) {
      for (let i = 0; i < metadata.attrs.length; i++) {
        let attr = metadata.attrs[i];
        if (attr.type == "group" || attr.items) {
          for (let j = 0; j < attr.items.length; j++) {
            attrs.push(attr.items[j]);
          }
        } else {
          attrs.push(attr);
        }
      }
    }
    return attrs;
  }
  set attrTypes(value) { }

  setParentOnly(value) {
    this[_parent] = value;
    this.level = this[_parent].level + 1;
  }

  attr(attrName, attrValue) {
    if (arguments.length == 0) return;
    //获取属性值
    if (arguments.length == 1) {
      return this[attrName];
    }
    this[attrName] = attrValue;
  }

  /**
   * 获得属性定义信息
   * @param  {[type]} attrName [description]
   * @return {[type]}          [description]
   */
  getAttrTypeInfo(attrName) {
    if (!this.metadata) {
      throw new Error("没有找到组件类型定义信息，请先引用类型信息！");
      return;
    }
    if (!this.metadata.attrs) {
      //throw new Error("没有找到组件的属性定义信息！");
      return null;
    }
    for (var i = 0; i < this.metadata.attrs.length; i++) {
      let attrInfo = this.metadata.attrs[i];
      if (attrInfo.type == "group" || attrInfo.items) {
        for (var j = 0; j < attrInfo.items.length; j++) {
          if (attrInfo.items[j].name == attrName) {
            return attrInfo.items[j];
          }
        }
      } else {
        if (this.metadata.attrs[i].name == attrName) {
          return this.metadata.attrs[i];
        }
      }
    }
    return null;
  }

  /**
   * 检查属性选项是否是组件类型定义中设置的选项
   * @param  {[type]} attrName [description]
   * @param  {[type]} attrVal  [description]
   * @return {[type]}          [description]
   */
  checkAttrOption(attrName, attrVal) {
    let attr = this.getAttrTypeInfo(attrName);
    if (!attr) return false;
    let options = attr.options;
    if (!options) return false;
    for (var i = 0; i < options.length; i++) {
      if (options[i].value == attrVal) return true;
    }
    return false;
  }

  /**
   * 执行事件处理函数
   * @param  {[type]} eventName [description]
   * @return {[type]}           [description]
   */
  exeEventHandler(eventName) {
    //设计时不执行任何事件处理函数
    if (this._tfp.isDesigning) return;
    let ret;
    if (this[eventName] && typeof this[eventName] == "function") {
      ret = this[eventName](arguments);
    } else if (this.dataModel[eventName]) {
      let funcStr = this.dataModel[eventName];
      if (funcStr.indexOf("(") > 0) {
        let funcName = funcStr.substr(0, funcStr.indexOf("("));
        if (typeof (window[funcName]) == "function") {
          let args = [];
          if (arguments.length > 1) {
            for (var i = 1; i < arguments.length; i++) {
              args.push(arguments[i]);
            }
          }
          let func = window[funcName];
          ret = func.apply(window, args);
        } else {
          ret = eval(funcStr);
        }
      } else {
        ret = eval(funcStr);
      }
    }
    if (ret) return ret;
  }
}

export class InvisibleComponent extends Component {

  constructor(__tfp, typeName, dataModel, parent) {
    super(__tfp, typeName, dataModel, parent);
  }

  get isInvisible() { return true }
  set isInvisible(value) { }

  render() {
    //如果是不可视组件，只有在设计时需要渲染
    if (this._tfp.isDesigning) {
      if (window.parent.uiDesigner)
        window.parent.uiDesigner.addInvisibleComponent(this);
      this._tfp.initCptDesignSetting(this);
    }
  }
}

export class VisibleComponent extends Component {

  constructor(__tfp, typeName, dataModel, parent) {
    super(__tfp, typeName, dataModel, parent);

    this._jqObj = null;
    this.el = null;
    this.isRendered = false;
  }

  get isInvisible() { return false }
  set isInvisible(value) { }

  get isContainer() { return false }
  set isContainer(value) { }

  get isShow() { return this._jqObj.is(":visible") }
  set isShow(value) { }

  get styles() { return this.dataModel.styles; }
  set styles(value) { }

  get style() { return this.dataModel.style; }
  set style(value) {
    if (!value) value = "";
    this.dataModel.style = value;
    if (this._jqObj) {
      let style = value.trim();
      if (style != "" && !style.endsWith(";")) style += ";";
      if (this.dataModel.styles) {
        for (let styleTmp in this.dataModel.styles) {
          style += styleTmp + ": " + this.dataModel.styles[styleTmp] + ";";
        }
      }
      //要保留设计时的外边框
      let outline = this._jqObj.css("outline");
      this._jqObj.attr("style", style);
      this._jqObj.css("outline", outline);
    }
  }

  // 更改样式对象
  get styleObj() {
    //console.log(this.dataModel)
    return this.dataModel.styleObj
  }
  set styleObj(value) {
    this.dataModel.styleObj = value;
  }

  // 显示隐藏对象
  get hideObj() {
    return this.dataModel.hideObj
  }
  set hideObj(value) {
    this.dataModel.hideObj = value;
  }

  get class() { return this.dataModel.class; }
  set class(value) {
    if (!value) value = "";
    this.dataModel.class = value;
    if (this._jqObj) {
      this._jqObj.attr("class", value);
    }
  }

  get indent() {
    var _indent = "";
    for (var i = 0; i < this.level; i++) {
      _indent += "\t";
    }
    return _indent;
  }
  set indent(value) { }

  get controlId() { return this.dataModel.id }
  set controlId(value) {
    this.id = value;
    if (this.type.toLowerCase() == "dataset") {
      this.dataBindingMember = "data_" + this.id
    }
    else {
      this.dataBindingFormat = "{" + this.id + "}";
    }
  }

  get controlName() { return this.dataModel.titleName }
  set controlName(value) {
    this.dataModel.titleName = value;

    this._tfp.get(this.parent.parent.components[0].components[0].id)["value"] = value + "：";
  }

  get controlWidth() { return this.parent.parent.parent["colWidth"]; }
  set controlWidth(value) {
    this._tfp.get(this.parent.parent.parent.id)["colWidth"] = value;
  }

  get controlDirection() { return this.parent.parent["direction"]; }
  set controlDirection(value) {
    this._tfp.get(this.parent.parent.id)["direction"] = value;
  }

  css(styleName, styleValue) {
    if (arguments.length == 0) return;
    //获取样式值
    if (arguments.length == 1) {
      if (!this.dataModel.styles) return null;
      return this.dataModel.styles[styleName];
    }
    if (this._jqObj) this._jqObj.css(styleName, styleValue);
    if (!this.dataModel.styles) this.dataModel.styles = {};
    if (styleValue == "" || styleValue == null) {
      delete this.dataModel.styles[styleName];
    } else {
      this.dataModel.styles[styleName] = styleValue;
      //如果定义了样式修改监听事件处理器，则执行
      if(this.styleOnChange) this.styleOnChange(styleName, styleValue);
    }
  }

  val(value) {
    if (arguments.length == 0) return this.value;
    this.value = value;
  }

  show() {
    if (this._jqObj) this._jqObj.show();
  }

  hide() {
    if (this._jqObj) this._jqObj.hide();
  }

  toggle(speed, easing, cb) {
    if (this._jqObj) this._jqObj.toggle(speed, easing, cb);
  }

  animate(styles, speed, easing, cb) {
    if (this._jqObj) this._jqObj.animate(styles, speed, easing, cb);
  }

  clearQueue(queueName) {
    if (this._jqObj) this._jqObj.clearQueue(queueName);
  }

  delay(speed, queueName) {
    if (this._jqObj) this._jqObj.delay(speed, queueName);
  }

  dequeue(queueName) {
    if (this._jqObj) this._jqObj.dequeue(queueName);
  }

  fadeIn(speed, easing, cb) {
    if (this._jqObj) this._jqObj.fadeIn(speed, easing, cb);
  }

  fadeOut(speed, easing, cb) {
    if (this._jqObj) this._jqObj.fadeOut(speed, easing, cb);
  }

  fadeTo(speed, opacity, easing, cb) {
    if (this._jqObj) this._jqObj.fadeTo(speed, opacity, easing, cb);
  }

  fadeToggle(speed, easing, cb) {
    if (this._jqObj) this._jqObj.fadeToggle(speed, easing, cb);
  }

  finish(queueName) {
    if (this._jqObj) this._jqObj.finish(queueName);
  }

  queue(queueName) {
    if (this._jqObj) this._jqObj.queue(queueName);
  }

  slideDown(speed, easing, cb) {
    if (this._jqObj) this._jqObj.slideDown(speed, easing, cb);
  }

  slideToggle(speed, easing, cb) {
    if (this._jqObj) this._jqObj.slideToggle(speed, easing, cb);
  }

  slideUp(speed, easing, cb) {
    if (this._jqObj) this._jqObj.slideUp(speed, easing, cb);
  }

  stop(stopAll,goToEnd) {
    if (this._jqObj) this._jqObj.stop(stopAll,goToEnd);
  }

  bind(eventName, cb) {
    if (this._jqObj) this._jqObj.bind(eventName, cb);
  }

  unbind(eventName, cb) {
    if (this._jqObj) this._jqObj.unbind(eventName, cb);
  }

  click(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.click(cb);
    } else {
      this._jqObj.click();
    }
  }

  dblclick(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.dblclick(cb);
    } else {
      this._jqObj.dblclick();
    }
  }

  blur(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.blur(cb);
    } else {
      this._jqObj.blur();
    }
  }

  focus(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.focus(cb);
    } else {
      this._jqObj.focus();
    }
  }

  focusin(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.focusin(cb);
    } else {
      this._jqObj.focusin();
    }
  }

  focusout(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.focusout(cb);
    } else {
      this._jqObj.focusout();
    }
  }

  hover(inFunction,outFunction) {
    if (this._jqObj) this._jqObj.hover(inFunction,outFunction);
  }

  keydown(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.keydown(cb);
    } else {
      this._jqObj.keydown();
    }
  }

  keypress(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.keypress(cb);
    } else {
      this._jqObj.keypress();
    }
  }

  keyup(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.keyup(cb);
    } else {
      this._jqObj.keyup();
    }
  }

  mousedown(cb) {
    if (this._jqObj) this._jqObj.mousedown(cb);
  }

  mouseenter(cb) {
    if (this._jqObj) this._jqObj.mouseenter(cb);
  }

  mouseleave(cb) {
    if (this._jqObj) this._jqObj.mouseleave(cb);
  }

  mousemove(cb) {
    if (this._jqObj) this._jqObj.mousemove(cb);
  }

  mouseout(cb) {
    if (this._jqObj) this._jqObj.mouseout(cb);
  }

  mouseover(cb) {
    if (this._jqObj) this._jqObj.mouseover(cb);
  }

  mouseup(cb) {
    if (this._jqObj) this._jqObj.mouseup(cb);
  }

  resize(cb) {
    if (this._jqObj) this._jqObj.resize(cb);
  }

  scroll(cb) {
    if(!this._jqObj) return;
    if (cb) {
      this._jqObj.scroll(cb);
    } else {
      this._jqObj.scroll();
    }
  }

  select() {
    if (this._jqObj) this._jqObj.select();
  }

  trigger(eventName) {
    if (this._jqObj) this._jqObj.trigger(eventName);
  }

  triggerHandler(eventName) {
    if (this._jqObj) this._jqObj.triggerHandler(eventName);
  }

  width() {
    if (this._jqObj) this._jqObj.width();
  }

  height() {
    if (this._jqObj) this._jqObj.height();
  }

  html(html) {
    if (this._jqObj) {
      if(arguments.length == 0) {
        return this._jqObj.html();
      } else {
        this._jqObj.html(html);
      }
    }
  }

  text(txt) {
    if (this._jqObj) {
      if(arguments.length == 0) {
        return this._jqObj.text();
      } else {
        this._jqObj.text(txt);
      }
    }
  }

  innerHeight() {
    if (this._jqObj) this._jqObj.innerHeight();
  }

  innerWidth() {
    if (this._jqObj) this._jqObj.innerWidth();
  }

  offset() {
    if (this._jqObj) this._jqObj.offset();
  }

  outerHeight() {
    if (this._jqObj) this._jqObj.outerHeight();
  }

  outerWidth() {
    if (this._jqObj) this._jqObj.outerWidth();
  }

  position() {
    if (this._jqObj) this._jqObj.position();
  }

  scrollLeft() {
    if (this._jqObj) this._jqObj.scrollLeft();
  }

  scrollTop() {
    if (this._jqObj) this._jqObj.scrollTop();
  }

  getHtmlIndent() {
    if (this.indent) return this.indent;
    let indent = "";
    for (let i = 0; i < this.level; i++) {
      indent += "\t";
    }
    return indent;
  }

  render(target, direction) {
    if ((!this.parent || (!this.parent.containerEl && !this.parent.getContainerEl)) && this.type != "Page") return;
    let Render = this._tfp.renders[this.type];
    let render = new Render(this._tfp, this.dataModel, this.level);
    if (this.type == "Page") {
      if(this.dataModel.pageType=="print-report") {
        this._jqObj = $(".print-page-bg-panel");
      } else if (this.dataModel.pageElId && this._tfp.isRuntime) {
        this._jqObj = $("#" + this.dataModel.pageElId);
      } else {
        this._jqObj = $("body");
      }
      this._jqObj.append(render.getHtml());
      this._tfp.curPage = this;
    } else {
      let containerEl = null;
      //getContainerEl方法是为了支持父组件不止一个容器元素的情况，例如布局组件，需要通过子组件id提供不同的容器元素
      if (this.parent.getContainerEl) {
        containerEl = this.parent.getContainerEl(this.id, this.dataModel);
      } else {
        containerEl = this.parent.containerEl;
      }
      if (!containerEl) return;

      if (!target) {
        if (direction == "before" || direction == "prepend" || direction == "left" || direction == "top") {
          $(containerEl).prepend(render.getHtml());
          this.parent.dataModel.components.pop();
          this.parent.dataModel.components.unshift(this.dataModel);
        }
        else if (direction == "after" || direction == "append" || direction == "right" || direction == "bottom") {
          $(containerEl).append(render.getHtml());
        }
        else {
          $(containerEl).append(render.getHtml());
        }
      }
      else {
        this.parent.dataModel.components.pop();
        let targetCptIndex = 0;
        for(let i=0;i<this.parent.dataModel.components.length;i++) {
          let cdmChild = this.parent.dataModel.components[i];
          if(cdmChild.id==target.id) {
            targetCptIndex = i;
            break;
          }
        }
        if (direction == "before" || direction == "prepend" || direction == "left" || direction == "top") {
          $("#" + target.id).before(render.getHtml());
          // this.parent.dataModel.components.splice(targetCptIndex, 0, this.dataModel);
        }
        else if (direction == "after" || direction == "append" || direction == "right" || direction == "bottom") {
          $("#" + target.id).after(render.getHtml());
          this.parent.dataModel.components.splice(targetCptIndex+1, 0, this.dataModel);
        }
        else {
          $(containerEl).append(render.getHtml());
          this.parent.dataModel.components.push(this.dataModel);
        }
      }
      this._jqObj = $("#" + this.id);
    }
    if (this._jqObj.length > 0) this.el = this._jqObj.get(0);
    if (!this.isRendered && this._tfp.isDesigning) {
      this._tfp.initCptDesignSetting(this);
    }
    if (typeof window != "undefined" && !this._tfp.isDesigning) {
      window[this.id] = this;
    }
    if (this.dataModel.components) {
      for (var i = 0; i < this.dataModel.components.length; i++) {
        let cdmChild = this.dataModel.components[i];
        let cptChild = this._tfp.render(cdmChild, this);
      }
    }

    if (this._tfp.isDesigning) {
      if (this["initDesigning"]) this.initDesigning();
    } else if (this._tfp.isRuntime) {
      //如果页面已经渲染完，然后动态创建新的组件，则需要初始化组件运行时
      if (this._tfp.curPage && this._tfp.curPage.isRendered && this["initRuntime"]) this.initRuntime();
    }
    this.isRendered = true;
  }

  renderByTarget(target, targetParent, direction) {
    this.setParentOnly(targetParent);

    var index = 0;
    if (!target) {
      if (direction == "before" || direction == "prepend" || direction == "left" || direction == "top") {
        index = 0;
      }
      else if (direction == "after" || direction == "append" || direction == "rignt" || direction == "bottom") {
        index = targetParent.dataModel.components.length - 1;
        if (index < 0) {
          index = 0;
          direction = "prepend";
        }
        else {
          target = targetParent.dataModel.components[index];
          index = index + 1;
        }
      }
    }
    else {
      for (var j = 0; j < targetParent.dataModel.components.length; j++) {
        if (targetParent.dataModel.components[j].id == target.id) {
          if (direction == "before" || direction == "prepend" || direction == "left" || direction == "top") {
            index = j;
          }
          else if (direction == "after" || direction == "append" || direction == "rignt" || direction == "bottom") {
            index = j + 1;
          }
          else {
            index = j + 1;
          }
          break;
        }
      }
    }
    targetParent.dataModel.components.splice(index, 0, this.dataModel);

    this.render(target, direction);
  }

  remove() {
    this._tfp.remove(this, true);
  }

  clear() {
    if (this.dataModel.components) {
      for (var i = 0; i < this.dataModel.components.length; i++) {
        let childCdm = this.dataModel.components[i];
        let child = this._tfp.get(childCdm.id);
        child.clear();
        if (child._jqObj) child._jqObj.remove();
        delete this._tfp.components[child.id];
      }
      //this.dataModel.components = [];
    }
  }

  print() {
    this._tfp.print(this.el);
  }

  setAttr(attr, value) {
    if (attr.cellType == "col") {
      this.parent.parent.parent[attr.name] = value;
    }
    else if (attr.cellType == "label") {
      this.parent.parent.components[0].components[0][attr.name] = value;
    }
    else {
      this[attr.name] = value;
    }
  }

  getAttr(attr) {
    if (attr.cellType == "col") {
      return this.parent.parent.parent[attr.name];
    }
    else if (attr.cellType == "label") {
      return this.parent.parent.components[0].components[0][attr.name];
    }
    else {
      return this[attr.name];
    }
  }

  /**
   * 为可视组件定义的统一的数据加载方法
   * @returns 
   */
  loadData(args) {
    //如果没有设置数据查询有关的参数，则不执行数据加载操作
    if (!this.dataModel.loadDataListService //某些老版本的组件使用的该属性，需要兼容
      && !this.dataModel.loadDataService 
      && !this.dataModel.dataQuerySetting) {
        //this._tfp.showMsg("请为[" + this.id + "]设置数据查询有关参数！");
      return;
    }
    //获得数据绑定成员对象名
    let dataBindingMember = this.dataModel.dataBindingMember;  //某些老版本的组件使用的该属性，需要兼容
    if(this.dataModel.dataListBindingMember) {
      dataBindingMember = this.dataModel.dataListBindingMember;
    } else if(this.dataModel.dataQuerySetting && this.dataModel.dataQuerySetting.dataMember) {
      dataBindingMember = this.dataModel.dataQuerySetting.dataMember;
    }
    if (!dataBindingMember) {
      this._tfp.showMsg("请为[" + this.id + "]设置数据绑定成员！");
      return;
    }

    //如果设置了检查数据加载必须参数的方法，则执行该方法进行检查
    if(this.checkLoadDataArg) {
      let ret = this.checkLoadDataArg();
      if(!ret) return;
    }

    this.exeEventHandler("onBeforeLoadData");

    let that = this;

    //如果设置了加载数据的后台服务组件，则通过该组件加载数据
    let loadDataService = this.dataModel.loadDataService;  //兼容老版本组件
    if(this.dataModel.loadDataListService) {
      loadDataService = this.dataModel.loadDataListService;
    } else if(this.dataModel.dataQuerySetting && this.dataModel.dataQuerySetting.serviceCpt) {
      loadDataService = this.dataModel.dataQuerySetting.serviceCpt;
    }
    if(loadDataService) {
      let serviceCpt = this._tfp.get(loadDataService);
      if (!serviceCpt) {
        this._tfp.showMsg("ID为[" + loadDataService + "]的后台服务组件不存在！");
        return;
      }

      //与后台服务组件建立绑定
      if (!serviceCpt.bindCpts) serviceCpt.bindCpts = [];
      if (!serviceCpt.bindCpts.contains(this.id)) serviceCpt.bindCpts.push(this.id);
      serviceCpt.status = 0;
      serviceCpt.request(args, function (req, res) {
        that.isLoadedData = true;
        that.exeEventHandler("onAfterLoadData", req, res);
        //如果定义了数据加载后需要执行的方法，则执行该方法
        if(that.onLoadData) {
          that.onLoadData(req, res);
        } else {  //否则直接执行数据绑定的方法
          var data = res[dataBindingMember];
          that.bindData(data);
        }
      });
      return;
    }

    //如果设置了加载数据的后台服务路径，则动态创建服务组件请求该地址进行数据加载
    if(this.dataModel.dataQuerySetting && this.dataModel.dataQuerySetting.servicePath) {
      let options = {
        cptId: this.id,
        servicePath: this.dataModel.dataQuerySetting.servicePath,
        autoReload: false,
        autoShowError: true,
        showLoading: true,
        encryptRequestArgs: this.dataModel.dataQuerySetting.encryptRequestArgs,
        encryptResponseArgs: this.dataModel.dataQuerySetting.encryptResponseArgs,
        args: args
      };
      if(this.dataModel.dataQuerySetting.disableAutoLoad) {
        options.disableAutoLoad = true;
        //第一次加载后就要把这个开关改过来，不然后面再手动加载也无法加载
        this.dataModel.dataQuerySetting.disableAutoLoad = false;
      }
      //如果设置了查询的请求参数，则绑定到服务对象上
      if(this.dataModel.dataQuerySetting.requestArgs && this.dataModel.dataQuerySetting.requestArgs.length>0) {
        options.argSettings = this.dataModel.dataQuerySetting.requestArgs;
      }
      //发起动态请求
      this._tfp.request(options, function(req, res, service) {
        that.isLoadedData = true;
        that.exeEventHandler("onAfterLoadData", req, res);
        //如果定义了数据加载后需要执行的方法，则执行该方法
        if(that.onLoadData) {
          that.onLoadData(req, res);
        } else {  //否则直接执行数据绑定的方法
          var data = res[dataBindingMember];
          that.bindData(data);
        }
      });
      return;
    }
  }
}

//容器组件
export class ContainerComponent extends VisibleComponent {

  constructor(__tfp, typeName, dataModel, parent) {
    super(__tfp, typeName, dataModel, parent);

    if (!this.dataModel.components) this.dataModel.components = [];
  }

  get isContainer() { return true }
  set isContainer(value) { }

  get containerEl() { return this.el }
  set containerEl(value) {
    this.el = value;
  }

  get components() { return this.dataModel.components }
  set components(value) { }

  /**
   * 添加子组件
   * @param {[type]} cptChild [description]
   */
  addChild(cptChild) {
    cptChild.parent = this;
  }

  /**
   * 移除子组件
   * @param  {[type]} cptId [description]
   * @return {[type]}       [description]
   */
  removeChild(cptId) {
    for (var i = 0; i < this.dataModel.components.length; i++) {
      let childCdm = this.dataModel.components[i];
      if (childCdm.id == cptId) {
        let child = this._tfp.components[childCdm.id];
        child.clear();
        if (child._jqObj) child._jqObj.remove();
        if (this._tfp.isRuntime) {
          delete this._tfp.components[child.id];
          delete window[child.id];
        }
        child = null;
        childCdm = null;
        return;
      }
    }
  }

  /**
   * 清空子组件
   * @return {[type]} [description]
   */
  clearChildren() {
    for (var i = 0; i < this.dataModel.components.length; i++) {
      let childCdm = this.dataModel.components[i];
      let child = this._tfp.components[childCdm.id];
      child.clear();
      if (child._jqObj) child._jqObj.remove();
      delete this._tfp.components[child.id];
      if (this._tfp.isRuntime) delete window[child.id];
      child = null;
      childCdm = null;
    }
    this.dataModel.components = [];
  }
}

//表单输入项组件
export class FormInput extends VisibleComponent {

  constructor(__tfp, typeName, dataModel, parent) {
    super(__tfp, typeName, dataModel, parent);
  }

  get isFormInput() { return true }

  //组件备注
  get comment() { return this.dataModel.comment }
  set comment(value) { this.dataModel.comment = value }

  //数据绑定格式
  get dataBindingFormat() { return this.dataModel.dataBindingFormat }
  set dataBindingFormat(value) { this.dataModel.dataBindingFormat = value }

  //是否必填
  get required() { return this.dataModel.required ? true : false }
  set required(value) { this.dataModel.required = value ? true : false }

  //是否只读
  get readonly() { return this.dataModel.readonly ? true : false }
  set readonly(value) {
    this.dataModel.readonly = value ? true : false;
    if (!this.dataModel.readonly) delete this.dataModel.readonly;
    if (this._jqObj && this._jqObj.length > 0 && !this._tfp.isDesigning) {
      let el = this._jqObj.get(0);
      if (el.tagName == "INPUT" || el.tagName == "TEXTAREA") {
        el.readOnly = this.dataModel.readonly;
      } else if (el.tagName == "SELECT") {
        el.disabled = this.dataModel.disabled;
      } else if (el.tagName == "DIV") {
        var that = this;
        this._jqObj.find("input").each(function () {
          $(this).get(0).readOnly = that.dataModel.readonly;
        });
      }
    }
  }

  //是否加密发送
  get encrypted() { return this.dataModel.encrypted ? true : false }
  set encrypted(value) { this.dataModel.encrypted = value ? true : false }

  //是否禁用
  get disabled() { return this.dataModel.disabled ? true : false }
  set disabled(value) {
    this.dataModel.disabled = value ? true : false;
    if (this._jqObj && this._jqObj.length > 0 && !this._tfp.isDesigning) {
      let el = this._jqObj.get(0);
      if (el.tagName == "INPUT" || el.tagName == "SELECT" || el.tagName == "TEXTAREA") {
        el.disabled = this.dataModel.disabled;
      } else if (el.tagName == "DIV") {
        var that = this;
        this._jqObj.find("input").each(function () {
          $(this).get(0).disabled = that.dataModel.disabled;
        });
      }
    }
    if (!this.dataModel.disabled) delete this.dataModel.disabled;
  }

  //自动计算表达式
  get formula() { return this.dataModel.formula }
  set formula(value) {
    if (isNull(value)) {
      delete this.dataModel["formula"];
      return;
    }
    this.dataModel.formula = value;
  }

  /**
   * 当值发生变化时
   * @return {[type]} [description]
   */
  valueOnChange() {
    this._tfp.iptValueOnChange(this);
  }

  /**
   * 执行计算表达式
   * @return {[type]} [description]
   */
  exeFormula() {
    if (!this.dataModel.formula) return;
    let val = this.dataModel.formula;
    let ipts = this.dataModel.formula.match(/\$\{[\w]+\}/g);
    for (let i = 0; i < ipts.length; i++) {
      let iptId = ipts[i].substr(2, ipts[i].length - 3);
      let ipt = this._tfp.components[iptId];
      if (!ipt) continue;
      let iptVal = ipt.value;
      if (ipt.type == "Text" && isNull(iptVal)) {
        //如果是单行输入框，且数据类型不是文本，则值为空时，将值设置为0，以便计算
        //不判断类型了，用户经常不设置类型
        //if(ipt.dataType=="int" || ipt.dataType=="float" || ipt.dataType=="money") {
        iptVal = "0";
        //}
      } else if (ipt.type == "DataSet") {
        iptVal = ipt.id;
      }
      val = val.replace(new RegExp("\\$\\{" + iptId + "\\}"), iptVal);
    }
    try {
      this.value = eval(val);
    } catch (err) {
      console.error("执行计算表达式出错：" + err.message);
    }
  }

  /**
   * 设置输入项的选项
   * @param {[type]} value [description]
   */
  setOptions(value) {
    if (!value) return;
    let options = [];
    if (typeof (value) == "string") {
      let arr = value.split(",");
      for (var i = 0; i < arr.length; i++) {
        let str = arr[i];
        if (str.trim() != "") {
          options.push({
            value: str,
            text: str
          });
        }
      }
    } else if (Array.isArray(value)) {
      for (var i = 0; i < value.length; i++) {
        let val = value[i];
        if (Object.prototype.toString.call(val) === '[object Object]') {
          options.push(val);
        } else {
          options.push({
            value: val,
            text: val
          });
        }
      }
    }
    this.dataModel.options = options;
  }

  /**
   * 触发input类型组件的值改变事件
   * @param {*} cb 
   */
  change(cb) {
    if (this._jqObj) this._jqObj.change(cb);
  }
}