import TFPStatements from './TFPStatements.js'

export default class TFPGUIFuncCompiler {

    constructor() {
        this.smCategoryDic = {};
        this.smTypesDic = {};
    }

    init() {
        for(let i=0;i<TFPStatements.length;i++) {
            let category = TFPStatements[i];
            this.smCategoryDic[category.name] = category;
            for(let j=0;j<category.statements.length;j++) {
                let sm = category.statements[j];
                this.smTypesDic[category.name+"_"+sm.name] = sm;
            }
        }
    }

    getSMTypeInfo(categoryName, smName) {
        return this.smTypesDic[categoryName+"_"+smName];
    }

    getVars(funcObj, jsGlobalVars) {
        //存储以定义的变量信息，以便后续语句判断变量是否已声明
        let vars = {};
        //全局变量
        if(jsGlobalVars && jsGlobalVars.length>0) {
            for(let i=0; i<jsGlobalVars.length; i++) {
                vars[jsGlobalVars[i].name] = "global";
            }
        }
        //函数参数
        if(funcObj.args && funcObj.args.length>0) {
            for(let i=0; i<funcObj.args.length; i++) {
                vars[funcObj.args[i].name] = "arg";
            }
        }
        return vars;
    }

    compile(funcObj, jsGlobalVars) {
        if (!funcObj || !funcObj.statements || funcObj.statements.length == 0) return "";
        let vars = this.getVars(funcObj, jsGlobalVars);
        let js = "";
        for (let i = 0; i < funcObj.statements.length; i++) {
            js += this.getStatementJs(vars, funcObj.statements[i], "\t");
        }
        return js;
    }

    formatStrVar(str) {
        var ret = str;
        var patt = new RegExp("\\{([\\w|\.|%|\\u4E00-\\u9FA5]*)\\}", "g");
        var result = patt.exec(ret);
        while (result != null) {
            let varName = result[0];
            varName = varName.substring(1, varName.length - 1);
            ret = ret.replace("{" + varName + "}", "'+"+varName+"+'");
            patt = new RegExp("\\{([\\w|\.|%|\\u4E00-\\u9FA5]*)\\}", "g");
            result = patt.exec(ret);
        }
        return ret;
    }

    getSMInputInfo(smtype, inputName) {
        if(!smtype.inputs || smtype.inputs.length==0) return null;
        for(let i=0;i<smtype.inputs.length;i++) {
            let ipt = smtype.inputs[i];
            if(ipt.name==inputName) return ipt;
        }
        return null;
    }

    formatExpress(smType, sm, prefix) {
        if(!smType.format) return "";
        var val = smType.format;
        var patt = new RegExp("\\{([\\w|\.|%|\\u4E00-\\u9FA5]*)\\}", "g");
        var result = patt.exec(val);
        while (result != null) {
            let argName = result[0];
            argName = argName.substring(1, argName.length - 1);
            let argVal = "";
            if(argName=="_prefix") {
                argVal = prefix;
            } else if(sm[argName] || sm[argName]==0) {
                argVal = sm[argName];
                if(typeof(argVal)=="string" && argVal.indexOf("{")>=0 && argVal.indexOf("}")>0) {
                    argVal = this.formatStrVar(argVal);
                }
            } else {
                let ipt = this.getSMInputInfo(smType, argName);
                //如果设置了用户输入的值为空时
                if(ipt && ipt.nullVal) {
                    argVal = ipt.nullVal;
                }
            }
            val = val.replace("{" + argName + "}", argVal);
            patt = new RegExp("\\{([\\w|\.|%|\\u4E00-\\u9FA5]*)\\}", "g");
            result = patt.exec(val);
        }
        return val;
    }

    getStatementJs(vars, sm, prefix, isArg) {
        if(sm.disabled) return "";
        let smType = this.getSMTypeInfo(sm.category, sm.type);
        if(!smType) return "";

        //检查必填项，如果必填项为空，则不生成该语句
        if(smType.inputs && smType.inputs.length>0) {
            for(let i=0;i<smType.inputs.length;i++) {
                let ipt = smType.inputs[i];
                let iptVal = sm[ipt.name];
                if(ipt.required && !iptVal && iptVal!=0) {
                    return "";
                }
            }
        }
        let js = "";
        if (sm.category == "logic" || sm.category=="common") {
            return this.getCommonStatementJs(vars, smType, sm, prefix, isArg);
        } else {
            if(sm.category == "component") {
                js += this.getCptStatementJs(vars, smType, sm, prefix, isArg);
            } else {
                if(!isArg || sm.statements) js += prefix;
                js += this.addReturnVar(vars, sm);
                js += this.formatExpress(smType, sm, prefix);
                if(!isArg || sm.statements) js += ";\r\n";
            }
        }
        let mergeArgs = {};
        let haveMergeCbFunc = false;
        //判断有没有需要合并的参数
        if(smType.inputs && smType.inputs.length>0) {
            for(let i=0;i<smType.inputs.length;i++) {
                let ipt = smType.inputs[i];
                if(ipt.mergeTo) {
                    if(!mergeArgs[ipt.mergeTo]) mergeArgs[ipt.mergeTo] = {};
                    let mergeArg = mergeArgs[ipt.mergeTo];
                    if(ipt.type=="cbFunc") {
                        mergeArg[ipt.name] = this.formatExpress(ipt, sm, prefix);
                        haveMergeCbFunc = true;
                    } else {
                        mergeArg[ipt.name] = sm[ipt.name];
                    }
                }
            }
        }
        for(let p in mergeArgs) {
            let mergeArg = mergeArgs[p];
            let jsMerge = "";
            for(let pp in mergeArg) {
                let v = mergeArg[pp];
                if(v==null || v==undefined) continue;
                if(jsMerge!="") jsMerge += ",\r\n";
                jsMerge += prefix+"\t\""+pp+"\": "+(v+"").replaceAll("'", '"');
            }
            let ii = js.indexOf("[["+p+"]]");
            if(ii>0) {
                js = js.substring(0, ii)+"{\r\n" + jsMerge +"\r\n"+prefix+"}" + js.substring(ii+p.length+4);
            }
        }
        let jss = "";
        //判断有没有子语句
        if(sm.statements && sm.statements.length>0) {
            for(let i=0;i<sm.statements.length;i++) {
                jss += this.getStatementJs(vars, sm.statements[i], prefix+"\t"+ (haveMergeCbFunc ? "\t" : ""), isArg);
            }
        }
        let ii = js.indexOf("[[statements]]");
        if(ii>0) {
            js = js.substring(0, ii) +"\r\n"+ jss + (haveMergeCbFunc ? "\t" : "") +prefix+ js.substring(ii+14);
        }
        return js;
    }

    getCommonStatementJs(vars, smType, sm, prefix, isArg) {
        let js = "";
        switch (sm.type.toLowerCase()) {
            case "if":
                let condition = sm.condition;
                js = prefix + "if(" + condition + ") {\r\n";
                if (sm.statements) {
                    for (let i = 0; i < sm.statements.length; i++) {
                        js += this.getStatementJs(vars, sm.statements[i], prefix + "\t", isArg);
                    }
                }
                js += prefix + "}";
                if(sm.elseIfStatements) {
                    for(let i=0;i<sm.elseIfStatements.length;i++) {
                        let smElseIf = sm.elseIfStatements[i];
                        js += " else if(" + smElseIf.condition + ") {\r\n";
                        if (smElseIf.statements) {
                            for (let j = 0; j < smElseIf.statements.length; j++) {
                                js += this.getStatementJs(vars, smElseIf.statements[j], prefix + "\t");
                            }
                        }
                        js += prefix + "}";
                    }
                }
                if(sm.elseStatements) {
                    js += " else {\r\n";
                    for (let i = 0; i < sm.elseStatements.length; i++) {
                        js += this.getStatementJs(vars, sm.elseStatements[i], prefix + "\t");
                    }
                    js += prefix + "}";
                }
                js += "\r\n";
                break;
            case "for":
                js = prefix + "for(let "+sm.indexVar+"="+sm.indexStart+"; "+sm.indexVar+"<"+sm.cycleNumber+"; "+sm.indexVar+"++) {\r\n";
                if (sm.statements) {
                    for (let i = 0; i < sm.statements.length; i++) {
                        js += this.getStatementJs(vars, sm.statements[i], prefix + "\t");
                    }
                }
                js += prefix + "}\r\n";
                break;
            case "iteratearray":
                let attValJs = sm.arrVar;
                js = prefix + "for(let " + sm.indexVar + " = " + sm.startIndex + "; " + sm.indexVar + "<" + attValJs + ".length; " + sm.indexVar + "++) {\r\n";
                if (sm.statements) {
                    for (let i = 0; i < sm.statements.length; i++) {
                        js += this.getStatementJs(vars, sm.statements[i], prefix + "\t");
                    }
                }
                js += prefix + "}\r\n";
                break;
            case "iterateobject":
                let objVarJs = sm.objVar;
                js = prefix + "for(let " + sm.propVar + " in " + objVarJs + ") {\r\n";
                if (sm.statements) {
                    for (let i = 0; i < sm.statements.length; i++) {
                        js += this.getStatementJs(vars, sm.statements[i], prefix + "\t");
                    }
                }
                js += prefix + "}\r\n";
                break;
            case "return":
                js = prefix + "return";
                js += (sm.retVal || sm.retVal==0) ? " " + sm.retVal : "";
                js += ";\r\n";
                break;
            case "callfunc":
                js = this.getCallFuncJs(vars, sm, prefix, isArg);
                break;
            case "comment":
                if (!sm.message) return "";
                js += prefix+"//"+sm.message+"\r\n";
                break;
            case "declare":
                vars[sm.varName] = "local";
                js += prefix+"let "+sm.varName;
                if(sm.defaultVal) js += " = " + sm.defaultVal;
                js += ";\r\n";
                break;
            case "assign":
                js += prefix+sm.varName+" "+sm.operator+" " + sm.value+";\r\n";
                break;
            case "custom":
                if (!sm.content) return "";
                if(!isArg) js = prefix;
                js += sm.content;
                if(!isArg) js += ";\r\n";
                break;
            default:
                js += prefix + this.formatExpress(smType, sm, prefix)+";\r\n";
                break;
        }
        return js;
    }

    getCallFuncJs(vars, sm, prefix, isArg) {
        let js = "";
        if (!sm.funcName) return "";
        if(!isArg || sm.statements) js = prefix;

        //如果有返回值变量
        if(sm.args && sm.args.length>0) {
            for(let i=0;i<sm.args.length;i++) {
                let arg = sm.args[i];
                if(arg.name=="retVarName" && arg.value) {
                    js += "let "+arg.value+" = ";
                    break;
                }
            }
        }
        
        if(sm.cptId) js += sm.cptId+".";
        js += sm.funcName+ "(";
        if(sm.args && sm.args.length>0) {
            for(let i=0;i<sm.args.length;i++) {
                let arg = sm.args[i];
                if(arg.name=="retVarName") continue;
                if(i>0) js += ", ";
                let argType = arg.type.toLowerCase();

                if((!arg.value || arg.value=="") && (arg.value+"")!="0" && argType!="function") {
                    //如果必须的参数为空，则不编译
                    if(arg.required) return "";
                    js += "null";
                    continue;
                }

                js += arg.value;

                /*if(argType=="int" || argType=="float" || argType=="number" || argType=="bool" 
                  || argType=="array" || argType=="object" || argType=="undefined") {
                    js += arg.value;
                } else if(argType=="px") {
                    let val = arg.value+"";
                    if(val.indexOf("px")<0 && val.indexOf("%")<0) val += "px";
                    js += val;
                } else if(argType=="date") {
                    let val = arg.value+"";
                    if(arg.value.indexOf("Date(")<0) {
                        js += "'"+arg.value+"'";
                    } else {
                        js += arg.value;
                    }
                } else if(argType=="function") {
                    if(arg.enabled) {
                        js += "function(";
                        if(arg.args) {
                            for(let j=0;j<arg.args.length;j++) {
                                let funcArg = arg.args[j];
                                if(j>0) js += ", ";
                                js += funcArg.value;
                            }
                        }
                        js += ") {\r\n";
                        if(sm.statements) {
                            for (let i = 0; i < sm.statements.length; i++) {
                                js += this.getStatementJs(vars, sm.statements[i], prefix + "\t");
                            }
                        }
                        js += prefix + "}";
                    }
                } else {
                    js += "'"+arg.value+"'";
                }*/
            }
        }
        if(sm.haveCbFunc) {
            if(sm.args && sm.args.length>0) js += " ,";
            js += " function(";
            if(sm.cbArgs && sm.cbArgs.length>0) {
                for(let i=0;i<sm.cbArgs.length;i++) {
                    js += (i>0 ? ", " : "")+sm.cbArgs[i].name;
                }
            }
            js += ") {\r\n";
            if(sm.statements) {
                for (let i = 0; i < sm.statements.length; i++) {
                    js += this.getStatementJs(vars, sm.statements[i], prefix + "\t");
                }
            }
            js += prefix + "}";
        }
        js += ");\r\n";
        return js;
    }

    addReturnVar(vars, sm) {
        let js = "";
        //如果有返回值变量
        if(sm.retVarName) {
            //如果语句格式字符串里没有retVarName参数
            if(!sm.format || sm.format.indexOf("{retVarName}")<0) {
                //如果该变量没有定义，则加上变量定义关键字，并添加到以定义变量列表中
                if(!vars[sm.retVarName]) {
                    js += "let ";
                    vars[sm.retVarName] = "local";
                }
            }
            js += sm.retVarName + " = ";
        }
        return js;
    }

    getCptStatementJs(vars, smType, sm, prefix, isArg) {
        if (!sm.cptId && !(sm.type=="tfp.use" || sm.type=="tfp.new" || sm.type=="tfp.render")) return "";
        let js = "";
        switch (sm.type) {
            case "batchSetCptAttr":
                if (!sm.attrs) return "";
                for(let attrName in sm.attrs) {
                    js += prefix+sm.cptId+".attr('"+attrName+"', "+sm.attrs[attrName]+");\r\n";
                }
                break;
            case "batchSetCptCss":
                if (!sm.styles) return "";
                for(let styleName in sm.styles) {
                    js += prefix+sm.cptId+".css('"+styleName+"', '"+sm.styles[styleName]+"');\r\n";
                }
                break;
            case "callCptFunc":
                js = this.getCallFuncJs(vars, sm, prefix, isArg);
                break;
            default:
                if(!isArg || sm.statements) js += prefix;
                js += this.addReturnVar(vars, sm);
                js += this.formatExpress(smType, sm, prefix);
                if(!isArg || sm.statements) js += ";\r\n";
                break;
        }
        return js;
    }
}

if(typeof(window)!="undefined") {
  window.guiFuncCompiler = new TFPGUIFuncCompiler();
  window.guiFuncCompiler.init();
}