import Spreadsheet from "../x-data-spreadsheet/src/index.js";
import zhCN from "../x-data-spreadsheet/src/locale/zh-cn.js";

const PAGER_SIZES = [
	['A3', 11.69, 16.54],
	['A4', 8.27, 11.69],
	['A5', 5.83, 8.27],
	['B4', 9.84, 13.90],
	['B5', 6.93, 9.84],
];

class TaskReportRender {

	constructor() {
		this.xss = null;          //x_spreadsheet插件对象
		this.trpData = null;      //报表数据模型
		this.trpFilePath = null;  //报表文件路径
		this.imageDatas = [];     //绑定完数据之后的图片信息
		this.qrCodeDatas = [];    //绑定完数据之后的二维码信息
	}

	/**
	 * 初始化报表渲染器
	 * @param  {[type]} _trpData [description]
	 * @return {[type]}          [description]
	 */
	init(_trpData) {
		//保存报表数据模型
		this.trpData = _trpData;

		//初始化x_spreadsheet插件对象
		x_spreadsheet.locale('zh-cn', zhCN);
		this.xss = x_spreadsheet('body', {
			mode: 'read', //read
			showToolbar: false,
			showGrid: true,
			showContextmenu: false,
			showBottomBar: false,
			view: {
				height: () => document.documentElement.clientHeight,
				width: () => document.documentElement.clientWidth,
			},
			row: this.trpData.pageSetting.row,
			col: this.trpData.pageSetting.col
		});
	}

	/**
	 * 获得数据源
	 * @param  {[type]} dsId 数据源id
	 * @return {[type]}      数据源对象
	 */
	getDataSource(dsId) {
		if (!this.trpData.dataSources) return null;
		for (var i = 0; i < this.trpData.dataSources.length; i++) {
			let ds = this.trpData.dataSources[i];
			if (ds.id == dsId) {
				return ds;
			}
		}
		return null;
	}

	/**
	 * 获得单元格内容
	 * @param  {[type]} data   [description]
	 * @param  {[type]} format [description]
	 * @return {[type]}        [description]
	 */
	formatCellContent(data, format) {
		let cellText = format;
		if (!cellText) return "";
		try {
			cellText = replaceDataField(data, cellText);
			if (cellText) cellText = exeExpress(cellText);
		} catch (err) {
			console.log(err);
			cellText = "";
		}
		return cellText;
	}

	/**
	 * 绑定数据集，适用于循环绑定多条数据记录
	 * @param  {[type]} cdb     [description]
	 * @param  {[type]} data    [description]
	 * @param  {[type]} rowsTmp [description]
	 * @return {[type]}         [description]
	 */
	bindDataSet(cdb, data, rowsTmp) {
		//先删除该区域所在的模板行
		for (let ri = cdb.sri; ri <= cdb.eri; ri++) {
			delete this.trpData.data[0].rows[ri];
		}
		this.dataRecords = data;  //需要动态显示的数据记录集
		this.dataStartRowIndex = cdb.sri;  //表体开始行索引
		this.dataRowLength = (cdb.eri - cdb.sri) + 1;  //表体行数
		//然后把超过当前区域的行暂存到一个临时对象里
		let rowsDown = {};
		for (let ri in this.trpData.data[0].rows) {
			if (ri == "len" || parseInt(ri) <= cdb.sri) continue;
			let row = this.trpData.data[0].rows[ri];
			rowsDown[ri] = cloneObj(row);
		}
		//删除模板行下面的行
		for (let ri in rowsDown) {
			delete this.trpData.data[0].rows[ri];
		}
		let rowIndex = cdb.sri;
		cdb.data = data;
		let newMerges = [];
		//轮询插入绑定了数据的行
		for (var i = 0; i < data.length; i++) {
			let obj = data[i];
			for (var j = 0; j < rowsTmp.length; j++) {
				let rowTmp = cloneObj(rowsTmp[j]);
				if (rowTmp.cells) {
					for (let ci in rowTmp.cells) {
						let cell = rowTmp.cells[ci];
						if (cell.text) {
							cell.text = this.formatCellContent(obj, cell.text);
						}
						//生成动态绑定的图片数据，先暂存在 this.imageDatas 里
						if (this.trpData.images) {
							for (let imgId in this.trpData.images) {
								let img = this.trpData.images[imgId];
								if (img.ri == (cdb.sri + j) && img.ci == ci) {
									img.isBinding = true;
									let imgData = cloneObj(img);
									imgData.ri = rowIndex;
									imgData.src = this.formatCellContent(obj, img.src);
									this.imageDatas.push(imgData);
								}
							}
						}
						//生成动态绑定的二维码数据，先暂存在 this.qrCodeDatas 里
						if (this.trpData.qrCodes) {
							for (let qrCodeId in this.trpData.qrCodes) {
								let qrCode = this.trpData.qrCodes[qrCodeId];
								if (qrCode.ri == (cdb.sri + j) && qrCode.ci == ci) {
									qrCode.isBinding = true;
									let qrCodeData = cloneObj(qrCode);
									qrCodeData.ri = rowIndex;
									qrCodeData.text = this.formatCellContent(obj, qrCode.text);
									this.qrCodeDatas.push(qrCodeData);
								}
							}
						}
					}
					this.trpData.data[0].rows[rowIndex] = rowTmp;
				}
				rowIndex++;
			}
			//设置合并单元格
			if (i > 0 && this.trpData.data[0].merges) {
				for (let j = 0; j < this.trpData.data[0].merges.length; j++) {
					let merge = this.trpData.data[0].merges[j].split(":");
					let sri = parseInt(merge[0].replace(/[A-z]+/gi, "")) - 1;
					let eri = parseInt(merge[1].replace(/[A-z]+/gi, "")) - 1;
					if (sri >= cdb.sri && eri <= cdb.eri) {
						newMerges.push(merge[0].replace(/\d/g, "") + (sri + i * (cdb.eri - cdb.sri + 1) + 1) + ":"
							+ merge[1].replace(/\d/g, "") + (eri + i * (cdb.eri - cdb.sri + 1) + 1));
					}
				}
			}
			//设置计算字段
			if (this.calcFields) {
				for (let fieldName in this.calcFields) {
					let field = this.calcFields[fieldName];
					let fieldVal = obj[fieldName];
					if (fieldVal) {
						fieldVal += "";
						try {
							if (fieldVal.indexOf(".") > 0) {
								fieldVal = parseFloat(fieldVal);
							} else {
								fieldVal = parseInt(fieldVal);
							}
							field.total += fieldVal;
							if (field.max == null || fieldVal > field.max) field.max = fieldVal;
							if (field.min == null || fieldVal < field.min) field.min = fieldVal;
						} catch (err) {
							console.log(err);
						}
					}
				}
			}
		}

		this.trpData.data[0].merges.push.apply(this.trpData.data[0].merges, newMerges);

		//动态数据的最后一行索引
		this.lastDataRecordIndex = rowIndex - 1;
		let footerStart = null;
		let footerEnd = null;
		if (this.trpData.pageSetting.footer) {
			footerStart = this.trpData.pageSetting.footer.start;
			footerEnd = this.trpData.pageSetting.footer.end;
			if (footerStart && !footerEnd) {
				footerEnd = footerStart;
			} else if (!footerStart && footerEnd) {
				footerStart = footerEnd;
			}
		}
		//然后行索引加上数据集数量后重新放到data里
		for (let ri in rowsDown) {
			if (footerStart && parseInt(ri) >= (footerStart - 1) && parseInt(ri) <= (footerEnd - 1)) continue;
			let row = rowsDown[ri];
			if (row.cells) {
				for (let ci in row.cells) {
					let cell = row.cells[ci];
					if (!cell.text || (cell.text + "").indexOf("#[") < 0) continue;
					let func = cell.text.substring(cell.text.indexOf("[") + 1, cell.text.indexOf("("));
					if (func == "count" || func == "总数") {
						cell.text = this.dataRecords.length;
						break;
					}
					let fieldName = cell.text.substring(cell.text.indexOf("{") + 1, cell.text.indexOf("}"));
					let field = this.calcFields[fieldName];
					if (field) {
						if (func == "sum" || func == "合计") {
							cell.text = field.total;
						} else if (func == "avg" || func == "平均值") {
							cell.text = 0;
							if (this.dataRecords.length > 0) cell.text = this.formatNumber(field.total / this.dataRecords.length);
						}
					}
				}
			}
			this.trpData.data[0].rows[rowIndex] = row;
			rowIndex++;
		}
	}

	/**
	 * 绑定指定区域的数据
	 * @param  {[type]} cdb  [description]
	 * @param  {[type]} data [description]
	 * @return {[type]}      [description]
	 */
	bindRangeData(cdb, data) {
		//先取出动态绑定区域的模板
		let rowsTmp = [];
		for (let ri = cdb.sri; ri <= cdb.eri; ri++) {
			let rowTmp = {
				cells: {}
			};
			let row = this.trpData.data[0].rows[ri + ""];
			if (!row) {
				rowsTmp.push({});
				continue;
			}
			if (row.height) rowTmp.height = row.height;
			for (let ci = cdb.sci; ci <= cdb.eci; ci++) {
				let cell = row.cells[ci];
				if (cell) {
					if (cell.text && (cell.text + "").indexOf("{") >= 0 && (cell.text + "").indexOf("}") > 0) {
						//如果没有数据可以绑定，则需要清空绑定表达式
						if (data.length == 0) cell.text = "";
					}
					rowTmp.cells[ci] = cloneObj(cell);
				}
			}
			rowsTmp.push(rowTmp);
		}
		if (data.length == 0) return;
		//console.log(rowsTmp);
		//再绑定数据
		this.bindDataSet(cdb, data, rowsTmp);
	}

	/**
	 * 获得行高
	 * @param  {[type]} row [description]
	 * @return {[type]}     [description]
	 */
	getRowHeight(row) {
		let h = parseInt(row.height ? row.height : this.trpData.pageSetting.row.height);
		return parseInt(h * 1) + 1;
	}

	/**
	 * 获得表头表尾的行模板
	 * @return {[type]} [description]
	 */
	getHeaderAndFooterTemplate() {
		this.headerRows = [];
		this.headerHeight = 0;
		this.headerMerges = [];
		if (this.trpData.pageSetting.header
			&& (this.trpData.pageSetting.header.start
				|| this.trpData.pageSetting.header.end)) {
			let headerStart = this.trpData.pageSetting.header.start;
			let headerEnd = this.trpData.pageSetting.header.end;
			if (headerStart && !headerEnd) {
				headerEnd = headerStart;
			} else if (!headerStart && headerEnd) {
				headerStart = headerEnd;
			}
			for (let ri = headerStart; ri <= headerEnd; ri++) {
				let row = cloneObj(this.trpData.data[0].rows[ri - 1]);
				if(!row) continue;
				row.ri = ri;
				this.headerHeight += this.getRowHeight(row);
				this.headerRows.push(row);
			}
			//如果有单元格合并
			if (this.trpData.data[0].merges) {
				for (let i = 0; i < this.trpData.data[0].merges.length; i++) {
					let merge = this.trpData.data[0].merges[i].split(":");
					let sri = parseInt(merge[0].replace(/[A-z]+/gi, ""));
					let eri = parseInt(merge[1].replace(/[A-z]+/gi, ""));
					if (sri >= headerStart && sri <= headerEnd && eri >= headerStart && eri <= headerEnd) {
						this.headerMerges.push({
							sri: sri,
							eri: eri,
							sci: merge[0].replace(/\d/g, ""),
							eci: merge[1].replace(/\d/g, "")
						});
					}
				}
			}
		}

		this.footerRows = [];
		this.footerHeight = 0;
		this.calcFields = {};
		this.dataCount = 0;
		if (this.trpData.pageSetting.footer
			&& (this.trpData.pageSetting.footer.start
				|| this.trpData.pageSetting.footer.end)) {
			let footerStart = this.trpData.pageSetting.footer.start;
			let footerEnd = this.trpData.pageSetting.footer.end;
			if (footerStart && !footerEnd) {
				footerEnd = footerStart;
			} else if (!footerStart && footerEnd) {
				footerStart = footerEnd;
			}
			for (let ri = footerStart; ri <= footerEnd; ri++) {
				let row = cloneObj(this.trpData.data[0].rows[ri - 1]);
				for (let ci in row.cells) {
					let cell = row.cells[ci];
					//如果有计算列，则将该计算列的信息先保存起来，后续渲染时从这里取值
					if (cell.text && cell.text.indexOf("#[") >= 0 && cell.text.indexOf("count()") < 0
						&& cell.text.indexOf("pageIndex()") < 0 && cell.text.indexOf("页码()") < 0) {
						let calcField = {
							ri: ri - footerStart,
							ci: ci,
							func: cell.text.substring(cell.text.indexOf("#[") + 2, cell.text.indexOf("(")),
							field: cell.text.indexOf("{") >= 0 ? cell.text.substring(cell.text.indexOf("{") + 1, cell.text.indexOf("}")) : '',
							total: 0,
							max: null,
							min: null
						};
						this.calcFields[calcField.field] = calcField;
					}
				}
				this.footerHeight += this.getRowHeight(row);
				this.footerRows.push(row);
			}
		}
	}

	/**
	 * 格式化数字
	 * @param  {[type]} s [description]
	 * @param  {[type]} n [description]
	 * @param  {[type]} t [description]
	 * @return {[type]}   [description]
	 */
	formatNumber(s, n, t) {
		n = n > 0 && n <= 20 ? n : 2;
		s = parseFloat((s + "").replace(/[^\d\.-]/g, "")).toFixed(n) + "";
		if (!t) return s;
		var l = s.split(".")[0].split("").reverse();
		var r = s.split(".")[1];
		var t = "";
		for (var i = 0; i < l.length; i++) {
			t += l[i] + ((i + 1) % 3 === 0 && i + 1 != l.length ? "," : "");
		}
		return t.split("").reverse().join("") + "." + r;
	}

	/**
	 * 格式化表尾统计行
	 * @param  {[type]} pageDataCount  [description]
	 * @param  {[type]} pageCalcFields [description]
	 * @param  {[type]} rowIndex       [description]
	 * @param  {[type]} row            [description]
	 * @return {[type]}                [description]
	 */
	formatFooterStatRow(pageDataCount, pageCalcFields, rowIndex, row, pageIndex) {
		if (!row.cells) return;
		for (let fieldName in this.calcFields) {
			let field = this.calcFields[fieldName];
			if (field.ri == rowIndex) {
				let cellVal = 0;
				if (field.func == "sum" || field.func == "合计") {
					cellVal = this.formatNumber(field.total);
				} else if (field.func == "subtotal" || field.func == "小计") {
					cellVal = this.formatNumber(pageCalcFields[fieldName].total);
				} else if (field.func == "avg" || field.func == "合计平均") {
					if (this.dataRecords.length == 0) {
						cellVal = 0;
					} else {
						cellVal = this.formatNumber(field.total / this.dataRecords.length);
					}
				} else if (field.func == "subavg" || field.func == "小计平均") {
					if (pageDataCount == 0) {
						cellVal = 0;
					} else {
						cellVal = this.formatNumber(pageCalcFields[fieldName].total / pageDataCount);
					}
				} else if (field.func == "count" || field.func == "总数") {
					cellVal = this.dataRecords.length;
				} else if (field.func == "subcount" || field.func == "小计总数") {
					cellVal = pageDataCount;
				}
				row.cells[field.ci].text = cellVal;
			}
		}
		for (let ci in row.cells) {
			let cell = row.cells[ci];
			if (!cell.text) continue;
			if (cell.text.indexOf("#[pageIndex()]") >= 0) {
				cell.text = cell.text.replace("#[pageIndex()]", pageIndex);
			} else if (cell.text.indexOf("#[页码()]") >= 0) {
				cell.text = cell.text.replace("#[页码()]", pageIndex);
			}
		}
	}

	/**
	 * 插入表头和表尾
	 * @return {[type]} [description]
	 */
	insertHeaderAndFooter() {
		if (this.headerRows.length == 0 && this.footerRows.length == 0) return;

		let pageHeight = 0;
		//如果是自定义纸张大小
		if ((this.trpData.pageSetting.pageSize + "") == "-1") {
			pageHeight = this.trpData.pageSetting.pageHeight;
			if (this.trpData.pageSetting.orientation == "landscape") pageHeight = this.trpData.pageSetting.pageWidth;
			pageHeight = parseInt(96 * parseInt(pageHeight));
		} else {
			let pageSize = PAGER_SIZES[1];
			if (this.trpData.pageSetting.pageSize || this.trpData.pageSetting.pageSize == 0) {
				pageSize = PAGER_SIZES[this.trpData.pageSetting.pageSize];
			}
			pageHeight = pageSize[2];
			if (this.trpData.pageSetting.orientation == "landscape") pageHeight = pageSize[1];
			pageHeight = parseInt(96 * pageHeight);
		}
		//计算页面内容区高度
		let contentHeight = pageHeight - this.trpData.pageSetting.paddingTop * 2;
		contentHeight = parseInt(contentHeight * window.devicePixelRatio);
		let rows = [];
		let pageIndex = 1;
		let offset = 0;
		let keys = Object.keys(this.trpData.data[0].rows);
		let page = {
			start: 0,
			end: 0
		};
		let isEndData = false;
		let dataSri = this.dataStartRowIndex;  //数据模板行起止行索引
		let dataIndex = 0;  //当前数据记录索引
		//循环所有数据行
		for (let ri = 0; ri < keys.length; ri++) {
			if (keys[ri] == "len") continue;
			let row = this.trpData.data[0].rows[keys[ri]];
			let rowHeight = parseInt(parseInt(this.getRowHeight(row)) * window.devicePixelRatio);
			let totalHeight = offset + rowHeight;
			if (this.trpData.pageSetting.footer && this.trpData.pageSetting.footer.position == "all" && !isEndData)
				totalHeight += this.footerHeight;
			//如果加上新行后的高度超过当前页整个内容区的高度，就需要重新分页
			if (totalHeight > contentHeight) {
				//添加每一页的表尾
				if (this.trpData.pageSetting.footer && this.trpData.pageSetting.footer.position == "all" && !isEndData) {
					//计算当前页的统计数据
					let pageCalcFields = {};
					if (this.calcFields) {
						for (let fieldName in this.calcFields) {
							pageCalcFields[fieldName] = {
								name: fieldName,
								total: 0,
								max: null,
								min: null
							};
						}
					}
					//计算本页要显示多少条数据记录
					let curPageDataLenght = (ri - dataSri) / this.dataRowLength; //this.dataRowLength 需要循环显示的模板行行数
					for (; dataIndex < curPageDataLenght; dataIndex++) {
						let dataRecord = this.dataRecords[dataIndex];
						//设置计算字段
						for (let fieldName in pageCalcFields) {
							let field = pageCalcFields[fieldName];
							let fieldVal = dataRecord[fieldName];
							if (fieldVal) {
								fieldVal += "";
								try {
									if (fieldVal.indexOf(".") > 0) {
										fieldVal = parseFloat(fieldVal);
									} else {
										fieldVal = parseInt(fieldVal);
									}
									field.total += fieldVal;
									if (field.max == null || fieldVal > field.max) field.max = fieldVal;
									if (field.min == null || fieldVal < field.min) field.min = fieldVal;
								} catch (err) {
									console.log(err);
								}
							}
						}
						dataSri += this.dataRowLength;
					}

					for (let i = 0; i < this.footerRows.length; i++) {
						let row = cloneObj(this.footerRows[i]);
						rows.push(row);
						this.formatFooterStatRow(curPageDataLenght, pageCalcFields, i, row, pageIndex);
						this.lastDataRecordIndex++;
					}
				}
				page.end = rows.length - 1;
				this.pages.push([page.start, page.end]);
				page.start = rows.length;
				page.end = page.start;

				offset = 0;
				//添加每一页的表头（不含第1页）
				if (this.trpData.pageSetting.header.position == "all" && !isEndData) {
					//插入合并的行
					for (let i = 0; i < this.headerMerges.length; i++) {
						let merge = this.headerMerges[i];
						this.trpData.data[0].merges.push(
							merge.sci + (merge.sri + rows.length - 1) + ":"
							+ merge.eci + (merge.eri + rows.length - 1));
					}
					for (let i = 0; i < this.headerRows.length; i++) {
						let headerRow = this.headerRows[i];
						rows.push(cloneObj(headerRow));
						this.lastDataRecordIndex++;
						//生成动态绑定的图片数据，先暂存在 this.imageDatas 里
						if (this.trpData.images) {
							for (let imgId in this.trpData.images) {
								let img = this.trpData.images[imgId];
								if ((img.ri+1) == headerRow.ri) {
									img.isBinding = true;
									let imgData = cloneObj(img);
									imgData.ri = rows.length - 1;
									this.imageDatas.push(imgData);
								}
							}
						}
						//生成动态绑定的二维码数据，先暂存在 this.qrCodeDatas 里
						if (this.trpData.qrCodes) {
							for (let qrCodeId in this.trpData.qrCodes) {
								let qrCode = this.trpData.qrCodes[qrCodeId];
								if ((qrCode.ri+1) == headerRow.ri) {
									qrCode.isBinding = true;
									let qrCodeData = cloneObj(qrCode);
									qrCodeData.ri = rows.length - 1;
									this.qrCodeDatas.push(qrCodeData);
								}
							}
						}
					}
					offset = this.headerHeight;
				}
				ri--;
				pageIndex++;
				continue;
			}

			rows.push(cloneObj(row));
			offset += rowHeight;

			if ((rows.length - 1) == this.lastDataRecordIndex) {
				//计算当前页的统计数据
				let pageCalcFields = {};
				if (this.calcFields) {
					for (let fieldName in this.calcFields) {
						pageCalcFields[fieldName] = {
							name: fieldName,
							total: 0,
							max: null,
							min: null
						};
					}
				}
				let curPageDataLenght = (ri - dataSri) / this.dataRowLength;
				for (; dataIndex <= curPageDataLenght; dataIndex++) {
					let dataRow = this.dataRecords[dataIndex];
					//设置计算字段
					for (let fieldName in pageCalcFields) {
						let field = pageCalcFields[fieldName];
						let fieldVal = dataRow[fieldName];
						if (fieldVal) {
							fieldVal += "";
							try {
								if (fieldVal.indexOf(".") > 0) {
									fieldVal = parseFloat(fieldVal);
								} else {
									fieldVal = parseInt(fieldVal);
								}
								field.total += fieldVal;
								if (field.max == null || fieldVal > field.max) field.max = fieldVal;
								if (field.min == null || fieldVal < field.min) field.min = fieldVal;
							} catch (err) {
								console.log(err);
							}
						}
					}
					//dataSri += this.dataRowLength;
				}
				//添加最后一页的表尾
				for (let i = 0; i < this.footerRows.length; i++) {
					let row = cloneObj(this.footerRows[i]);
					rows.push(row);
					this.formatFooterStatRow(curPageDataLenght, pageCalcFields, i, row, pageIndex);
				}
				offset += this.footerHeight;
				isEndData = true;
			}
		}
		this.pages.push([page.start, rows.length-1]);
		this.trpData.pages = this.pages;
		//然后行索引加上数据集数量后重新放到data里
		for (let ri = 0; ri < rows.length; ri++) {
			this.trpData.data[0].rows[ri] = rows[ri];
		}
		this.trpData.data[0].rows["len"] = rows.length;
		//console.log(this.trpData.data[0].rows);
	}

	getCellOffset(cellRi, cellCi) {
		let offset = {
			left: 60,  //行索引列宽度
			top: this.trpData.pageSetting.row.height    //列标题高度
		};
		let row = this.trpData.data[0].rows[cellRi];
		if(row && row.cells) {
			for(let ci in row.cells) {
				if (parseInt(ci) >= cellCi) break;
				let cell = row.cells[ci];
				if(cell) {
					let cellWidthSetting = this.trpData.data[0].cols[ci];
					if(cellWidthSetting && (cellWidthSetting.width || cellWidthSetting.width==0)) {
						offset.left += cellWidthSetting.width;
					} else {  //没有找到，就用默认的列宽
						offset.left += this.trpData.pageSetting.col.width;
					}
				}
			}
		}
		for(let ri in this.trpData.data[0].rows) {
			if(ri=="len") continue;
			if (parseInt(ri) >= cellRi) break;
			let rowTmp = this.trpData.data[0].rows[ri];
			if(rowTmp.height || rowTmp.height==0) {
				offset.top += rowTmp.height;
			} else {  //没有找到，就用默认的列宽
				offset.top += this.trpData.pageSetting.row.height;
			}
		}
		return offset;
	}

	/**
	 * 显示指定图片
	 * @param  {[type]} imgData [description]
	 * @return {[type]}         [description]
	 */
	showImage(imgData) {
		let offset = this.getCellOffset(imgData.ri, imgData.ci);
		if (imgData.top) offset.top += imgData.top;
		if (imgData.left) offset.left += imgData.left;
		imgData.previewTop = offset.top;
		imgData.left = offset.left;
		
		let zIndex = 1;
		if (imgData.zIndex) zIndex = imgData.zIndex;

		$(this.xss.sheet.el.el).append("<img id=\"imgCell_" + imgData.ri + "_" + imgData.ci + "\" class=\"image-cell\" "
			+ "style=\"position:absolute; z-index:" + zIndex + "; left:" + offset.left + "px; top:" + offset.top + "px;width:"
			+ imgData.width + "px;height:" + imgData.height + "px;\" src=\"" + imgData.src + "\">");
	}

	/**
	 * 显示所有图片
	 * @return {[type]} [description]
	 */
	showImages() {
		for (var i = 0; i < this.imageDatas.length; i++) {
			this.showImage(this.imageDatas[i]);
		}
	}

	/**
	 * 界面滚动时，重置图片位置
	 * @param  {[type]} scrollLeft [description]
	 * @param  {[type]} scrollTop  [description]
	 * @return {[type]}            [description]
	 */
	resetImagesOffset(offset) {
		for (var i = 0; i < this.imageDatas.length; i++) {
			let img = this.imageDatas[i];
			let imgEl = $("#imgCell_" + img.ri + "_" + img.ci);
			imgEl.css("left", (img.previewLeft + offset.left) + "px");
			imgEl.css("top", (img.previewTop + offset.top) + "px");
		}
	}

	/**
	 * 显示指定二维码
	 * @param  {[type]} code [description]
	 * @return {[type]}      [description]
	 */
	showQRCode(code) {
		let offset = this.getCellOffset(code.ri, code.ci);
		if (code.top) offset.top += code.top;
		if (code.left) offset.left += code.left;
		code.previewTop = offset.top;
		code.left = offset.left;

		let zIndex = 1;
		if (code.zIndex) zIndex = code.zIndex;

		$(this.xss.sheet.el.el).append("<div id=\"qrCell_" + code.ri + "_" + code.ci + "\" class=\"qrcode-cell\" "
			+ "style=\"position:absolute; z-index:" + zIndex + "; left:" + offset.left + "px; top:" + code.previewTop + "px;width:"
			+ code.width + "px;height:" + code.height + "px; border:1px solid #cccccc;\"></div>");
		code.obj = new QRCode("qrCell_" + code.ri + "_" + code.ci, {
			text: code.text,
			width: code.width,
			height: code.height,
			colorDark: "#000000",
			colorLight: "#ffffff",
			correctLevel: QRCode.CorrectLevel[code.level]
		});
	}

	/**
	 * 显示全部二维码
	 * @return {[type]} [description]
	 */
	showQRCodes() {
		for (var i = 0; i < this.qrCodeDatas.length; i++) {
			this.showQRCode(this.qrCodeDatas[i]);
		}
	}

	/**
	 * 重置二维码显示位置
	 * @param  {[type]} offset [description]
	 * @return {[type]}        [description]
	 */
	resetQRCodesOffset(offset) {
		for (var i = 0; i < this.qrCodeDatas.length; i++) {
			let code = this.qrCodeDatas[i];
			let codeEl = $("#qrCell_" + code.ri + "_" + code.ci);
			codeEl.css("left", (code.previewLeft + offset.left) + "px");
			codeEl.css("top", (code.previewTop + offset.top) + "px");
		}
	}

	/**
	 * 格式化数据绑定表达式
	 * @param  {[type]} data    [description]
	 * @param  {[type]} express [description]
	 * @return {[type]}         [description]
	 */
	formatExpress(data, express) {
		let val = express;
		if (!val || !((val + "").indexOf("{") >= 0 && (val + "").indexOf("}") > 0)) return val;
		try {
			val = replaceDataField(data, val);
			if (val) val = exeExpress(val);
		} catch (err) {
			console.log(err);
		}
		return val;
	}

	/**
	 * 绑定单元格数据，适用于：单个单元格绑定数据、多个单元格绑定单条数据记录
	 * @param  {[type]} ri     行索引
	 * @param  {[type]} ci     列索引
	 * @param  {[type]} data   要绑定的数据
	 * @return {[type]}        无
	 */
	bindCellData(ri, ci, data) {
		let row = this.trpData.data[0].rows[ri + ""];
		if (row && row.cells) {
			let cell = row.cells[ci + ""];
			if (cell) {
				cell.text = this.formatExpress(data, cell.text);
			}
		}

		if (this.trpData.images) {
			for (let imgId in this.trpData.images) {
				let img = this.trpData.images[imgId];
				//如果已经生成动态绑定的图片，则跳过
				if (img.isBinding) continue;
				if (img.ri == ri && img.ci == ci) {
					img.src = this.formatExpress(data, img.src);
					this.imageDatas.push(img);
				}
			}
		}

		if (this.trpData.qrCodes) {
			for (let codeId in this.trpData.qrCodes) {
				let code = this.trpData.qrCodes[codeId];
				//如果已经生成动态绑定的图片，则跳过
				if (code.isBinding) continue;
				if (code.ri == ri && code.ci == ci) {
					code.text = this.formatExpress(data, code.text);
					this.qrCodeDatas.push(code);
				}
			}
		}
	}

	/**
	 * 绑定报表数据
	 * @return {[type]} 无
	 */
	bindData() {
		this.pages = [];

		//轮询单元格数据绑定设置
		for (let cellId in this.trpData.dataBindCells) {
			let cdb = this.trpData.dataBindCells[cellId];

			//获得数据源
			let ds = this.getDataSource(cdb.dataSource);
			if (!ds || !ds.data) continue;

			//获得要绑定的数据
			let data = ds.data;
			if (cdb.dataBindMember) data = ds.data[cdb.dataBindMember];
			if (!data) continue;

			//如果数据绑定对象是多个单元格
			if (cdb.type == "range") {
				//如果要绑定多行数据集
				if (Array.isArray(data)) {
					//暂不处理，等表头表尾的数据绑定完再处理
				} else {  //如果要绑定单条数据记录
					for (var ri = cdb.sri; ri <= cdb.eri; ri++) {
						for (var ci = cdb.sci; ci <= cdb.eci; ci++) {
							this.bindCellData(ri, ci, data);
						}
					}
				}
			} else {  //如果数据绑定对象是单个单元格
				if (Array.isArray(data)) continue;  //单个单元格不能绑定多行记录
				this.bindCellData(cdb.ri, cdb.ci, data);
			}
		}

		//获得表头和表尾的数据模板
		this.getHeaderAndFooterTemplate();

		//轮询单元格数据绑定设置
		for (let cellId in this.trpData.dataBindCells) {
			let cdb = this.trpData.dataBindCells[cellId];

			//获得数据源
			let ds = this.getDataSource(cdb.dataSource);
			if (!ds || !ds.data) continue;

			//获得要绑定的数据
			let data = ds.data;
			if (cdb.dataBindMember) data = ds.data[cdb.dataBindMember];
			if (!data) continue;

			//如果数据绑定对象是多个单元格
			if (cdb.type == "range") {
				//如果要绑定多行数据集
				if (Array.isArray(data)) {
					this.bindRangeData(cdb, data);
				}
			}
		}

		//插入表头和表尾数据
		this.insertHeaderAndFooter();

		//加载报表数据并渲染
		this.xss.loadData(this.trpData.data);
		this.xss.reRender();

		let that = this;

		//界面滚动时需要重置图片和二维码的显示位置
		$(".x-spreadsheet-scrollbar").each(function () {
			this.onscroll = function (e) {
				let offset = that.xss.sheet.editor.areaOffset;
				that.resetImagesOffset(offset);
				that.resetQRCodesOffset(offset);
			}
		});

		//等绑定完报表数据，再显示图片和二维码，以免被报表内容覆盖
		this.showImages();
		this.showQRCodes();

		if (this.trpData.pageSetting.bgImage) {
			$(this.xss.sheet.el.el).css("background", "url(" + this.trpData.pageSetting.bgImage + ")");
			$(this.xss.sheet.el.el).css("background-repeat", "no-repeat");
			$(this.xss.sheet.el.el).css("background-attachment", "fixed");
			$(this.xss.sheet.el.el).css("background-position", "60px 30px");
		}
	}
}

window.trpRender = new TaskReportRender();