/*! @azure/msal-browser v2.22.1 2022-03-07 */
'use strict';
import { __extends, __awaiter, __generator, __assign } from '../_virtual/_tslib.js';
import { AuthError, StringUtils, PromptValue } from '@azure/msal-common';
import { StandardInteractionClient } from './StandardInteractionClient.js';
import { BrowserAuthError } from '../error/BrowserAuthError.js';
import { InteractionType } from '../utils/BrowserConstants.js';
import { SilentHandler } from '../interaction_handler/SilentHandler.js';

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 */
var SilentIframeClient = /** @class */ (function (_super) {
    __extends(SilentIframeClient, _super);
    function SilentIframeClient(config, storageImpl, browserCrypto, logger, eventHandler, navigationClient, apiId, correlationId) {
        var _this = _super.call(this, config, storageImpl, browserCrypto, logger, eventHandler, navigationClient, correlationId) || this;
        _this.apiId = apiId;
        return _this;
    }
    /**
     * Acquires a token silently by opening a hidden iframe to the /authorize endpoint with prompt=none
     * @param request
     */
    SilentIframeClient.prototype.acquireToken = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var silentRequest, serverTelemetryManager, authCodeRequest, authClient, navigateUrl, e_1;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.logger.verbose("acquireTokenByIframe called");
                        // Check that we have some SSO data
                        if (StringUtils.isEmpty(request.loginHint) && StringUtils.isEmpty(request.sid) && (!request.account || StringUtils.isEmpty(request.account.username))) {
                            this.logger.warning("No user hint provided. The authorization server may need more information to complete this request.");
                        }
                        // Check that prompt is set to none, throw error if it is set to anything else.
                        if (request.prompt && request.prompt !== PromptValue.NONE) {
                            throw BrowserAuthError.createSilentPromptValueError(request.prompt);
                        }
                        return [4 /*yield*/, this.initializeAuthorizationRequest(__assign(__assign({}, request), { prompt: PromptValue.NONE }), InteractionType.Silent)];
                    case 1:
                        silentRequest = _a.sent();
                        this.browserStorage.updateCacheEntries(silentRequest.state, silentRequest.nonce, silentRequest.authority, silentRequest.loginHint || "", silentRequest.account || null);
                        serverTelemetryManager = this.initializeServerTelemetryManager(this.apiId);
                        _a.label = 2;
                    case 2:
                        _a.trys.push([2, 7, , 8]);
                        return [4 /*yield*/, this.initializeAuthorizationCodeRequest(silentRequest)];
                    case 3:
                        authCodeRequest = _a.sent();
                        return [4 /*yield*/, this.createAuthCodeClient(serverTelemetryManager, silentRequest.authority, silentRequest.azureCloudOptions)];
                    case 4:
                        authClient = _a.sent();
                        this.logger.verbose("Auth code client created");
                        return [4 /*yield*/, authClient.getAuthCodeUrl(silentRequest)];
                    case 5:
                        navigateUrl = _a.sent();
                        return [4 /*yield*/, this.silentTokenHelper(navigateUrl, authCodeRequest, authClient, this.logger)];
                    case 6: return [2 /*return*/, _a.sent()];
                    case 7:
                        e_1 = _a.sent();
                        if (e_1 instanceof AuthError) {
                            e_1.setCorrelationId(this.correlationId);
                        }
                        serverTelemetryManager.cacheFailedRequest(e_1);
                        this.browserStorage.cleanRequestByState(silentRequest.state);
                        throw e_1;
                    case 8: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Currently Unsupported
     */
    SilentIframeClient.prototype.logout = function () {
        // Synchronous so we must reject
        return Promise.reject(BrowserAuthError.createSilentLogoutUnsupportedError());
    };
    /**
     * Helper which acquires an authorization code silently using a hidden iframe from given url
     * using the scopes requested as part of the id, and exchanges the code for a set of OAuth tokens.
     * @param navigateUrl
     * @param userRequestScopes
     */
    SilentIframeClient.prototype.silentTokenHelper = function (navigateUrl, authCodeRequest, authClient, browserRequestLogger) {
        return __awaiter(this, void 0, void 0, function () {
            var silentHandler, msalFrame, hash, state;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        silentHandler = new SilentHandler(authClient, this.browserStorage, authCodeRequest, browserRequestLogger, this.config.system.navigateFrameWait);
                        return [4 /*yield*/, silentHandler.initiateAuthRequest(navigateUrl)];
                    case 1:
                        msalFrame = _a.sent();
                        return [4 /*yield*/, silentHandler.monitorIframeForHash(msalFrame, this.config.system.iframeHashTimeout)];
                    case 2:
                        hash = _a.sent();
                        state = this.validateAndExtractStateFromHash(hash, InteractionType.Silent, authCodeRequest.correlationId);
                        // Handle response from hash string
                        return [2 /*return*/, silentHandler.handleCodeResponseFromHash(hash, state, authClient.authority, this.networkClient)];
                }
            });
        });
    };
    return SilentIframeClient;
}(StandardInteractionClient));

export { SilentIframeClient };
//# sourceMappingURL=SilentIframeClient.js.map
