/*! @azure/msal-browser v2.22.1 2022-03-07 */
'use strict';
import { Authority, IdTokenEntity, AuthToken, AccountEntity, ScopeSet, AccessTokenEntity } from '@azure/msal-common';
import { BrowserAuthError } from '../error/BrowserAuthError.js';

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 */
/**
 * Token cache manager
 */
var TokenCache = /** @class */ (function () {
    function TokenCache(configuration, storage, logger, cryptoObj) {
        this.isBrowserEnvironment = typeof window !== "undefined";
        this.config = configuration;
        this.storage = storage;
        this.logger = logger;
        this.cryptoObj = cryptoObj;
    }
    // Move getAllAccounts here and cache utility APIs
    /**
     * API to load tokens to msal-browser cache.
     * @param request
     * @param response
     * @param options
     */
    TokenCache.prototype.loadExternalTokens = function (request, response, options) {
        this.logger.info("TokenCache - loadExternalTokens called");
        if (!response.id_token) {
            throw BrowserAuthError.createUnableToLoadTokenError("Please ensure server response includes id token.");
        }
        if (request.account) {
            this.loadIdToken(response.id_token, request.account.homeAccountId, request.account.environment, request.account.tenantId, options);
            this.loadAccessToken(request, response, request.account.homeAccountId, request.account.environment, request.account.tenantId, options);
        }
        else if (request.authority) {
            var authorityUrl = Authority.generateAuthority(request.authority, request.azureCloudOptions);
            var authorityOptions = {
                protocolMode: this.config.auth.protocolMode,
                knownAuthorities: this.config.auth.knownAuthorities,
                cloudDiscoveryMetadata: this.config.auth.cloudDiscoveryMetadata,
                authorityMetadata: this.config.auth.authorityMetadata,
            };
            var authority = new Authority(authorityUrl, this.config.system.networkClient, this.storage, authorityOptions);
            // "clientInfo" from options takes precedence over "clientInfo" in response
            if (options.clientInfo) {
                this.logger.trace("TokenCache - homeAccountId from options");
                this.loadIdToken(response.id_token, options.clientInfo, authority.hostnameAndPort, authority.tenant, options);
                this.loadAccessToken(request, response, options.clientInfo, authority.hostnameAndPort, authority.tenant, options);
            }
            else if (response.client_info) {
                this.logger.trace("TokenCache - homeAccountId from response");
                this.loadIdToken(response.id_token, response.client_info, authority.hostnameAndPort, authority.tenant, options);
                this.loadAccessToken(request, response, response.client_info, authority.hostnameAndPort, authority.tenant, options);
            }
            else {
                throw BrowserAuthError.createUnableToLoadTokenError("Please provide clientInfo in the response or options.");
            }
        }
        else {
            throw BrowserAuthError.createUnableToLoadTokenError("Please provide a request with an account or a request with authority.");
        }
    };
    /**
     * Helper function to load id tokens to msal-browser cache
     * @param idToken
     * @param homeAccountId
     * @param environment
     * @param tenantId
     * @param options
     */
    TokenCache.prototype.loadIdToken = function (idToken, homeAccountId, environment, tenantId, options) {
        var idTokenEntity = IdTokenEntity.createIdTokenEntity(homeAccountId, environment, idToken, this.config.auth.clientId, tenantId);
        var idAuthToken = new AuthToken(idToken, this.cryptoObj);
        var accountEntity = options.clientInfo ?
            AccountEntity.createAccount(options.clientInfo, homeAccountId, idAuthToken, undefined, undefined, undefined, undefined, environment) :
            AccountEntity.createGenericAccount(homeAccountId, idAuthToken, undefined, undefined, undefined, undefined, environment);
        if (this.isBrowserEnvironment) {
            this.logger.verbose("TokenCache - loading id token");
            this.storage.setAccount(accountEntity);
            this.storage.setIdTokenCredential(idTokenEntity);
        }
        else {
            throw BrowserAuthError.createUnableToLoadTokenError("loadExternalTokens is designed to work in browser environments only.");
        }
    };
    /**
     * Helper function to load access tokens to msal-browser cache
     * @param request
     * @param response
     * @param options
     * @param homeAccountId
     * @param environment
     * @param tenantId
     * @returns
     */
    TokenCache.prototype.loadAccessToken = function (request, response, homeAccountId, environment, tenantId, options) {
        if (!response.access_token) {
            this.logger.verbose("TokenCache - No access token provided for caching");
            return;
        }
        if (!response.expires_in) {
            throw BrowserAuthError.createUnableToLoadTokenError("Please ensure server response includes expires_in value.");
        }
        if (!options.extendedExpiresOn) {
            throw BrowserAuthError.createUnableToLoadTokenError("Please provide an extendedExpiresOn value in the options.");
        }
        var scopes = new ScopeSet(request.scopes).printScopes();
        var expiresOn = response.expires_in;
        var extendedExpiresOn = options.extendedExpiresOn;
        var accessTokenEntity = AccessTokenEntity.createAccessTokenEntity(homeAccountId, environment, response.access_token, this.config.auth.clientId, tenantId, scopes, expiresOn, extendedExpiresOn, this.cryptoObj);
        if (this.isBrowserEnvironment) {
            this.logger.verbose("TokenCache - loading access token");
            this.storage.setAccessTokenCredential(accessTokenEntity);
        }
        else {
            throw BrowserAuthError.createUnableToLoadTokenError("loadExternalTokens is designed to work in browser environments only.");
        }
    };
    return TokenCache;
}());

export { TokenCache };
//# sourceMappingURL=TokenCache.js.map
