/*! @azure/msal-browser v2.22.1 2022-03-07 */
'use strict';
import { __awaiter, __generator } from '../_virtual/_tslib.js';
import { CryptoOps } from '../crypto/CryptoOps.js';
import { StringUtils, Logger, DEFAULT_CRYPTO_IMPLEMENTATION, Constants, ServerError, InteractionRequiredAuthError } from '@azure/msal-common';
import { BrowserCacheManager, DEFAULT_BROWSER_CACHE_MANAGER } from '../cache/BrowserCacheManager.js';
import { buildConfiguration } from '../config/Configuration.js';
import { InteractionType, TemporaryCacheKeys, ApiId, BrowserCacheLocation, BrowserConstants } from '../utils/BrowserConstants.js';
import { BrowserUtils } from '../utils/BrowserUtils.js';
import { name, version } from '../packageMetadata.js';
import { EventType } from '../event/EventType.js';
import { BrowserConfigurationAuthError } from '../error/BrowserConfigurationAuthError.js';
import { EventHandler } from '../event/EventHandler.js';
import { PopupClient } from '../interaction_client/PopupClient.js';
import { RedirectClient } from '../interaction_client/RedirectClient.js';
import { SilentIframeClient } from '../interaction_client/SilentIframeClient.js';
import { SilentRefreshClient } from '../interaction_client/SilentRefreshClient.js';
import { TokenCache } from '../cache/TokenCache.js';
import { SilentAuthCodeClient } from '../interaction_client/SilentAuthCodeClient.js';
import { BrowserAuthError } from '../error/BrowserAuthError.js';

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 */
var ClientApplication = /** @class */ (function () {
    /**
     * @constructor
     * Constructor for the PublicClientApplication used to instantiate the PublicClientApplication object
     *
     * Important attributes in the Configuration object for auth are:
     * - clientID: the application ID of your application. You can obtain one by registering your application with our Application registration portal : https://portal.azure.com/#blade/Microsoft_AAD_IAM/ActiveDirectoryMenuBlade/RegisteredAppsPreview
     * - authority: the authority URL for your application.
     * - redirect_uri: the uri of your application registered in the portal.
     *
     * In Azure AD, authority is a URL indicating the Azure active directory that MSAL uses to obtain tokens.
     * It is of the form https://login.microsoftonline.com/{Enter_the_Tenant_Info_Here}
     * If your application supports Accounts in one organizational directory, replace "Enter_the_Tenant_Info_Here" value with the Tenant Id or Tenant name (for example, contoso.microsoft.com).
     * If your application supports Accounts in any organizational directory, replace "Enter_the_Tenant_Info_Here" value with organizations.
     * If your application supports Accounts in any organizational directory and personal Microsoft accounts, replace "Enter_the_Tenant_Info_Here" value with common.
     * To restrict support to Personal Microsoft accounts only, replace "Enter_the_Tenant_Info_Here" value with consumers.
     *
     * In Azure B2C, authority is of the form https://{instance}/tfp/{tenant}/{policyName}/
     * Full B2C functionality will be available in this library in future versions.
     *
     * @param configuration Object for the MSAL PublicClientApplication instance
     */
    function ClientApplication(configuration) {
        /*
         * If loaded in an environment where window is not available,
         * set internal flag to false so that further requests fail.
         * This is to support server-side rendering environments.
         */
        this.isBrowserEnvironment = typeof window !== "undefined";
        // Set the configuration.
        this.config = buildConfiguration(configuration, this.isBrowserEnvironment);
        // Initialize logger
        this.logger = new Logger(this.config.system.loggerOptions, name, version);
        // Initialize the network module class.
        this.networkClient = this.config.system.networkClient;
        // Initialize the navigation client class.
        this.navigationClient = this.config.system.navigationClient;
        // Initialize redirectResponse Map
        this.redirectResponse = new Map();
        // Initial hybrid spa map
        this.hybridAuthCodeResponses = new Map();
        // Initialize the crypto class.
        this.browserCrypto = this.isBrowserEnvironment ? new CryptoOps(this.logger) : DEFAULT_CRYPTO_IMPLEMENTATION;
        this.eventHandler = new EventHandler(this.logger, this.browserCrypto);
        // Initialize the browser storage class.
        this.browserStorage = this.isBrowserEnvironment ?
            new BrowserCacheManager(this.config.auth.clientId, this.config.cache, this.browserCrypto, this.logger) :
            DEFAULT_BROWSER_CACHE_MANAGER(this.config.auth.clientId, this.logger);
        // Initialize the token cache
        this.tokenCache = new TokenCache(this.config, this.browserStorage, this.logger, this.browserCrypto);
    }
    // #region Redirect Flow
    /**
     * Event handler function which allows users to fire events after the PublicClientApplication object
     * has loaded during redirect flows. This should be invoked on all page loads involved in redirect
     * auth flows.
     * @param hash Hash to process. Defaults to the current value of window.location.hash. Only needs to be provided explicitly if the response to be handled is not contained in the current value.
     * @returns Token response or null. If the return value is null, then no auth redirect was detected.
     */
    ClientApplication.prototype.handleRedirectPromise = function (hash) {
        return __awaiter(this, void 0, void 0, function () {
            var loggedInAccounts, redirectResponseKey, response, correlationId, redirectClient;
            var _this = this;
            return __generator(this, function (_a) {
                this.logger.verbose("handleRedirectPromise called");
                loggedInAccounts = this.getAllAccounts();
                if (this.isBrowserEnvironment) {
                    redirectResponseKey = hash || Constants.EMPTY_STRING;
                    response = this.redirectResponse.get(redirectResponseKey);
                    if (typeof response === "undefined") {
                        this.eventHandler.emitEvent(EventType.HANDLE_REDIRECT_START, InteractionType.Redirect);
                        this.logger.verbose("handleRedirectPromise has been called for the first time, storing the promise");
                        correlationId = this.browserStorage.getTemporaryCache(TemporaryCacheKeys.CORRELATION_ID, true) || "";
                        redirectClient = new RedirectClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, correlationId);
                        response = redirectClient.handleRedirectPromise(hash)
                            .then(function (result) {
                            if (result) {
                                // Emit login event if number of accounts change
                                var isLoggingIn = loggedInAccounts.length < _this.getAllAccounts().length;
                                if (isLoggingIn) {
                                    _this.eventHandler.emitEvent(EventType.LOGIN_SUCCESS, InteractionType.Redirect, result);
                                    _this.logger.verbose("handleRedirectResponse returned result, login success");
                                }
                                else {
                                    _this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_SUCCESS, InteractionType.Redirect, result);
                                    _this.logger.verbose("handleRedirectResponse returned result, acquire token success");
                                }
                            }
                            _this.eventHandler.emitEvent(EventType.HANDLE_REDIRECT_END, InteractionType.Redirect);
                            return result;
                        })
                            .catch(function (e) {
                            // Emit login event if there is an account
                            if (loggedInAccounts.length > 0) {
                                _this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_FAILURE, InteractionType.Redirect, null, e);
                            }
                            else {
                                _this.eventHandler.emitEvent(EventType.LOGIN_FAILURE, InteractionType.Redirect, null, e);
                            }
                            _this.eventHandler.emitEvent(EventType.HANDLE_REDIRECT_END, InteractionType.Redirect);
                            throw e;
                        });
                        this.redirectResponse.set(redirectResponseKey, response);
                    }
                    else {
                        this.logger.verbose("handleRedirectPromise has been called previously, returning the result from the first call");
                    }
                    return [2 /*return*/, response];
                }
                this.logger.verbose("handleRedirectPromise returns null, not browser environment");
                return [2 /*return*/, null];
            });
        });
    };
    /**
     * Use when you want to obtain an access_token for your API by redirecting the user's browser window to the authorization endpoint. This function redirects
     * the page, so any code that follows this function will not execute.
     *
     * IMPORTANT: It is NOT recommended to have code that is dependent on the resolution of the Promise. This function will navigate away from the current
     * browser window. It currently returns a Promise in order to reflect the asynchronous nature of the code running in this function.
     *
     * @param request
     */
    ClientApplication.prototype.acquireTokenRedirect = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var isLoggedIn, redirectClient;
            var _this = this;
            return __generator(this, function (_a) {
                // Preflight request
                this.logger.verbose("acquireTokenRedirect called");
                this.preflightBrowserEnvironmentCheck(InteractionType.Redirect);
                isLoggedIn = this.getAllAccounts().length > 0;
                if (isLoggedIn) {
                    this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_START, InteractionType.Redirect, request);
                }
                else {
                    this.eventHandler.emitEvent(EventType.LOGIN_START, InteractionType.Redirect, request);
                }
                redirectClient = new RedirectClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, request.correlationId);
                return [2 /*return*/, redirectClient.acquireToken(request).catch(function (e) {
                        // If logged in, emit acquire token events
                        if (isLoggedIn) {
                            _this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_FAILURE, InteractionType.Redirect, null, e);
                        }
                        else {
                            _this.eventHandler.emitEvent(EventType.LOGIN_FAILURE, InteractionType.Redirect, null, e);
                        }
                        throw e;
                    })];
            });
        });
    };
    // #endregion
    // #region Popup Flow
    /**
     * Use when you want to obtain an access_token for your API via opening a popup window in the user's browser
     *
     * @param request
     *
     * @returns A promise that is fulfilled when this function has completed, or rejected if an error was raised.
     */
    ClientApplication.prototype.acquireTokenPopup = function (request) {
        var _this = this;
        try {
            this.logger.verbose("acquireTokenPopup called", request.correlationId);
            this.preflightBrowserEnvironmentCheck(InteractionType.Popup);
        }
        catch (e) {
            // Since this function is syncronous we need to reject
            return Promise.reject(e);
        }
        // If logged in, emit acquire token events
        var loggedInAccounts = this.getAllAccounts();
        if (loggedInAccounts.length > 0) {
            this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_START, InteractionType.Popup, request);
        }
        else {
            this.eventHandler.emitEvent(EventType.LOGIN_START, InteractionType.Popup, request);
        }
        var popupClient = new PopupClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, request.correlationId);
        return popupClient.acquireToken(request).then(function (result) {
            // If logged in, emit acquire token events
            var isLoggingIn = loggedInAccounts.length < _this.getAllAccounts().length;
            if (isLoggingIn) {
                _this.eventHandler.emitEvent(EventType.LOGIN_SUCCESS, InteractionType.Popup, result);
            }
            else {
                _this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_SUCCESS, InteractionType.Popup, result);
            }
            return result;
        }).catch(function (e) {
            if (loggedInAccounts.length > 0) {
                _this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_FAILURE, InteractionType.Popup, null, e);
            }
            else {
                _this.eventHandler.emitEvent(EventType.LOGIN_FAILURE, InteractionType.Popup, null, e);
            }
            // Since this function is syncronous we need to reject
            return Promise.reject(e);
        });
    };
    // #endregion
    // #region Silent Flow
    /**
     * This function uses a hidden iframe to fetch an authorization code from the eSTS. There are cases where this may not work:
     * - Any browser using a form of Intelligent Tracking Prevention
     * - If there is not an established session with the service
     *
     * In these cases, the request must be done inside a popup or full frame redirect.
     *
     * For the cases where interaction is required, you cannot send a request with prompt=none.
     *
     * If your refresh token has expired, you can use this function to fetch a new set of tokens silently as long as
     * you session on the server still exists.
     * @param request {@link SsoSilentRequest}
     *
     * @returns A promise that is fulfilled when this function has completed, or rejected if an error was raised.
     */
    ClientApplication.prototype.ssoSilent = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var silentIframeClient, silentTokenResult, e_1;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.preflightBrowserEnvironmentCheck(InteractionType.Silent);
                        this.logger.verbose("ssoSilent called", request.correlationId);
                        this.eventHandler.emitEvent(EventType.SSO_SILENT_START, InteractionType.Silent, request);
                        _a.label = 1;
                    case 1:
                        _a.trys.push([1, 3, , 4]);
                        silentIframeClient = new SilentIframeClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, ApiId.ssoSilent, request.correlationId);
                        return [4 /*yield*/, silentIframeClient.acquireToken(request)];
                    case 2:
                        silentTokenResult = _a.sent();
                        this.eventHandler.emitEvent(EventType.SSO_SILENT_SUCCESS, InteractionType.Silent, silentTokenResult);
                        return [2 /*return*/, silentTokenResult];
                    case 3:
                        e_1 = _a.sent();
                        this.eventHandler.emitEvent(EventType.SSO_SILENT_FAILURE, InteractionType.Silent, null, e_1);
                        throw e_1;
                    case 4: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * This function redeems an authorization code (passed as code) from the eSTS token endpoint.
     * This authorization code should be acquired server-side using a confidential client to acquire a spa_code.
     * This API is not indended for normal authorization code acquisition and redemption.
     *
     * Redemption of this authorization code will not require PKCE, as it was acquired by a confidential client.
     *
     * @param request {@link AuthorizationCodeRequest}
     * @returns A promise that is fulfilled when this function has completed, or rejected if an error was raised.
     */
    ClientApplication.prototype.acquireTokenByCode = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var response;
            var _this = this;
            return __generator(this, function (_a) {
                this.preflightBrowserEnvironmentCheck(InteractionType.Silent);
                this.logger.trace("acquireTokenByCode called", request.correlationId);
                this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_BY_CODE_START, InteractionType.Silent, request);
                try {
                    if (!request.code) {
                        throw BrowserAuthError.createAuthCodeRequiredError();
                    }
                    response = this.hybridAuthCodeResponses.get(request.code);
                    if (!response) {
                        this.logger.verbose("Initiating new acquireTokenByCode request", request.correlationId);
                        response = this.acquireTokenByCodeAsync(request)
                            .then(function (result) {
                            _this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_BY_CODE_SUCCESS, InteractionType.Silent, result);
                            _this.hybridAuthCodeResponses.delete(request.code);
                            return result;
                        })
                            .catch(function (error) {
                            _this.hybridAuthCodeResponses.delete(request.code);
                            throw error;
                        });
                        this.hybridAuthCodeResponses.set(request.code, response);
                    }
                    else {
                        this.logger.verbose("Existing acquireTokenByCode request found", request.correlationId);
                    }
                    return [2 /*return*/, response];
                }
                catch (e) {
                    this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_BY_CODE_FAILURE, InteractionType.Silent, null, e);
                    throw e;
                }
                return [2 /*return*/];
            });
        });
    };
    /**
     * Creates a SilentAuthCodeClient to redeem an authorization code.
     * @param request
     * @returns Result of the operation to redeem the authorization code
     */
    ClientApplication.prototype.acquireTokenByCodeAsync = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var silentAuthCodeClient, silentTokenResult;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.logger.trace("acquireTokenByCodeAsync called", request.correlationId);
                        silentAuthCodeClient = new SilentAuthCodeClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, ApiId.acquireTokenByCode, request.correlationId);
                        return [4 /*yield*/, silentAuthCodeClient.acquireToken(request)];
                    case 1:
                        silentTokenResult = _a.sent();
                        return [2 /*return*/, silentTokenResult];
                }
            });
        });
    };
    /**
     * Use this function to obtain a token before every call to the API / resource provider
     *
     * MSAL return's a cached token when available
     * Or it send's a request to the STS to obtain a new token using a refresh token.
     *
     * @param {@link SilentRequest}
     *
     * To renew idToken, please pass clientId as the only scope in the Authentication Parameters
     * @returns A promise that is fulfilled when this function has completed, or rejected if an error was raised.
     */
    ClientApplication.prototype.acquireTokenByRefreshToken = function (request) {
        return __awaiter(this, void 0, void 0, function () {
            var silentRefreshClient;
            var _this = this;
            return __generator(this, function (_a) {
                this.eventHandler.emitEvent(EventType.ACQUIRE_TOKEN_NETWORK_START, InteractionType.Silent, request);
                // block the reload if it occurred inside a hidden iframe
                BrowserUtils.blockReloadInHiddenIframes();
                silentRefreshClient = new SilentRefreshClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, request.correlationId);
                return [2 /*return*/, silentRefreshClient.acquireToken(request).catch(function (e) {
                        var isServerError = e instanceof ServerError;
                        var isInteractionRequiredError = e instanceof InteractionRequiredAuthError;
                        var isInvalidGrantError = (e.errorCode === BrowserConstants.INVALID_GRANT_ERROR);
                        if (isServerError && isInvalidGrantError && !isInteractionRequiredError) {
                            _this.logger.verbose("Refresh token expired or invalid, attempting acquire token by iframe", request.correlationId);
                            var silentIframeClient = new SilentIframeClient(_this.config, _this.browserStorage, _this.browserCrypto, _this.logger, _this.eventHandler, _this.navigationClient, ApiId.acquireTokenSilent_authCode, request.correlationId);
                            return silentIframeClient.acquireToken(request);
                        }
                        throw e;
                    })];
            });
        });
    };
    // #endregion
    // #region Logout
    /**
     * Deprecated logout function. Use logoutRedirect or logoutPopup instead
     * @param logoutRequest
     * @deprecated
     */
    ClientApplication.prototype.logout = function (logoutRequest) {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                this.logger.warning("logout API is deprecated and will be removed in msal-browser v3.0.0. Use logoutRedirect instead.");
                return [2 /*return*/, this.logoutRedirect(logoutRequest)];
            });
        });
    };
    /**
     * Use to log out the current user, and redirect the user to the postLogoutRedirectUri.
     * Default behaviour is to redirect the user to `window.location.href`.
     * @param logoutRequest
     */
    ClientApplication.prototype.logoutRedirect = function (logoutRequest) {
        return __awaiter(this, void 0, void 0, function () {
            var redirectClient;
            return __generator(this, function (_a) {
                this.preflightBrowserEnvironmentCheck(InteractionType.Redirect);
                redirectClient = new RedirectClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, logoutRequest === null || logoutRequest === void 0 ? void 0 : logoutRequest.correlationId);
                return [2 /*return*/, redirectClient.logout(logoutRequest)];
            });
        });
    };
    /**
     * Clears local cache for the current user then opens a popup window prompting the user to sign-out of the server
     * @param logoutRequest
     */
    ClientApplication.prototype.logoutPopup = function (logoutRequest) {
        try {
            this.preflightBrowserEnvironmentCheck(InteractionType.Popup);
            var popupClient = new PopupClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, logoutRequest === null || logoutRequest === void 0 ? void 0 : logoutRequest.correlationId);
            return popupClient.logout(logoutRequest);
        }
        catch (e) {
            // Since this function is syncronous we need to reject
            return Promise.reject(e);
        }
    };
    // #endregion
    // #region Account APIs
    /**
     * Returns all accounts that MSAL currently has data for.
     * (the account object is created at the time of successful login)
     * or empty array when no accounts are found
     * @returns Array of account objects in cache
     */
    ClientApplication.prototype.getAllAccounts = function () {
        this.logger.verbose("getAllAccounts called");
        return this.isBrowserEnvironment ? this.browserStorage.getAllAccounts() : [];
    };
    /**
     * Returns the signed in account matching username.
     * (the account object is created at the time of successful login)
     * or null when no matching account is found.
     * This API is provided for convenience but getAccountById should be used for best reliability
     * @param userName
     * @returns The account object stored in MSAL
     */
    ClientApplication.prototype.getAccountByUsername = function (userName) {
        var allAccounts = this.getAllAccounts();
        if (!StringUtils.isEmpty(userName) && allAccounts && allAccounts.length) {
            this.logger.verbose("Account matching username found, returning");
            this.logger.verbosePii("Returning signed-in accounts matching username: " + userName);
            return allAccounts.filter(function (accountObj) { return accountObj.username.toLowerCase() === userName.toLowerCase(); })[0] || null;
        }
        else {
            this.logger.verbose("getAccountByUsername: No matching account found, returning null");
            return null;
        }
    };
    /**
     * Returns the signed in account matching homeAccountId.
     * (the account object is created at the time of successful login)
     * or null when no matching account is found
     * @param homeAccountId
     * @returns The account object stored in MSAL
     */
    ClientApplication.prototype.getAccountByHomeId = function (homeAccountId) {
        var allAccounts = this.getAllAccounts();
        if (!StringUtils.isEmpty(homeAccountId) && allAccounts && allAccounts.length) {
            this.logger.verbose("Account matching homeAccountId found, returning");
            this.logger.verbosePii("Returning signed-in accounts matching homeAccountId: " + homeAccountId);
            return allAccounts.filter(function (accountObj) { return accountObj.homeAccountId === homeAccountId; })[0] || null;
        }
        else {
            this.logger.verbose("getAccountByHomeId: No matching account found, returning null");
            return null;
        }
    };
    /**
     * Returns the signed in account matching localAccountId.
     * (the account object is created at the time of successful login)
     * or null when no matching account is found
     * @param localAccountId
     * @returns The account object stored in MSAL
     */
    ClientApplication.prototype.getAccountByLocalId = function (localAccountId) {
        var allAccounts = this.getAllAccounts();
        if (!StringUtils.isEmpty(localAccountId) && allAccounts && allAccounts.length) {
            this.logger.verbose("Account matching localAccountId found, returning");
            this.logger.verbosePii("Returning signed-in accounts matching localAccountId: " + localAccountId);
            return allAccounts.filter(function (accountObj) { return accountObj.localAccountId === localAccountId; })[0] || null;
        }
        else {
            this.logger.verbose("getAccountByLocalId: No matching account found, returning null");
            return null;
        }
    };
    /**
     * Sets the account to use as the active account. If no account is passed to the acquireToken APIs, then MSAL will use this active account.
     * @param account
     */
    ClientApplication.prototype.setActiveAccount = function (account) {
        this.browserStorage.setActiveAccount(account);
    };
    /**
     * Gets the currently active account
     */
    ClientApplication.prototype.getActiveAccount = function () {
        return this.browserStorage.getActiveAccount();
    };
    // #endregion
    // #region Helpers
    /**
     * Helper to validate app environment before making an auth request
     * * @param interactionType
     */
    ClientApplication.prototype.preflightBrowserEnvironmentCheck = function (interactionType) {
        this.logger.verbose("preflightBrowserEnvironmentCheck started");
        // Block request if not in browser environment
        BrowserUtils.blockNonBrowserEnvironment(this.isBrowserEnvironment);
        // Block redirects if in an iframe
        BrowserUtils.blockRedirectInIframe(interactionType, this.config.system.allowRedirectInIframe);
        // Block auth requests inside a hidden iframe
        BrowserUtils.blockReloadInHiddenIframes();
        // Block redirectUri opened in a popup from calling MSAL APIs
        BrowserUtils.blockAcquireTokenInPopups();
        // Block redirects if memory storage is enabled but storeAuthStateInCookie is not
        if (interactionType === InteractionType.Redirect &&
            this.config.cache.cacheLocation === BrowserCacheLocation.MemoryStorage &&
            !this.config.cache.storeAuthStateInCookie) {
            throw BrowserConfigurationAuthError.createInMemoryRedirectUnavailableError();
        }
        if (interactionType === InteractionType.Redirect || interactionType === InteractionType.Popup) {
            this.preflightInteractiveRequest();
        }
    };
    /**
     * Helper to validate app environment before making a request.
     * @param request
     * @param interactionType
     */
    ClientApplication.prototype.preflightInteractiveRequest = function () {
        this.logger.verbose("preflightInteractiveRequest called, validating app environment");
        // block the reload if it occurred inside a hidden iframe
        BrowserUtils.blockReloadInHiddenIframes();
        // Set interaction in progress temporary cache or throw if alread set.
        this.browserStorage.setInteractionInProgress(true);
    };
    /**
     * Adds event callbacks to array
     * @param callback
     */
    ClientApplication.prototype.addEventCallback = function (callback) {
        return this.eventHandler.addEventCallback(callback);
    };
    /**
     * Removes callback with provided id from callback array
     * @param callbackId
     */
    ClientApplication.prototype.removeEventCallback = function (callbackId) {
        this.eventHandler.removeEventCallback(callbackId);
    };
    /**
     * Adds event listener that emits an event when a user account is added or removed from localstorage in a different browser tab or window
     */
    ClientApplication.prototype.enableAccountStorageEvents = function () {
        this.eventHandler.enableAccountStorageEvents();
    };
    /**
     * Removes event listener that emits an event when a user account is added or removed from localstorage in a different browser tab or window
     */
    ClientApplication.prototype.disableAccountStorageEvents = function () {
        this.eventHandler.disableAccountStorageEvents();
    };
    /**
     * Gets the token cache for the application.
     */
    ClientApplication.prototype.getTokenCache = function () {
        return this.tokenCache;
    };
    /**
     * Returns the logger instance
     */
    ClientApplication.prototype.getLogger = function () {
        return this.logger;
    };
    /**
     * Replaces the default logger set in configurations with new Logger with new configurations
     * @param logger Logger instance
     */
    ClientApplication.prototype.setLogger = function (logger) {
        this.logger = logger;
    };
    /**
     * Called by wrapper libraries (Angular & React) to set SKU and Version passed down to telemetry, logger, etc.
     * @param sku
     * @param version
     */
    ClientApplication.prototype.initializeWrapperLibrary = function (sku, version) {
        // Validate the SKU passed in is one we expect
        this.browserStorage.setWrapperMetadata(sku, version);
    };
    /**
     * Sets navigation client
     * @param navigationClient
     */
    ClientApplication.prototype.setNavigationClient = function (navigationClient) {
        this.navigationClient = navigationClient;
    };
    /**
     * Returns the configuration object
     */
    ClientApplication.prototype.getConfiguration = function () {
        return this.config;
    };
    return ClientApplication;
}());

export { ClientApplication };
//# sourceMappingURL=ClientApplication.js.map
