// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { __rest } from "tslib";
import { createPipelineFromOptions, isTokenCredential, signingPolicy } from "@azure/core-http";
import { SDK_VERSION } from "../constants";
import { KeyVaultClient } from "../generated";
import { parseKeyVaultKeyIdentifier } from "../identifier";
import { LATEST_API_VERSION } from "../keysModels";
import { getKeyFromKeyBundle } from "../transformations";
import { createHash } from "./crypto";
import { logger } from "../log";
import { createTraceFunction, challengeBasedAuthenticationPolicy } from "../../../keyvault-common/src";
const withTrace = createTraceFunction("Azure.KeyVault.Keys.RemoteCryptographyProvider");
/**
 * The remote cryptography provider is used to run crypto operations against KeyVault.
 * @internal
 */
export class RemoteCryptographyProvider {
    constructor(key, credential, pipelineOptions = {}) {
        const libInfo = `azsdk-js-keyvault-keys/${SDK_VERSION}`;
        const userAgentOptions = pipelineOptions.userAgentOptions;
        pipelineOptions.userAgentOptions = {
            userAgentPrefix: userAgentOptions && userAgentOptions.userAgentPrefix
                ? `${userAgentOptions.userAgentPrefix} ${libInfo}`
                : libInfo
        };
        const authPolicy = isTokenCredential(credential)
            ? challengeBasedAuthenticationPolicy(credential)
            : signingPolicy(credential);
        const internalPipelineOptions = Object.assign(Object.assign({}, pipelineOptions), { loggingOptions: {
                logger: logger.info,
                allowedHeaderNames: [
                    "x-ms-keyvault-region",
                    "x-ms-keyvault-network-info",
                    "x-ms-keyvault-service-version"
                ]
            } });
        this.client = new KeyVaultClient(pipelineOptions.serviceVersion || LATEST_API_VERSION, createPipelineFromOptions(internalPipelineOptions, authPolicy));
        this.key = key;
        let keyId;
        if (typeof key === "string") {
            keyId = key;
        }
        else {
            keyId = key.id;
        }
        try {
            const parsed = parseKeyVaultKeyIdentifier(keyId);
            if (parsed.name === "") {
                throw new Error("Could not find 'name' of key in key URL");
            }
            if (!parsed.version || parsed.version === "") {
                throw new Error("Could not find 'version' of key in key URL");
            }
            if (!parsed.vaultUrl || parsed.vaultUrl === "") {
                throw new Error("Could not find 'vaultUrl' of key in key URL");
            }
            this.vaultUrl = parsed.vaultUrl;
            this.name = parsed.name;
            this.version = parsed.version;
        }
        catch (err) {
            logger.error(err);
            throw new Error(`${keyId} is not a valid Key Vault key ID`);
        }
    }
    // The remote client supports all algorithms and all operations.
    isSupported(_algorithm, _operation) {
        return true;
    }
    encrypt(encryptParameters, options = {}) {
        const { algorithm, plaintext } = encryptParameters, params = __rest(encryptParameters, ["algorithm", "plaintext"]);
        const requestOptions = Object.assign(Object.assign({}, options), params);
        return withTrace("encrypt", requestOptions, async (updatedOptions) => {
            const result = await this.client.encrypt(this.vaultUrl, this.name, this.version, algorithm, plaintext, updatedOptions);
            return {
                algorithm: encryptParameters.algorithm,
                result: result.result,
                keyID: this.getKeyID(),
                additionalAuthenticatedData: result.additionalAuthenticatedData,
                authenticationTag: result.authenticationTag,
                iv: result.iv
            };
        });
    }
    decrypt(decryptParameters, options = {}) {
        const { algorithm, ciphertext } = decryptParameters, params = __rest(decryptParameters, ["algorithm", "ciphertext"]);
        const requestOptions = Object.assign(Object.assign({}, options), params);
        return withTrace("decrypt", requestOptions, async (updatedOptions) => {
            const result = await this.client.decrypt(this.vaultUrl, this.name, this.version, algorithm, ciphertext, updatedOptions);
            return {
                result: result.result,
                keyID: this.getKeyID(),
                algorithm
            };
        });
    }
    wrapKey(algorithm, keyToWrap, options = {}) {
        return withTrace("wrapKey", options, async (updatedOptions) => {
            const result = await this.client.wrapKey(this.vaultUrl, this.name, this.version, algorithm, keyToWrap, updatedOptions);
            return {
                result: result.result,
                algorithm,
                keyID: this.getKeyID()
            };
        });
    }
    unwrapKey(algorithm, encryptedKey, options = {}) {
        return withTrace("unwrapKey", options, async (updatedOptions) => {
            const result = await this.client.unwrapKey(this.vaultUrl, this.name, this.version, algorithm, encryptedKey, updatedOptions);
            return {
                result: result.result,
                algorithm,
                keyID: this.getKeyID()
            };
        });
    }
    sign(algorithm, digest, options = {}) {
        return withTrace("sign", options, async (updatedOptions) => {
            const result = await this.client.sign(this.vaultUrl, this.name, this.version, algorithm, digest, updatedOptions);
            return { result: result.result, algorithm, keyID: this.getKeyID() };
        });
    }
    verifyData(algorithm, data, signature, options = {}) {
        return withTrace("verifyData", options, async (updatedOptions) => {
            const hash = await createHash(algorithm, data);
            return this.verify(algorithm, hash, signature, updatedOptions);
        });
    }
    verify(algorithm, digest, signature, options = {}) {
        return withTrace("verify", options, async (updatedOptions) => {
            const response = await this.client.verify(this.vaultUrl, this.name, this.version, algorithm, digest, signature, updatedOptions);
            return {
                result: response.value ? response.value : false,
                keyID: this.getKeyID()
            };
        });
    }
    signData(algorithm, data, options = {}) {
        return withTrace("signData", options, async (updatedOptions) => {
            const digest = await createHash(algorithm, data);
            const result = await this.client.sign(this.vaultUrl, this.name, this.version, algorithm, digest, updatedOptions);
            return { result: result.result, algorithm, keyID: this.getKeyID() };
        });
    }
    /**
     * The ID of the key used to perform cryptographic operations for the client.
     */
    get keyId() {
        return this.getKeyID();
    }
    /**
     * Gets the {@link KeyVaultKey} used for cryptography operations, fetching it
     * from KeyVault if necessary.
     * @param options - Additional options.
     */
    getKey(options = {}) {
        return withTrace("getKey", options, async (updatedOptions) => {
            if (typeof this.key === "string") {
                if (!this.name || this.name === "") {
                    throw new Error("getKey requires a key with a name");
                }
                const response = await this.client.getKey(this.vaultUrl, this.name, options && options.version ? options.version : this.version ? this.version : "", updatedOptions);
                this.key = getKeyFromKeyBundle(response);
            }
            return this.key;
        });
    }
    /**
     * Attempts to retrieve the ID of the key.
     * @internal
     */
    getKeyID() {
        let kid;
        if (typeof this.key !== "string") {
            kid = this.key.id;
        }
        else {
            kid = this.key;
        }
        return kid;
    }
}
//# sourceMappingURL=remoteCryptographyProvider.js.map