// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { createSpanFunction, SpanStatusCode } from "@azure/core-tracing";
/**
 * Returns a function that can be used for tracing options.
 *
 * @param prefix - The prefix to use, likely the name of the class / client.
 *
 * @example const withTrace = createTraceFunction("Azure.KeyVault.Certificates.CertificateClient")
 *
 * @internal
 */
export function createTraceFunction(prefix) {
    const createSpan = createSpanFunction({
        namespace: "Microsoft.KeyVault",
        packagePrefix: prefix
    });
    return async function (operationName, options, cb) {
        const { updatedOptions, span } = createSpan(operationName, options);
        try {
            // NOTE: we really do need to await on this function here so we can handle any exceptions thrown and properly
            // close the span.
            const result = await cb(updatedOptions, span);
            // otel 0.16+ needs this or else the code ends up being set as UNSET
            span.setStatus({
                code: SpanStatusCode.OK
            });
            return result;
        }
        catch (err) {
            span.setStatus({
                code: SpanStatusCode.ERROR,
                message: err.message
            });
            throw err;
        }
        finally {
            span.end();
        }
    };
}
//# sourceMappingURL=tracingHelpers.js.map