"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _metadataParser = require("./metadata-parser");

var _dataType = require("./data-type");

var _iconvLite = _interopRequireDefault(require("iconv-lite"));

var _sprintfJs = require("sprintf-js");

var _guidParser = require("./guid-parser");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const NULL = (1 << 16) - 1;
const MAX = (1 << 16) - 1;
const THREE_AND_A_THIRD = 3 + 1 / 3;
const MONEY_DIVISOR = 10000;
const PLP_NULL = Buffer.from([0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF]);
const UNKNOWN_PLP_LEN = Buffer.from([0xFE, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF]);
const DEFAULT_ENCODING = 'utf8';

function readTinyInt(parser, callback) {
  parser.readUInt8(callback);
}

function readSmallInt(parser, callback) {
  parser.readInt16LE(callback);
}

function readInt(parser, callback) {
  parser.readInt32LE(callback);
}

function readBigInt(parser, callback) {
  parser.readBigInt64LE(value => {
    callback(value.toString());
  });
}

function readReal(parser, callback) {
  parser.readFloatLE(callback);
}

function readFloat(parser, callback) {
  parser.readDoubleLE(callback);
}

function readSmallMoney(parser, callback) {
  parser.readInt32LE(value => {
    callback(value / MONEY_DIVISOR);
  });
}

function readMoney(parser, callback) {
  parser.readInt32LE(high => {
    parser.readUInt32LE(low => {
      callback((low + 0x100000000 * high) / MONEY_DIVISOR);
    });
  });
}

function readBit(parser, callback) {
  parser.readUInt8(value => {
    callback(!!value);
  });
}

function valueParse(parser, metadata, options, callback) {
  const type = metadata.type;

  switch (type.name) {
    case 'Null':
      return callback(null);

    case 'TinyInt':
      return readTinyInt(parser, callback);

    case 'SmallInt':
      return readSmallInt(parser, callback);

    case 'Int':
      return readInt(parser, callback);

    case 'BigInt':
      return readBigInt(parser, callback);

    case 'IntN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 1:
            return readTinyInt(parser, callback);

          case 2:
            return readSmallInt(parser, callback);

          case 4:
            return readInt(parser, callback);

          case 8:
            return readBigInt(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for IntN');
        }
      });

    case 'Real':
      return readReal(parser, callback);

    case 'Float':
      return readFloat(parser, callback);

    case 'FloatN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 4:
            return readReal(parser, callback);

          case 8:
            return readFloat(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for FloatN');
        }
      });

    case 'SmallMoney':
      return readSmallMoney(parser, callback);

    case 'Money':
      return readMoney(parser, callback);

    case 'MoneyN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 4:
            return readSmallMoney(parser, callback);

          case 8:
            return readMoney(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for MoneyN');
        }
      });

    case 'Bit':
      return readBit(parser, callback);

    case 'BitN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 1:
            return readBit(parser, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for BitN');
        }
      });

    case 'VarChar':
    case 'Char':
      const codepage = metadata.collation.codepage;

      if (metadata.dataLength === MAX) {
        return readMaxChars(parser, codepage, callback);
      } else {
        return parser.readUInt16LE(dataLength => {
          if (dataLength === NULL) {
            return callback(null);
          }

          readChars(parser, dataLength, codepage, callback);
        });
      }

    case 'NVarChar':
    case 'NChar':
      if (metadata.dataLength === MAX) {
        return readMaxNChars(parser, callback);
      } else {
        return parser.readUInt16LE(dataLength => {
          if (dataLength === NULL) {
            return callback(null);
          }

          readNChars(parser, dataLength, callback);
        });
      }

    case 'VarBinary':
    case 'Binary':
      if (metadata.dataLength === MAX) {
        return readMaxBinary(parser, callback);
      } else {
        return parser.readUInt16LE(dataLength => {
          if (dataLength === NULL) {
            return callback(null);
          }

          readBinary(parser, dataLength, callback);
        });
      }

    case 'Text':
      return parser.readUInt8(textPointerLength => {
        if (textPointerLength === 0) {
          return callback(null);
        }

        parser.readBuffer(textPointerLength, _textPointer => {
          parser.readBuffer(8, _timestamp => {
            parser.readUInt32LE(dataLength => {
              readChars(parser, dataLength, metadata.collation.codepage, callback);
            });
          });
        });
      });

    case 'NText':
      return parser.readUInt8(textPointerLength => {
        if (textPointerLength === 0) {
          return callback(null);
        }

        parser.readBuffer(textPointerLength, _textPointer => {
          parser.readBuffer(8, _timestamp => {
            parser.readUInt32LE(dataLength => {
              readNChars(parser, dataLength, callback);
            });
          });
        });
      });

    case 'Image':
      return parser.readUInt8(textPointerLength => {
        if (textPointerLength === 0) {
          return callback(null);
        }

        parser.readBuffer(textPointerLength, _textPointer => {
          parser.readBuffer(8, _timestamp => {
            parser.readUInt32LE(dataLength => {
              readBinary(parser, dataLength, callback);
            });
          });
        });
      });

    case 'Xml':
      return readMaxNChars(parser, callback);

    case 'SmallDateTime':
      return readSmallDateTime(parser, options.useUTC, callback);

    case 'DateTime':
      return readDateTime(parser, options.useUTC, callback);

    case 'DateTimeN':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 4:
            return readSmallDateTime(parser, options.useUTC, callback);

          case 8:
            return readDateTime(parser, options.useUTC, callback);

          default:
            throw new Error('Unsupported dataLength ' + dataLength + ' for DateTimeN');
        }
      });

    case 'Time':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readTime(parser, dataLength, metadata.scale, options.useUTC, callback);
        }
      });

    case 'Date':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readDate(parser, options.useUTC, callback);
        }
      });

    case 'DateTime2':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readDateTime2(parser, dataLength, metadata.scale, options.useUTC, callback);
        }
      });

    case 'DateTimeOffset':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readDateTimeOffset(parser, dataLength, metadata.scale, callback);
        }
      });

    case 'NumericN':
    case 'DecimalN':
      return parser.readUInt8(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        } else {
          return readNumeric(parser, dataLength, metadata.precision, metadata.scale, callback);
        }
      });

    case 'UniqueIdentifier':
      return parser.readUInt8(dataLength => {
        switch (dataLength) {
          case 0:
            return callback(null);

          case 0x10:
            return readUniqueIdentifier(parser, options, callback);

          default:
            throw new Error((0, _sprintfJs.sprintf)('Unsupported guid size %d', dataLength - 1));
        }
      });

    case 'UDT':
      return readMaxBinary(parser, callback);

    case 'Variant':
      return parser.readUInt32LE(dataLength => {
        if (dataLength === 0) {
          return callback(null);
        }

        readVariant(parser, options, dataLength, callback);
      });

    default:
      throw new Error((0, _sprintfJs.sprintf)('Unrecognised type %s', type.name));
  }
}

function readUniqueIdentifier(parser, options, callback) {
  parser.readBuffer(0x10, data => {
    callback(options.lowerCaseGuids ? (0, _guidParser.bufferToLowerCaseGuid)(data) : (0, _guidParser.bufferToUpperCaseGuid)(data));
  });
}

function readNumeric(parser, dataLength, _precision, scale, callback) {
  parser.readUInt8(sign => {
    sign = sign === 1 ? 1 : -1;
    let readValue;

    if (dataLength === 5) {
      readValue = parser.readUInt32LE;
    } else if (dataLength === 9) {
      readValue = parser.readUNumeric64LE;
    } else if (dataLength === 13) {
      readValue = parser.readUNumeric96LE;
    } else if (dataLength === 17) {
      readValue = parser.readUNumeric128LE;
    } else {
      throw new Error((0, _sprintfJs.sprintf)('Unsupported numeric dataLength %d', dataLength));
    }

    readValue.call(parser, value => {
      callback(value * sign / Math.pow(10, scale));
    });
  });
}

function readVariant(parser, options, dataLength, callback) {
  return parser.readUInt8(baseType => {
    const type = _dataType.TYPE[baseType];
    return parser.readUInt8(propBytes => {
      dataLength = dataLength - propBytes - 2;

      switch (type.name) {
        case 'UniqueIdentifier':
          return readUniqueIdentifier(parser, options, callback);

        case 'Bit':
          return readBit(parser, callback);

        case 'TinyInt':
          return readTinyInt(parser, callback);

        case 'SmallInt':
          return readSmallInt(parser, callback);

        case 'Int':
          return readInt(parser, callback);

        case 'BigInt':
          return readBigInt(parser, callback);

        case 'SmallDateTime':
          return readSmallDateTime(parser, options.useUTC, callback);

        case 'DateTime':
          return readDateTime(parser, options.useUTC, callback);

        case 'Real':
          return readReal(parser, callback);

        case 'Float':
          return readFloat(parser, callback);

        case 'SmallMoney':
          return readSmallMoney(parser, callback);

        case 'Money':
          return readMoney(parser, callback);

        case 'Date':
          return readDate(parser, options.useUTC, callback);

        case 'Time':
          return parser.readUInt8(scale => {
            return readTime(parser, dataLength, scale, options.useUTC, callback);
          });

        case 'DateTime2':
          return parser.readUInt8(scale => {
            return readDateTime2(parser, dataLength, scale, options.useUTC, callback);
          });

        case 'DateTimeOffset':
          return parser.readUInt8(scale => {
            return readDateTimeOffset(parser, dataLength, scale, callback);
          });

        case 'VarBinary':
        case 'Binary':
          return parser.readUInt16LE(_maxLength => {
            readBinary(parser, dataLength, callback);
          });

        case 'NumericN':
        case 'DecimalN':
          return parser.readUInt8(precision => {
            parser.readUInt8(scale => {
              readNumeric(parser, dataLength, precision, scale, callback);
            });
          });

        case 'VarChar':
        case 'Char':
          return parser.readUInt16LE(_maxLength => {
            (0, _metadataParser.readCollation)(parser, collation => {
              readChars(parser, dataLength, collation.codepage, callback);
            });
          });

        case 'NVarChar':
        case 'NChar':
          return parser.readUInt16LE(_maxLength => {
            (0, _metadataParser.readCollation)(parser, _collation => {
              readNChars(parser, dataLength, callback);
            });
          });

        default:
          throw new Error('Invalid type!');
      }
    });
  });
}

function readBinary(parser, dataLength, callback) {
  return parser.readBuffer(dataLength, callback);
}

function readChars(parser, dataLength, codepage, callback) {
  if (codepage == null) {
    codepage = DEFAULT_ENCODING;
  }

  return parser.readBuffer(dataLength, data => {
    callback(_iconvLite.default.decode(data, codepage));
  });
}

function readNChars(parser, dataLength, callback) {
  parser.readBuffer(dataLength, data => {
    callback(data.toString('ucs2'));
  });
}

function readMaxBinary(parser, callback) {
  return readMax(parser, callback);
}

function readMaxChars(parser, codepage, callback) {
  if (codepage == null) {
    codepage = DEFAULT_ENCODING;
  }

  readMax(parser, data => {
    if (data) {
      callback(_iconvLite.default.decode(data, codepage));
    } else {
      callback(null);
    }
  });
}

function readMaxNChars(parser, callback) {
  readMax(parser, data => {
    if (data) {
      callback(data.toString('ucs2'));
    } else {
      callback(null);
    }
  });
}

function readMax(parser, callback) {
  parser.readBuffer(8, type => {
    if (type.equals(PLP_NULL)) {
      return callback(null);
    } else if (type.equals(UNKNOWN_PLP_LEN)) {
      return readMaxUnknownLength(parser, callback);
    } else {
      const low = type.readUInt32LE(0);
      const high = type.readUInt32LE(4);

      if (high >= 2 << 53 - 32) {
        console.warn('Read UInt64LE > 53 bits : high=' + high + ', low=' + low);
      }

      const expectedLength = low + 0x100000000 * high;
      return readMaxKnownLength(parser, expectedLength, callback);
    }
  });
}

function readMaxKnownLength(parser, totalLength, callback) {
  const data = Buffer.alloc(totalLength, 0);
  let offset = 0;

  function next(done) {
    parser.readUInt32LE(chunkLength => {
      if (!chunkLength) {
        return done();
      }

      parser.readBuffer(chunkLength, chunk => {
        chunk.copy(data, offset);
        offset += chunkLength;
        next(done);
      });
    });
  }

  next(() => {
    if (offset !== totalLength) {
      throw new Error('Partially Length-prefixed Bytes unmatched lengths : expected ' + totalLength + ', but got ' + offset + ' bytes');
    }

    callback(data);
  });
}

function readMaxUnknownLength(parser, callback) {
  const chunks = [];
  let length = 0;

  function next(done) {
    parser.readUInt32LE(chunkLength => {
      if (!chunkLength) {
        return done();
      }

      parser.readBuffer(chunkLength, chunk => {
        chunks.push(chunk);
        length += chunkLength;
        next(done);
      });
    });
  }

  next(() => {
    callback(Buffer.concat(chunks, length));
  });
}

function readSmallDateTime(parser, useUTC, callback) {
  parser.readUInt16LE(days => {
    parser.readUInt16LE(minutes => {
      let value;

      if (useUTC) {
        value = new Date(Date.UTC(1900, 0, 1 + days, 0, minutes));
      } else {
        value = new Date(1900, 0, 1 + days, 0, minutes);
      }

      callback(value);
    });
  });
}

function readDateTime(parser, useUTC, callback) {
  parser.readInt32LE(days => {
    parser.readUInt32LE(threeHundredthsOfSecond => {
      const milliseconds = Math.round(threeHundredthsOfSecond * THREE_AND_A_THIRD);
      let value;

      if (useUTC) {
        value = new Date(Date.UTC(1900, 0, 1 + days, 0, 0, 0, milliseconds));
      } else {
        value = new Date(1900, 0, 1 + days, 0, 0, 0, milliseconds);
      }

      callback(value);
    });
  });
}

function readTime(parser, dataLength, scale, useUTC, callback) {
  let readValue;

  switch (dataLength) {
    case 3:
      readValue = parser.readUInt24LE;
      break;

    case 4:
      readValue = parser.readUInt32LE;
      break;

    case 5:
      readValue = parser.readUInt40LE;
  }

  readValue.call(parser, value => {
    if (scale < 7) {
      for (let i = scale; i < 7; i++) {
        value *= 10;
      }
    }

    let date;

    if (useUTC) {
      date = new Date(Date.UTC(1970, 0, 1, 0, 0, 0, value / 10000));
    } else {
      date = new Date(1970, 0, 1, 0, 0, 0, value / 10000);
    }

    Object.defineProperty(date, 'nanosecondsDelta', {
      enumerable: false,
      value: value % 10000 / Math.pow(10, 7)
    });
    callback(date);
  });
}

function readDate(parser, useUTC, callback) {
  parser.readUInt24LE(days => {
    if (useUTC) {
      callback(new Date(Date.UTC(2000, 0, days - 730118)));
    } else {
      callback(new Date(2000, 0, days - 730118));
    }
  });
}

function readDateTime2(parser, dataLength, scale, useUTC, callback) {
  readTime(parser, dataLength - 3, scale, useUTC, time => {
    // TODO: 'input' is 'time', but TypeScript cannot find "time.nanosecondsDelta";
    parser.readUInt24LE(days => {
      let date;

      if (useUTC) {
        date = new Date(Date.UTC(2000, 0, days - 730118, 0, 0, 0, +time));
      } else {
        date = new Date(2000, 0, days - 730118, time.getHours(), time.getMinutes(), time.getSeconds(), time.getMilliseconds());
      }

      Object.defineProperty(date, 'nanosecondsDelta', {
        enumerable: false,
        value: time.nanosecondsDelta
      });
      callback(date);
    });
  });
}

function readDateTimeOffset(parser, dataLength, scale, callback) {
  readTime(parser, dataLength - 5, scale, true, time => {
    parser.readUInt24LE(days => {
      // offset
      parser.readInt16LE(() => {
        const date = new Date(Date.UTC(2000, 0, days - 730118, 0, 0, 0, +time));
        Object.defineProperty(date, 'nanosecondsDelta', {
          enumerable: false,
          value: time.nanosecondsDelta
        });
        callback(date);
      });
    });
  });
}

var _default = valueParse;
exports.default = _default;
module.exports = valueParse;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy92YWx1ZS1wYXJzZXIudHMiXSwibmFtZXMiOlsiTlVMTCIsIk1BWCIsIlRIUkVFX0FORF9BX1RISVJEIiwiTU9ORVlfRElWSVNPUiIsIlBMUF9OVUxMIiwiQnVmZmVyIiwiZnJvbSIsIlVOS05PV05fUExQX0xFTiIsIkRFRkFVTFRfRU5DT0RJTkciLCJyZWFkVGlueUludCIsInBhcnNlciIsImNhbGxiYWNrIiwicmVhZFVJbnQ4IiwicmVhZFNtYWxsSW50IiwicmVhZEludDE2TEUiLCJyZWFkSW50IiwicmVhZEludDMyTEUiLCJyZWFkQmlnSW50IiwicmVhZEJpZ0ludDY0TEUiLCJ2YWx1ZSIsInRvU3RyaW5nIiwicmVhZFJlYWwiLCJyZWFkRmxvYXRMRSIsInJlYWRGbG9hdCIsInJlYWREb3VibGVMRSIsInJlYWRTbWFsbE1vbmV5IiwicmVhZE1vbmV5IiwiaGlnaCIsInJlYWRVSW50MzJMRSIsImxvdyIsInJlYWRCaXQiLCJ2YWx1ZVBhcnNlIiwibWV0YWRhdGEiLCJvcHRpb25zIiwidHlwZSIsIm5hbWUiLCJkYXRhTGVuZ3RoIiwiRXJyb3IiLCJjb2RlcGFnZSIsImNvbGxhdGlvbiIsInJlYWRNYXhDaGFycyIsInJlYWRVSW50MTZMRSIsInJlYWRDaGFycyIsInJlYWRNYXhOQ2hhcnMiLCJyZWFkTkNoYXJzIiwicmVhZE1heEJpbmFyeSIsInJlYWRCaW5hcnkiLCJ0ZXh0UG9pbnRlckxlbmd0aCIsInJlYWRCdWZmZXIiLCJfdGV4dFBvaW50ZXIiLCJfdGltZXN0YW1wIiwicmVhZFNtYWxsRGF0ZVRpbWUiLCJ1c2VVVEMiLCJyZWFkRGF0ZVRpbWUiLCJyZWFkVGltZSIsInNjYWxlIiwicmVhZERhdGUiLCJyZWFkRGF0ZVRpbWUyIiwicmVhZERhdGVUaW1lT2Zmc2V0IiwicmVhZE51bWVyaWMiLCJwcmVjaXNpb24iLCJyZWFkVW5pcXVlSWRlbnRpZmllciIsInJlYWRWYXJpYW50IiwiZGF0YSIsImxvd2VyQ2FzZUd1aWRzIiwiX3ByZWNpc2lvbiIsInNpZ24iLCJyZWFkVmFsdWUiLCJyZWFkVU51bWVyaWM2NExFIiwicmVhZFVOdW1lcmljOTZMRSIsInJlYWRVTnVtZXJpYzEyOExFIiwiY2FsbCIsIk1hdGgiLCJwb3ciLCJiYXNlVHlwZSIsIlRZUEUiLCJwcm9wQnl0ZXMiLCJfbWF4TGVuZ3RoIiwiX2NvbGxhdGlvbiIsImljb252IiwiZGVjb2RlIiwicmVhZE1heCIsImVxdWFscyIsInJlYWRNYXhVbmtub3duTGVuZ3RoIiwiY29uc29sZSIsIndhcm4iLCJleHBlY3RlZExlbmd0aCIsInJlYWRNYXhLbm93bkxlbmd0aCIsInRvdGFsTGVuZ3RoIiwiYWxsb2MiLCJvZmZzZXQiLCJuZXh0IiwiZG9uZSIsImNodW5rTGVuZ3RoIiwiY2h1bmsiLCJjb3B5IiwiY2h1bmtzIiwibGVuZ3RoIiwicHVzaCIsImNvbmNhdCIsImRheXMiLCJtaW51dGVzIiwiRGF0ZSIsIlVUQyIsInRocmVlSHVuZHJlZHRoc09mU2Vjb25kIiwibWlsbGlzZWNvbmRzIiwicm91bmQiLCJyZWFkVUludDI0TEUiLCJyZWFkVUludDQwTEUiLCJpIiwiZGF0ZSIsIk9iamVjdCIsImRlZmluZVByb3BlcnR5IiwiZW51bWVyYWJsZSIsInRpbWUiLCJnZXRIb3VycyIsImdldE1pbnV0ZXMiLCJnZXRTZWNvbmRzIiwiZ2V0TWlsbGlzZWNvbmRzIiwibmFub3NlY29uZHNEZWx0YSIsIm1vZHVsZSIsImV4cG9ydHMiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFDQTs7QUFFQTs7QUFFQTs7QUFDQTs7QUFDQTs7OztBQUVBLE1BQU1BLElBQUksR0FBRyxDQUFDLEtBQUssRUFBTixJQUFZLENBQXpCO0FBQ0EsTUFBTUMsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFOLElBQVksQ0FBeEI7QUFDQSxNQUFNQyxpQkFBaUIsR0FBRyxJQUFLLElBQUksQ0FBbkM7QUFDQSxNQUFNQyxhQUFhLEdBQUcsS0FBdEI7QUFDQSxNQUFNQyxRQUFRLEdBQUdDLE1BQU0sQ0FBQ0MsSUFBUCxDQUFZLENBQUMsSUFBRCxFQUFPLElBQVAsRUFBYSxJQUFiLEVBQW1CLElBQW5CLEVBQXlCLElBQXpCLEVBQStCLElBQS9CLEVBQXFDLElBQXJDLEVBQTJDLElBQTNDLENBQVosQ0FBakI7QUFDQSxNQUFNQyxlQUFlLEdBQUdGLE1BQU0sQ0FBQ0MsSUFBUCxDQUFZLENBQUMsSUFBRCxFQUFPLElBQVAsRUFBYSxJQUFiLEVBQW1CLElBQW5CLEVBQXlCLElBQXpCLEVBQStCLElBQS9CLEVBQXFDLElBQXJDLEVBQTJDLElBQTNDLENBQVosQ0FBeEI7QUFDQSxNQUFNRSxnQkFBZ0IsR0FBRyxNQUF6Qjs7QUFFQSxTQUFTQyxXQUFULENBQXFCQyxNQUFyQixFQUFxQ0MsUUFBckMsRUFBeUU7QUFDdkVELEVBQUFBLE1BQU0sQ0FBQ0UsU0FBUCxDQUFpQkQsUUFBakI7QUFDRDs7QUFFRCxTQUFTRSxZQUFULENBQXNCSCxNQUF0QixFQUFzQ0MsUUFBdEMsRUFBMEU7QUFDeEVELEVBQUFBLE1BQU0sQ0FBQ0ksV0FBUCxDQUFtQkgsUUFBbkI7QUFDRDs7QUFFRCxTQUFTSSxPQUFULENBQWlCTCxNQUFqQixFQUFpQ0MsUUFBakMsRUFBcUU7QUFDbkVELEVBQUFBLE1BQU0sQ0FBQ00sV0FBUCxDQUFtQkwsUUFBbkI7QUFDRDs7QUFFRCxTQUFTTSxVQUFULENBQW9CUCxNQUFwQixFQUFvQ0MsUUFBcEMsRUFBd0U7QUFDdEVELEVBQUFBLE1BQU0sQ0FBQ1EsY0FBUCxDQUF1QkMsS0FBRCxJQUFXO0FBQy9CUixJQUFBQSxRQUFRLENBQUNRLEtBQUssQ0FBQ0MsUUFBTixFQUFELENBQVI7QUFDRCxHQUZEO0FBR0Q7O0FBRUQsU0FBU0MsUUFBVCxDQUFrQlgsTUFBbEIsRUFBa0NDLFFBQWxDLEVBQXNFO0FBQ3BFRCxFQUFBQSxNQUFNLENBQUNZLFdBQVAsQ0FBbUJYLFFBQW5CO0FBQ0Q7O0FBRUQsU0FBU1ksU0FBVCxDQUFtQmIsTUFBbkIsRUFBbUNDLFFBQW5DLEVBQXVFO0FBQ3JFRCxFQUFBQSxNQUFNLENBQUNjLFlBQVAsQ0FBb0JiLFFBQXBCO0FBQ0Q7O0FBRUQsU0FBU2MsY0FBVCxDQUF3QmYsTUFBeEIsRUFBd0NDLFFBQXhDLEVBQTRFO0FBQzFFRCxFQUFBQSxNQUFNLENBQUNNLFdBQVAsQ0FBb0JHLEtBQUQsSUFBVztBQUM1QlIsSUFBQUEsUUFBUSxDQUFDUSxLQUFLLEdBQUdoQixhQUFULENBQVI7QUFDRCxHQUZEO0FBR0Q7O0FBRUQsU0FBU3VCLFNBQVQsQ0FBbUJoQixNQUFuQixFQUFtQ0MsUUFBbkMsRUFBdUU7QUFDckVELEVBQUFBLE1BQU0sQ0FBQ00sV0FBUCxDQUFvQlcsSUFBRCxJQUFVO0FBQzNCakIsSUFBQUEsTUFBTSxDQUFDa0IsWUFBUCxDQUFxQkMsR0FBRCxJQUFTO0FBQzNCbEIsTUFBQUEsUUFBUSxDQUFDLENBQUNrQixHQUFHLEdBQUksY0FBY0YsSUFBdEIsSUFBK0J4QixhQUFoQyxDQUFSO0FBQ0QsS0FGRDtBQUdELEdBSkQ7QUFLRDs7QUFFRCxTQUFTMkIsT0FBVCxDQUFpQnBCLE1BQWpCLEVBQWlDQyxRQUFqQyxFQUFxRTtBQUNuRUQsRUFBQUEsTUFBTSxDQUFDRSxTQUFQLENBQWtCTyxLQUFELElBQVc7QUFDMUJSLElBQUFBLFFBQVEsQ0FBQyxDQUFDLENBQUNRLEtBQUgsQ0FBUjtBQUNELEdBRkQ7QUFHRDs7QUFFRCxTQUFTWSxVQUFULENBQW9CckIsTUFBcEIsRUFBb0NzQixRQUFwQyxFQUF3REMsT0FBeEQsRUFBNEZ0QixRQUE1RixFQUFzSTtBQUNwSSxRQUFNdUIsSUFBSSxHQUFHRixRQUFRLENBQUNFLElBQXRCOztBQUVBLFVBQVFBLElBQUksQ0FBQ0MsSUFBYjtBQUNFLFNBQUssTUFBTDtBQUNFLGFBQU94QixRQUFRLENBQUMsSUFBRCxDQUFmOztBQUVGLFNBQUssU0FBTDtBQUNFLGFBQU9GLFdBQVcsQ0FBQ0MsTUFBRCxFQUFTQyxRQUFULENBQWxCOztBQUVGLFNBQUssVUFBTDtBQUNFLGFBQU9FLFlBQVksQ0FBQ0gsTUFBRCxFQUFTQyxRQUFULENBQW5COztBQUVGLFNBQUssS0FBTDtBQUNFLGFBQU9JLE9BQU8sQ0FBQ0wsTUFBRCxFQUFTQyxRQUFULENBQWQ7O0FBRUYsU0FBSyxRQUFMO0FBQ0UsYUFBT00sVUFBVSxDQUFDUCxNQUFELEVBQVNDLFFBQVQsQ0FBakI7O0FBRUYsU0FBSyxNQUFMO0FBQ0UsYUFBT0QsTUFBTSxDQUFDRSxTQUFQLENBQWtCd0IsVUFBRCxJQUFnQjtBQUN0QyxnQkFBUUEsVUFBUjtBQUNFLGVBQUssQ0FBTDtBQUNFLG1CQUFPekIsUUFBUSxDQUFDLElBQUQsQ0FBZjs7QUFFRixlQUFLLENBQUw7QUFDRSxtQkFBT0YsV0FBVyxDQUFDQyxNQUFELEVBQVNDLFFBQVQsQ0FBbEI7O0FBQ0YsZUFBSyxDQUFMO0FBQ0UsbUJBQU9FLFlBQVksQ0FBQ0gsTUFBRCxFQUFTQyxRQUFULENBQW5COztBQUNGLGVBQUssQ0FBTDtBQUNFLG1CQUFPSSxPQUFPLENBQUNMLE1BQUQsRUFBU0MsUUFBVCxDQUFkOztBQUNGLGVBQUssQ0FBTDtBQUNFLG1CQUFPTSxVQUFVLENBQUNQLE1BQUQsRUFBU0MsUUFBVCxDQUFqQjs7QUFFRjtBQUNFLGtCQUFNLElBQUkwQixLQUFKLENBQVUsNEJBQTRCRCxVQUE1QixHQUF5QyxXQUFuRCxDQUFOO0FBZEo7QUFnQkQsT0FqQk0sQ0FBUDs7QUFtQkYsU0FBSyxNQUFMO0FBQ0UsYUFBT2YsUUFBUSxDQUFDWCxNQUFELEVBQVNDLFFBQVQsQ0FBZjs7QUFFRixTQUFLLE9BQUw7QUFDRSxhQUFPWSxTQUFTLENBQUNiLE1BQUQsRUFBU0MsUUFBVCxDQUFoQjs7QUFFRixTQUFLLFFBQUw7QUFDRSxhQUFPRCxNQUFNLENBQUNFLFNBQVAsQ0FBa0J3QixVQUFELElBQWdCO0FBQ3RDLGdCQUFRQSxVQUFSO0FBQ0UsZUFBSyxDQUFMO0FBQ0UsbUJBQU96QixRQUFRLENBQUMsSUFBRCxDQUFmOztBQUVGLGVBQUssQ0FBTDtBQUNFLG1CQUFPVSxRQUFRLENBQUNYLE1BQUQsRUFBU0MsUUFBVCxDQUFmOztBQUNGLGVBQUssQ0FBTDtBQUNFLG1CQUFPWSxTQUFTLENBQUNiLE1BQUQsRUFBU0MsUUFBVCxDQUFoQjs7QUFFRjtBQUNFLGtCQUFNLElBQUkwQixLQUFKLENBQVUsNEJBQTRCRCxVQUE1QixHQUF5QyxhQUFuRCxDQUFOO0FBVko7QUFZRCxPQWJNLENBQVA7O0FBZUYsU0FBSyxZQUFMO0FBQ0UsYUFBT1gsY0FBYyxDQUFDZixNQUFELEVBQVNDLFFBQVQsQ0FBckI7O0FBRUYsU0FBSyxPQUFMO0FBQ0UsYUFBT2UsU0FBUyxDQUFDaEIsTUFBRCxFQUFTQyxRQUFULENBQWhCOztBQUVGLFNBQUssUUFBTDtBQUNFLGFBQU9ELE1BQU0sQ0FBQ0UsU0FBUCxDQUFrQndCLFVBQUQsSUFBZ0I7QUFDdEMsZ0JBQVFBLFVBQVI7QUFDRSxlQUFLLENBQUw7QUFDRSxtQkFBT3pCLFFBQVEsQ0FBQyxJQUFELENBQWY7O0FBRUYsZUFBSyxDQUFMO0FBQ0UsbUJBQU9jLGNBQWMsQ0FBQ2YsTUFBRCxFQUFTQyxRQUFULENBQXJCOztBQUNGLGVBQUssQ0FBTDtBQUNFLG1CQUFPZSxTQUFTLENBQUNoQixNQUFELEVBQVNDLFFBQVQsQ0FBaEI7O0FBRUY7QUFDRSxrQkFBTSxJQUFJMEIsS0FBSixDQUFVLDRCQUE0QkQsVUFBNUIsR0FBeUMsYUFBbkQsQ0FBTjtBQVZKO0FBWUQsT0FiTSxDQUFQOztBQWVGLFNBQUssS0FBTDtBQUNFLGFBQU9OLE9BQU8sQ0FBQ3BCLE1BQUQsRUFBU0MsUUFBVCxDQUFkOztBQUVGLFNBQUssTUFBTDtBQUNFLGFBQU9ELE1BQU0sQ0FBQ0UsU0FBUCxDQUFrQndCLFVBQUQsSUFBZ0I7QUFDdEMsZ0JBQVFBLFVBQVI7QUFDRSxlQUFLLENBQUw7QUFDRSxtQkFBT3pCLFFBQVEsQ0FBQyxJQUFELENBQWY7O0FBRUYsZUFBSyxDQUFMO0FBQ0UsbUJBQU9tQixPQUFPLENBQUNwQixNQUFELEVBQVNDLFFBQVQsQ0FBZDs7QUFFRjtBQUNFLGtCQUFNLElBQUkwQixLQUFKLENBQVUsNEJBQTRCRCxVQUE1QixHQUF5QyxXQUFuRCxDQUFOO0FBUko7QUFVRCxPQVhNLENBQVA7O0FBYUYsU0FBSyxTQUFMO0FBQ0EsU0FBSyxNQUFMO0FBQ0UsWUFBTUUsUUFBUSxHQUFHTixRQUFRLENBQUNPLFNBQVQsQ0FBb0JELFFBQXJDOztBQUNBLFVBQUlOLFFBQVEsQ0FBQ0ksVUFBVCxLQUF3Qm5DLEdBQTVCLEVBQWlDO0FBQy9CLGVBQU91QyxZQUFZLENBQUM5QixNQUFELEVBQVM0QixRQUFULEVBQW1CM0IsUUFBbkIsQ0FBbkI7QUFDRCxPQUZELE1BRU87QUFDTCxlQUFPRCxNQUFNLENBQUMrQixZQUFQLENBQXFCTCxVQUFELElBQWdCO0FBQ3pDLGNBQUlBLFVBQVUsS0FBS3BDLElBQW5CLEVBQXlCO0FBQ3ZCLG1CQUFPVyxRQUFRLENBQUMsSUFBRCxDQUFmO0FBQ0Q7O0FBRUQrQixVQUFBQSxTQUFTLENBQUNoQyxNQUFELEVBQVMwQixVQUFULEVBQXNCRSxRQUF0QixFQUFnQzNCLFFBQWhDLENBQVQ7QUFDRCxTQU5NLENBQVA7QUFPRDs7QUFFSCxTQUFLLFVBQUw7QUFDQSxTQUFLLE9BQUw7QUFDRSxVQUFJcUIsUUFBUSxDQUFDSSxVQUFULEtBQXdCbkMsR0FBNUIsRUFBaUM7QUFDL0IsZUFBTzBDLGFBQWEsQ0FBQ2pDLE1BQUQsRUFBU0MsUUFBVCxDQUFwQjtBQUNELE9BRkQsTUFFTztBQUNMLGVBQU9ELE1BQU0sQ0FBQytCLFlBQVAsQ0FBcUJMLFVBQUQsSUFBZ0I7QUFDekMsY0FBSUEsVUFBVSxLQUFLcEMsSUFBbkIsRUFBeUI7QUFDdkIsbUJBQU9XLFFBQVEsQ0FBQyxJQUFELENBQWY7QUFDRDs7QUFFRGlDLFVBQUFBLFVBQVUsQ0FBQ2xDLE1BQUQsRUFBUzBCLFVBQVQsRUFBc0J6QixRQUF0QixDQUFWO0FBQ0QsU0FOTSxDQUFQO0FBT0Q7O0FBRUgsU0FBSyxXQUFMO0FBQ0EsU0FBSyxRQUFMO0FBQ0UsVUFBSXFCLFFBQVEsQ0FBQ0ksVUFBVCxLQUF3Qm5DLEdBQTVCLEVBQWlDO0FBQy9CLGVBQU80QyxhQUFhLENBQUNuQyxNQUFELEVBQVNDLFFBQVQsQ0FBcEI7QUFDRCxPQUZELE1BRU87QUFDTCxlQUFPRCxNQUFNLENBQUMrQixZQUFQLENBQXFCTCxVQUFELElBQWdCO0FBQ3pDLGNBQUlBLFVBQVUsS0FBS3BDLElBQW5CLEVBQXlCO0FBQ3ZCLG1CQUFPVyxRQUFRLENBQUMsSUFBRCxDQUFmO0FBQ0Q7O0FBRURtQyxVQUFBQSxVQUFVLENBQUNwQyxNQUFELEVBQVMwQixVQUFULEVBQXNCekIsUUFBdEIsQ0FBVjtBQUNELFNBTk0sQ0FBUDtBQU9EOztBQUVILFNBQUssTUFBTDtBQUNFLGFBQU9ELE1BQU0sQ0FBQ0UsU0FBUCxDQUFrQm1DLGlCQUFELElBQXVCO0FBQzdDLFlBQUlBLGlCQUFpQixLQUFLLENBQTFCLEVBQTZCO0FBQzNCLGlCQUFPcEMsUUFBUSxDQUFDLElBQUQsQ0FBZjtBQUNEOztBQUVERCxRQUFBQSxNQUFNLENBQUNzQyxVQUFQLENBQWtCRCxpQkFBbEIsRUFBc0NFLFlBQUQsSUFBa0I7QUFDckR2QyxVQUFBQSxNQUFNLENBQUNzQyxVQUFQLENBQWtCLENBQWxCLEVBQXNCRSxVQUFELElBQWdCO0FBQ25DeEMsWUFBQUEsTUFBTSxDQUFDa0IsWUFBUCxDQUFxQlEsVUFBRCxJQUFnQjtBQUNsQ00sY0FBQUEsU0FBUyxDQUFDaEMsTUFBRCxFQUFTMEIsVUFBVCxFQUFzQkosUUFBUSxDQUFDTyxTQUFULENBQW9CRCxRQUExQyxFQUFxRDNCLFFBQXJELENBQVQ7QUFDRCxhQUZEO0FBR0QsV0FKRDtBQUtELFNBTkQ7QUFPRCxPQVpNLENBQVA7O0FBY0YsU0FBSyxPQUFMO0FBQ0UsYUFBT0QsTUFBTSxDQUFDRSxTQUFQLENBQWtCbUMsaUJBQUQsSUFBdUI7QUFDN0MsWUFBSUEsaUJBQWlCLEtBQUssQ0FBMUIsRUFBNkI7QUFDM0IsaUJBQU9wQyxRQUFRLENBQUMsSUFBRCxDQUFmO0FBQ0Q7O0FBRURELFFBQUFBLE1BQU0sQ0FBQ3NDLFVBQVAsQ0FBa0JELGlCQUFsQixFQUFzQ0UsWUFBRCxJQUFrQjtBQUNyRHZDLFVBQUFBLE1BQU0sQ0FBQ3NDLFVBQVAsQ0FBa0IsQ0FBbEIsRUFBc0JFLFVBQUQsSUFBZ0I7QUFDbkN4QyxZQUFBQSxNQUFNLENBQUNrQixZQUFQLENBQXFCUSxVQUFELElBQWdCO0FBQ2xDUSxjQUFBQSxVQUFVLENBQUNsQyxNQUFELEVBQVMwQixVQUFULEVBQXNCekIsUUFBdEIsQ0FBVjtBQUNELGFBRkQ7QUFHRCxXQUpEO0FBS0QsU0FORDtBQU9ELE9BWk0sQ0FBUDs7QUFjRixTQUFLLE9BQUw7QUFDRSxhQUFPRCxNQUFNLENBQUNFLFNBQVAsQ0FBa0JtQyxpQkFBRCxJQUF1QjtBQUM3QyxZQUFJQSxpQkFBaUIsS0FBSyxDQUExQixFQUE2QjtBQUMzQixpQkFBT3BDLFFBQVEsQ0FBQyxJQUFELENBQWY7QUFDRDs7QUFFREQsUUFBQUEsTUFBTSxDQUFDc0MsVUFBUCxDQUFrQkQsaUJBQWxCLEVBQXNDRSxZQUFELElBQWtCO0FBQ3JEdkMsVUFBQUEsTUFBTSxDQUFDc0MsVUFBUCxDQUFrQixDQUFsQixFQUFzQkUsVUFBRCxJQUFnQjtBQUNuQ3hDLFlBQUFBLE1BQU0sQ0FBQ2tCLFlBQVAsQ0FBcUJRLFVBQUQsSUFBZ0I7QUFDbENVLGNBQUFBLFVBQVUsQ0FBQ3BDLE1BQUQsRUFBUzBCLFVBQVQsRUFBc0J6QixRQUF0QixDQUFWO0FBQ0QsYUFGRDtBQUdELFdBSkQ7QUFLRCxTQU5EO0FBT0QsT0FaTSxDQUFQOztBQWNGLFNBQUssS0FBTDtBQUNFLGFBQU9nQyxhQUFhLENBQUNqQyxNQUFELEVBQVNDLFFBQVQsQ0FBcEI7O0FBRUYsU0FBSyxlQUFMO0FBQ0UsYUFBT3dDLGlCQUFpQixDQUFDekMsTUFBRCxFQUFTdUIsT0FBTyxDQUFDbUIsTUFBakIsRUFBeUJ6QyxRQUF6QixDQUF4Qjs7QUFFRixTQUFLLFVBQUw7QUFDRSxhQUFPMEMsWUFBWSxDQUFDM0MsTUFBRCxFQUFTdUIsT0FBTyxDQUFDbUIsTUFBakIsRUFBeUJ6QyxRQUF6QixDQUFuQjs7QUFFRixTQUFLLFdBQUw7QUFDRSxhQUFPRCxNQUFNLENBQUNFLFNBQVAsQ0FBa0J3QixVQUFELElBQWdCO0FBQ3RDLGdCQUFRQSxVQUFSO0FBQ0UsZUFBSyxDQUFMO0FBQ0UsbUJBQU96QixRQUFRLENBQUMsSUFBRCxDQUFmOztBQUVGLGVBQUssQ0FBTDtBQUNFLG1CQUFPd0MsaUJBQWlCLENBQUN6QyxNQUFELEVBQVN1QixPQUFPLENBQUNtQixNQUFqQixFQUF5QnpDLFFBQXpCLENBQXhCOztBQUNGLGVBQUssQ0FBTDtBQUNFLG1CQUFPMEMsWUFBWSxDQUFDM0MsTUFBRCxFQUFTdUIsT0FBTyxDQUFDbUIsTUFBakIsRUFBeUJ6QyxRQUF6QixDQUFuQjs7QUFFRjtBQUNFLGtCQUFNLElBQUkwQixLQUFKLENBQVUsNEJBQTRCRCxVQUE1QixHQUF5QyxnQkFBbkQsQ0FBTjtBQVZKO0FBWUQsT0FiTSxDQUFQOztBQWVGLFNBQUssTUFBTDtBQUNFLGFBQU8xQixNQUFNLENBQUNFLFNBQVAsQ0FBa0J3QixVQUFELElBQWdCO0FBQ3RDLFlBQUlBLFVBQVUsS0FBSyxDQUFuQixFQUFzQjtBQUNwQixpQkFBT3pCLFFBQVEsQ0FBQyxJQUFELENBQWY7QUFDRCxTQUZELE1BRU87QUFDTCxpQkFBTzJDLFFBQVEsQ0FBQzVDLE1BQUQsRUFBUzBCLFVBQVQsRUFBc0JKLFFBQVEsQ0FBQ3VCLEtBQS9CLEVBQXVDdEIsT0FBTyxDQUFDbUIsTUFBL0MsRUFBdUR6QyxRQUF2RCxDQUFmO0FBQ0Q7QUFDRixPQU5NLENBQVA7O0FBUUYsU0FBSyxNQUFMO0FBQ0UsYUFBT0QsTUFBTSxDQUFDRSxTQUFQLENBQWtCd0IsVUFBRCxJQUFnQjtBQUN0QyxZQUFJQSxVQUFVLEtBQUssQ0FBbkIsRUFBc0I7QUFDcEIsaUJBQU96QixRQUFRLENBQUMsSUFBRCxDQUFmO0FBQ0QsU0FGRCxNQUVPO0FBQ0wsaUJBQU82QyxRQUFRLENBQUM5QyxNQUFELEVBQVN1QixPQUFPLENBQUNtQixNQUFqQixFQUF5QnpDLFFBQXpCLENBQWY7QUFDRDtBQUNGLE9BTk0sQ0FBUDs7QUFRRixTQUFLLFdBQUw7QUFDRSxhQUFPRCxNQUFNLENBQUNFLFNBQVAsQ0FBa0J3QixVQUFELElBQWdCO0FBQ3RDLFlBQUlBLFVBQVUsS0FBSyxDQUFuQixFQUFzQjtBQUNwQixpQkFBT3pCLFFBQVEsQ0FBQyxJQUFELENBQWY7QUFDRCxTQUZELE1BRU87QUFDTCxpQkFBTzhDLGFBQWEsQ0FBQy9DLE1BQUQsRUFBUzBCLFVBQVQsRUFBc0JKLFFBQVEsQ0FBQ3VCLEtBQS9CLEVBQXVDdEIsT0FBTyxDQUFDbUIsTUFBL0MsRUFBdUR6QyxRQUF2RCxDQUFwQjtBQUNEO0FBQ0YsT0FOTSxDQUFQOztBQVFGLFNBQUssZ0JBQUw7QUFDRSxhQUFPRCxNQUFNLENBQUNFLFNBQVAsQ0FBa0J3QixVQUFELElBQWdCO0FBQ3RDLFlBQUlBLFVBQVUsS0FBSyxDQUFuQixFQUFzQjtBQUNwQixpQkFBT3pCLFFBQVEsQ0FBQyxJQUFELENBQWY7QUFDRCxTQUZELE1BRU87QUFDTCxpQkFBTytDLGtCQUFrQixDQUFDaEQsTUFBRCxFQUFTMEIsVUFBVCxFQUFzQkosUUFBUSxDQUFDdUIsS0FBL0IsRUFBdUM1QyxRQUF2QyxDQUF6QjtBQUNEO0FBQ0YsT0FOTSxDQUFQOztBQVFGLFNBQUssVUFBTDtBQUNBLFNBQUssVUFBTDtBQUNFLGFBQU9ELE1BQU0sQ0FBQ0UsU0FBUCxDQUFrQndCLFVBQUQsSUFBZ0I7QUFDdEMsWUFBSUEsVUFBVSxLQUFLLENBQW5CLEVBQXNCO0FBQ3BCLGlCQUFPekIsUUFBUSxDQUFDLElBQUQsQ0FBZjtBQUNELFNBRkQsTUFFTztBQUNMLGlCQUFPZ0QsV0FBVyxDQUFDakQsTUFBRCxFQUFTMEIsVUFBVCxFQUFzQkosUUFBUSxDQUFDNEIsU0FBL0IsRUFBMkM1QixRQUFRLENBQUN1QixLQUFwRCxFQUE0RDVDLFFBQTVELENBQWxCO0FBQ0Q7QUFDRixPQU5NLENBQVA7O0FBUUYsU0FBSyxrQkFBTDtBQUNFLGFBQU9ELE1BQU0sQ0FBQ0UsU0FBUCxDQUFrQndCLFVBQUQsSUFBZ0I7QUFDdEMsZ0JBQVFBLFVBQVI7QUFDRSxlQUFLLENBQUw7QUFDRSxtQkFBT3pCLFFBQVEsQ0FBQyxJQUFELENBQWY7O0FBRUYsZUFBSyxJQUFMO0FBQ0UsbUJBQU9rRCxvQkFBb0IsQ0FBQ25ELE1BQUQsRUFBU3VCLE9BQVQsRUFBa0J0QixRQUFsQixDQUEzQjs7QUFFRjtBQUNFLGtCQUFNLElBQUkwQixLQUFKLENBQVUsd0JBQVEsMEJBQVIsRUFBb0NELFVBQVUsR0FBSSxDQUFsRCxDQUFWLENBQU47QUFSSjtBQVVELE9BWE0sQ0FBUDs7QUFhRixTQUFLLEtBQUw7QUFDRSxhQUFPUyxhQUFhLENBQUNuQyxNQUFELEVBQVNDLFFBQVQsQ0FBcEI7O0FBRUYsU0FBSyxTQUFMO0FBQ0UsYUFBT0QsTUFBTSxDQUFDa0IsWUFBUCxDQUFxQlEsVUFBRCxJQUFnQjtBQUN6QyxZQUFJQSxVQUFVLEtBQUssQ0FBbkIsRUFBc0I7QUFDcEIsaUJBQU96QixRQUFRLENBQUMsSUFBRCxDQUFmO0FBQ0Q7O0FBRURtRCxRQUFBQSxXQUFXLENBQUNwRCxNQUFELEVBQVN1QixPQUFULEVBQWtCRyxVQUFsQixFQUErQnpCLFFBQS9CLENBQVg7QUFDRCxPQU5NLENBQVA7O0FBUUY7QUFDRSxZQUFNLElBQUkwQixLQUFKLENBQVUsd0JBQVEsc0JBQVIsRUFBZ0NILElBQUksQ0FBQ0MsSUFBckMsQ0FBVixDQUFOO0FBM1JKO0FBNlJEOztBQUVELFNBQVMwQixvQkFBVCxDQUE4Qm5ELE1BQTlCLEVBQThDdUIsT0FBOUMsRUFBa0Z0QixRQUFsRixFQUFzSDtBQUNwSEQsRUFBQUEsTUFBTSxDQUFDc0MsVUFBUCxDQUFrQixJQUFsQixFQUF5QmUsSUFBRCxJQUFVO0FBQ2hDcEQsSUFBQUEsUUFBUSxDQUFDc0IsT0FBTyxDQUFDK0IsY0FBUixHQUF5Qix1Q0FBc0JELElBQXRCLENBQXpCLEdBQXVELHVDQUFzQkEsSUFBdEIsQ0FBeEQsQ0FBUjtBQUNELEdBRkQ7QUFHRDs7QUFFRCxTQUFTSixXQUFULENBQXFCakQsTUFBckIsRUFBcUMwQixVQUFyQyxFQUF5RDZCLFVBQXpELEVBQTZFVixLQUE3RSxFQUE0RjVDLFFBQTVGLEVBQWdJO0FBQzlIRCxFQUFBQSxNQUFNLENBQUNFLFNBQVAsQ0FBa0JzRCxJQUFELElBQVU7QUFDekJBLElBQUFBLElBQUksR0FBR0EsSUFBSSxLQUFLLENBQVQsR0FBYSxDQUFiLEdBQWlCLENBQUMsQ0FBekI7QUFFQSxRQUFJQyxTQUFKOztBQUNBLFFBQUkvQixVQUFVLEtBQUssQ0FBbkIsRUFBc0I7QUFDcEIrQixNQUFBQSxTQUFTLEdBQUd6RCxNQUFNLENBQUNrQixZQUFuQjtBQUNELEtBRkQsTUFFTyxJQUFJUSxVQUFVLEtBQUssQ0FBbkIsRUFBc0I7QUFDM0IrQixNQUFBQSxTQUFTLEdBQUd6RCxNQUFNLENBQUMwRCxnQkFBbkI7QUFDRCxLQUZNLE1BRUEsSUFBSWhDLFVBQVUsS0FBSyxFQUFuQixFQUF1QjtBQUM1QitCLE1BQUFBLFNBQVMsR0FBR3pELE1BQU0sQ0FBQzJELGdCQUFuQjtBQUNELEtBRk0sTUFFQSxJQUFJakMsVUFBVSxLQUFLLEVBQW5CLEVBQXVCO0FBQzVCK0IsTUFBQUEsU0FBUyxHQUFHekQsTUFBTSxDQUFDNEQsaUJBQW5CO0FBQ0QsS0FGTSxNQUVBO0FBQ0wsWUFBTSxJQUFJakMsS0FBSixDQUFVLHdCQUFRLG1DQUFSLEVBQTZDRCxVQUE3QyxDQUFWLENBQU47QUFDRDs7QUFFRCtCLElBQUFBLFNBQVMsQ0FBQ0ksSUFBVixDQUFlN0QsTUFBZixFQUF3QlMsS0FBRCxJQUFXO0FBQ2hDUixNQUFBQSxRQUFRLENBQUVRLEtBQUssR0FBRytDLElBQVQsR0FBaUJNLElBQUksQ0FBQ0MsR0FBTCxDQUFTLEVBQVQsRUFBYWxCLEtBQWIsQ0FBbEIsQ0FBUjtBQUNELEtBRkQ7QUFHRCxHQW5CRDtBQW9CRDs7QUFFRCxTQUFTTyxXQUFULENBQXFCcEQsTUFBckIsRUFBcUN1QixPQUFyQyxFQUF5RUcsVUFBekUsRUFBNkZ6QixRQUE3RixFQUFpSTtBQUMvSCxTQUFPRCxNQUFNLENBQUNFLFNBQVAsQ0FBa0I4RCxRQUFELElBQWM7QUFDcEMsVUFBTXhDLElBQUksR0FBR3lDLGVBQUtELFFBQUwsQ0FBYjtBQUVBLFdBQU9oRSxNQUFNLENBQUNFLFNBQVAsQ0FBa0JnRSxTQUFELElBQWU7QUFDckN4QyxNQUFBQSxVQUFVLEdBQUdBLFVBQVUsR0FBR3dDLFNBQWIsR0FBeUIsQ0FBdEM7O0FBRUEsY0FBUTFDLElBQUksQ0FBQ0MsSUFBYjtBQUNFLGFBQUssa0JBQUw7QUFDRSxpQkFBTzBCLG9CQUFvQixDQUFDbkQsTUFBRCxFQUFTdUIsT0FBVCxFQUFrQnRCLFFBQWxCLENBQTNCOztBQUVGLGFBQUssS0FBTDtBQUNFLGlCQUFPbUIsT0FBTyxDQUFDcEIsTUFBRCxFQUFTQyxRQUFULENBQWQ7O0FBRUYsYUFBSyxTQUFMO0FBQ0UsaUJBQU9GLFdBQVcsQ0FBQ0MsTUFBRCxFQUFTQyxRQUFULENBQWxCOztBQUVGLGFBQUssVUFBTDtBQUNFLGlCQUFPRSxZQUFZLENBQUNILE1BQUQsRUFBU0MsUUFBVCxDQUFuQjs7QUFFRixhQUFLLEtBQUw7QUFDRSxpQkFBT0ksT0FBTyxDQUFDTCxNQUFELEVBQVNDLFFBQVQsQ0FBZDs7QUFFRixhQUFLLFFBQUw7QUFDRSxpQkFBT00sVUFBVSxDQUFDUCxNQUFELEVBQVNDLFFBQVQsQ0FBakI7O0FBRUYsYUFBSyxlQUFMO0FBQ0UsaUJBQU93QyxpQkFBaUIsQ0FBQ3pDLE1BQUQsRUFBU3VCLE9BQU8sQ0FBQ21CLE1BQWpCLEVBQXlCekMsUUFBekIsQ0FBeEI7O0FBRUYsYUFBSyxVQUFMO0FBQ0UsaUJBQU8wQyxZQUFZLENBQUMzQyxNQUFELEVBQVN1QixPQUFPLENBQUNtQixNQUFqQixFQUF5QnpDLFFBQXpCLENBQW5COztBQUVGLGFBQUssTUFBTDtBQUNFLGlCQUFPVSxRQUFRLENBQUNYLE1BQUQsRUFBU0MsUUFBVCxDQUFmOztBQUVGLGFBQUssT0FBTDtBQUNFLGlCQUFPWSxTQUFTLENBQUNiLE1BQUQsRUFBU0MsUUFBVCxDQUFoQjs7QUFFRixhQUFLLFlBQUw7QUFDRSxpQkFBT2MsY0FBYyxDQUFDZixNQUFELEVBQVNDLFFBQVQsQ0FBckI7O0FBRUYsYUFBSyxPQUFMO0FBQ0UsaUJBQU9lLFNBQVMsQ0FBQ2hCLE1BQUQsRUFBU0MsUUFBVCxDQUFoQjs7QUFFRixhQUFLLE1BQUw7QUFDRSxpQkFBTzZDLFFBQVEsQ0FBQzlDLE1BQUQsRUFBU3VCLE9BQU8sQ0FBQ21CLE1BQWpCLEVBQXlCekMsUUFBekIsQ0FBZjs7QUFFRixhQUFLLE1BQUw7QUFDRSxpQkFBT0QsTUFBTSxDQUFDRSxTQUFQLENBQWtCMkMsS0FBRCxJQUFXO0FBQ2pDLG1CQUFPRCxRQUFRLENBQUM1QyxNQUFELEVBQVMwQixVQUFULEVBQXFCbUIsS0FBckIsRUFBNEJ0QixPQUFPLENBQUNtQixNQUFwQyxFQUE0Q3pDLFFBQTVDLENBQWY7QUFDRCxXQUZNLENBQVA7O0FBSUYsYUFBSyxXQUFMO0FBQ0UsaUJBQU9ELE1BQU0sQ0FBQ0UsU0FBUCxDQUFrQjJDLEtBQUQsSUFBVztBQUNqQyxtQkFBT0UsYUFBYSxDQUFDL0MsTUFBRCxFQUFTMEIsVUFBVCxFQUFxQm1CLEtBQXJCLEVBQTRCdEIsT0FBTyxDQUFDbUIsTUFBcEMsRUFBNEN6QyxRQUE1QyxDQUFwQjtBQUNELFdBRk0sQ0FBUDs7QUFJRixhQUFLLGdCQUFMO0FBQ0UsaUJBQU9ELE1BQU0sQ0FBQ0UsU0FBUCxDQUFrQjJDLEtBQUQsSUFBVztBQUNqQyxtQkFBT0csa0JBQWtCLENBQUNoRCxNQUFELEVBQVMwQixVQUFULEVBQXFCbUIsS0FBckIsRUFBNEI1QyxRQUE1QixDQUF6QjtBQUNELFdBRk0sQ0FBUDs7QUFJRixhQUFLLFdBQUw7QUFDQSxhQUFLLFFBQUw7QUFDRSxpQkFBT0QsTUFBTSxDQUFDK0IsWUFBUCxDQUFxQm9DLFVBQUQsSUFBZ0I7QUFDekMvQixZQUFBQSxVQUFVLENBQUNwQyxNQUFELEVBQVMwQixVQUFULEVBQXFCekIsUUFBckIsQ0FBVjtBQUNELFdBRk0sQ0FBUDs7QUFJRixhQUFLLFVBQUw7QUFDQSxhQUFLLFVBQUw7QUFDRSxpQkFBT0QsTUFBTSxDQUFDRSxTQUFQLENBQWtCZ0QsU0FBRCxJQUFlO0FBQ3JDbEQsWUFBQUEsTUFBTSxDQUFDRSxTQUFQLENBQWtCMkMsS0FBRCxJQUFXO0FBQzFCSSxjQUFBQSxXQUFXLENBQUNqRCxNQUFELEVBQVMwQixVQUFULEVBQXFCd0IsU0FBckIsRUFBZ0NMLEtBQWhDLEVBQXVDNUMsUUFBdkMsQ0FBWDtBQUNELGFBRkQ7QUFHRCxXQUpNLENBQVA7O0FBTUYsYUFBSyxTQUFMO0FBQ0EsYUFBSyxNQUFMO0FBQ0UsaUJBQU9ELE1BQU0sQ0FBQytCLFlBQVAsQ0FBcUJvQyxVQUFELElBQWdCO0FBQ3pDLCtDQUFjbkUsTUFBZCxFQUF1QjZCLFNBQUQsSUFBZTtBQUNuQ0csY0FBQUEsU0FBUyxDQUFDaEMsTUFBRCxFQUFTMEIsVUFBVCxFQUFxQkcsU0FBUyxDQUFDRCxRQUEvQixFQUEwQzNCLFFBQTFDLENBQVQ7QUFDRCxhQUZEO0FBR0QsV0FKTSxDQUFQOztBQU1GLGFBQUssVUFBTDtBQUNBLGFBQUssT0FBTDtBQUNFLGlCQUFPRCxNQUFNLENBQUMrQixZQUFQLENBQXFCb0MsVUFBRCxJQUFnQjtBQUN6QywrQ0FBY25FLE1BQWQsRUFBdUJvRSxVQUFELElBQWdCO0FBQ3BDbEMsY0FBQUEsVUFBVSxDQUFDbEMsTUFBRCxFQUFTMEIsVUFBVCxFQUFxQnpCLFFBQXJCLENBQVY7QUFDRCxhQUZEO0FBR0QsV0FKTSxDQUFQOztBQU1GO0FBQ0UsZ0JBQU0sSUFBSTBCLEtBQUosQ0FBVSxlQUFWLENBQU47QUF0Rko7QUF3RkQsS0EzRk0sQ0FBUDtBQTRGRCxHQS9GTSxDQUFQO0FBZ0dEOztBQUVELFNBQVNTLFVBQVQsQ0FBb0JwQyxNQUFwQixFQUFvQzBCLFVBQXBDLEVBQXdEekIsUUFBeEQsRUFBNEY7QUFDMUYsU0FBT0QsTUFBTSxDQUFDc0MsVUFBUCxDQUFrQlosVUFBbEIsRUFBOEJ6QixRQUE5QixDQUFQO0FBQ0Q7O0FBRUQsU0FBUytCLFNBQVQsQ0FBbUJoQyxNQUFuQixFQUFtQzBCLFVBQW5DLEVBQXVERSxRQUF2RCxFQUF5RTNCLFFBQXpFLEVBQTZHO0FBQzNHLE1BQUkyQixRQUFRLElBQUksSUFBaEIsRUFBc0I7QUFDcEJBLElBQUFBLFFBQVEsR0FBRzlCLGdCQUFYO0FBQ0Q7O0FBRUQsU0FBT0UsTUFBTSxDQUFDc0MsVUFBUCxDQUFrQlosVUFBbEIsRUFBK0IyQixJQUFELElBQVU7QUFDN0NwRCxJQUFBQSxRQUFRLENBQUNvRSxtQkFBTUMsTUFBTixDQUFhakIsSUFBYixFQUFtQnpCLFFBQW5CLENBQUQsQ0FBUjtBQUNELEdBRk0sQ0FBUDtBQUdEOztBQUVELFNBQVNNLFVBQVQsQ0FBb0JsQyxNQUFwQixFQUFvQzBCLFVBQXBDLEVBQXdEekIsUUFBeEQsRUFBNEY7QUFDMUZELEVBQUFBLE1BQU0sQ0FBQ3NDLFVBQVAsQ0FBa0JaLFVBQWxCLEVBQStCMkIsSUFBRCxJQUFVO0FBQ3RDcEQsSUFBQUEsUUFBUSxDQUFDb0QsSUFBSSxDQUFDM0MsUUFBTCxDQUFjLE1BQWQsQ0FBRCxDQUFSO0FBQ0QsR0FGRDtBQUdEOztBQUVELFNBQVN5QixhQUFULENBQXVCbkMsTUFBdkIsRUFBdUNDLFFBQXZDLEVBQTJFO0FBQ3pFLFNBQU9zRSxPQUFPLENBQUN2RSxNQUFELEVBQVNDLFFBQVQsQ0FBZDtBQUNEOztBQUVELFNBQVM2QixZQUFULENBQXNCOUIsTUFBdEIsRUFBc0M0QixRQUF0QyxFQUF3RDNCLFFBQXhELEVBQTRGO0FBQzFGLE1BQUkyQixRQUFRLElBQUksSUFBaEIsRUFBc0I7QUFDcEJBLElBQUFBLFFBQVEsR0FBRzlCLGdCQUFYO0FBQ0Q7O0FBRUR5RSxFQUFBQSxPQUFPLENBQUN2RSxNQUFELEVBQVVxRCxJQUFELElBQVU7QUFDeEIsUUFBSUEsSUFBSixFQUFVO0FBQ1JwRCxNQUFBQSxRQUFRLENBQUNvRSxtQkFBTUMsTUFBTixDQUFhakIsSUFBYixFQUFtQnpCLFFBQW5CLENBQUQsQ0FBUjtBQUNELEtBRkQsTUFFTztBQUNMM0IsTUFBQUEsUUFBUSxDQUFDLElBQUQsQ0FBUjtBQUNEO0FBQ0YsR0FOTSxDQUFQO0FBT0Q7O0FBRUQsU0FBU2dDLGFBQVQsQ0FBdUJqQyxNQUF2QixFQUF1Q0MsUUFBdkMsRUFBaUY7QUFDL0VzRSxFQUFBQSxPQUFPLENBQUN2RSxNQUFELEVBQVVxRCxJQUFELElBQVU7QUFDeEIsUUFBSUEsSUFBSixFQUFVO0FBQ1JwRCxNQUFBQSxRQUFRLENBQUNvRCxJQUFJLENBQUMzQyxRQUFMLENBQWMsTUFBZCxDQUFELENBQVI7QUFDRCxLQUZELE1BRU87QUFDTFQsTUFBQUEsUUFBUSxDQUFDLElBQUQsQ0FBUjtBQUNEO0FBQ0YsR0FOTSxDQUFQO0FBT0Q7O0FBRUQsU0FBU3NFLE9BQVQsQ0FBaUJ2RSxNQUFqQixFQUFpQ0MsUUFBakMsRUFBMkU7QUFDekVELEVBQUFBLE1BQU0sQ0FBQ3NDLFVBQVAsQ0FBa0IsQ0FBbEIsRUFBc0JkLElBQUQsSUFBVTtBQUM3QixRQUFJQSxJQUFJLENBQUNnRCxNQUFMLENBQVk5RSxRQUFaLENBQUosRUFBMkI7QUFDekIsYUFBT08sUUFBUSxDQUFDLElBQUQsQ0FBZjtBQUNELEtBRkQsTUFFTyxJQUFJdUIsSUFBSSxDQUFDZ0QsTUFBTCxDQUFZM0UsZUFBWixDQUFKLEVBQWtDO0FBQ3ZDLGFBQU80RSxvQkFBb0IsQ0FBQ3pFLE1BQUQsRUFBU0MsUUFBVCxDQUEzQjtBQUNELEtBRk0sTUFFQTtBQUNMLFlBQU1rQixHQUFHLEdBQUdLLElBQUksQ0FBQ04sWUFBTCxDQUFrQixDQUFsQixDQUFaO0FBQ0EsWUFBTUQsSUFBSSxHQUFHTyxJQUFJLENBQUNOLFlBQUwsQ0FBa0IsQ0FBbEIsQ0FBYjs7QUFFQSxVQUFJRCxJQUFJLElBQUssS0FBTSxLQUFLLEVBQXhCLEVBQThCO0FBQzVCeUQsUUFBQUEsT0FBTyxDQUFDQyxJQUFSLENBQWEsb0NBQW9DMUQsSUFBcEMsR0FBMkMsUUFBM0MsR0FBc0RFLEdBQW5FO0FBQ0Q7O0FBRUQsWUFBTXlELGNBQWMsR0FBR3pELEdBQUcsR0FBSSxjQUFjRixJQUE1QztBQUNBLGFBQU80RCxrQkFBa0IsQ0FBQzdFLE1BQUQsRUFBUzRFLGNBQVQsRUFBeUIzRSxRQUF6QixDQUF6QjtBQUNEO0FBQ0YsR0FoQkQ7QUFpQkQ7O0FBRUQsU0FBUzRFLGtCQUFULENBQTRCN0UsTUFBNUIsRUFBNEM4RSxXQUE1QyxFQUFpRTdFLFFBQWpFLEVBQTJHO0FBQ3pHLFFBQU1vRCxJQUFJLEdBQUcxRCxNQUFNLENBQUNvRixLQUFQLENBQWFELFdBQWIsRUFBMEIsQ0FBMUIsQ0FBYjtBQUVBLE1BQUlFLE1BQU0sR0FBRyxDQUFiOztBQUNBLFdBQVNDLElBQVQsQ0FBY0MsSUFBZCxFQUF5QjtBQUN2QmxGLElBQUFBLE1BQU0sQ0FBQ2tCLFlBQVAsQ0FBcUJpRSxXQUFELElBQWlCO0FBQ25DLFVBQUksQ0FBQ0EsV0FBTCxFQUFrQjtBQUNoQixlQUFPRCxJQUFJLEVBQVg7QUFDRDs7QUFFRGxGLE1BQUFBLE1BQU0sQ0FBQ3NDLFVBQVAsQ0FBa0I2QyxXQUFsQixFQUFnQ0MsS0FBRCxJQUFXO0FBQ3hDQSxRQUFBQSxLQUFLLENBQUNDLElBQU4sQ0FBV2hDLElBQVgsRUFBaUIyQixNQUFqQjtBQUNBQSxRQUFBQSxNQUFNLElBQUlHLFdBQVY7QUFFQUYsUUFBQUEsSUFBSSxDQUFDQyxJQUFELENBQUo7QUFDRCxPQUxEO0FBTUQsS0FYRDtBQVlEOztBQUVERCxFQUFBQSxJQUFJLENBQUMsTUFBTTtBQUNULFFBQUlELE1BQU0sS0FBS0YsV0FBZixFQUE0QjtBQUMxQixZQUFNLElBQUluRCxLQUFKLENBQVUsa0VBQWtFbUQsV0FBbEUsR0FBZ0YsWUFBaEYsR0FBK0ZFLE1BQS9GLEdBQXdHLFFBQWxILENBQU47QUFDRDs7QUFFRC9FLElBQUFBLFFBQVEsQ0FBQ29ELElBQUQsQ0FBUjtBQUNELEdBTkcsQ0FBSjtBQU9EOztBQUVELFNBQVNvQixvQkFBVCxDQUE4QnpFLE1BQTlCLEVBQThDQyxRQUE5QyxFQUF3RjtBQUN0RixRQUFNcUYsTUFBZ0IsR0FBRyxFQUF6QjtBQUVBLE1BQUlDLE1BQU0sR0FBRyxDQUFiOztBQUNBLFdBQVNOLElBQVQsQ0FBY0MsSUFBZCxFQUF5QjtBQUN2QmxGLElBQUFBLE1BQU0sQ0FBQ2tCLFlBQVAsQ0FBcUJpRSxXQUFELElBQWlCO0FBQ25DLFVBQUksQ0FBQ0EsV0FBTCxFQUFrQjtBQUNoQixlQUFPRCxJQUFJLEVBQVg7QUFDRDs7QUFFRGxGLE1BQUFBLE1BQU0sQ0FBQ3NDLFVBQVAsQ0FBa0I2QyxXQUFsQixFQUFnQ0MsS0FBRCxJQUFXO0FBQ3hDRSxRQUFBQSxNQUFNLENBQUNFLElBQVAsQ0FBWUosS0FBWjtBQUNBRyxRQUFBQSxNQUFNLElBQUlKLFdBQVY7QUFFQUYsUUFBQUEsSUFBSSxDQUFDQyxJQUFELENBQUo7QUFDRCxPQUxEO0FBTUQsS0FYRDtBQVlEOztBQUVERCxFQUFBQSxJQUFJLENBQUMsTUFBTTtBQUNUaEYsSUFBQUEsUUFBUSxDQUFDTixNQUFNLENBQUM4RixNQUFQLENBQWNILE1BQWQsRUFBc0JDLE1BQXRCLENBQUQsQ0FBUjtBQUNELEdBRkcsQ0FBSjtBQUdEOztBQUVELFNBQVM5QyxpQkFBVCxDQUEyQnpDLE1BQTNCLEVBQTJDMEMsTUFBM0MsRUFBNER6QyxRQUE1RCxFQUE2RjtBQUMzRkQsRUFBQUEsTUFBTSxDQUFDK0IsWUFBUCxDQUFxQjJELElBQUQsSUFBVTtBQUM1QjFGLElBQUFBLE1BQU0sQ0FBQytCLFlBQVAsQ0FBcUI0RCxPQUFELElBQWE7QUFDL0IsVUFBSWxGLEtBQUo7O0FBQ0EsVUFBSWlDLE1BQUosRUFBWTtBQUNWakMsUUFBQUEsS0FBSyxHQUFHLElBQUltRixJQUFKLENBQVNBLElBQUksQ0FBQ0MsR0FBTCxDQUFTLElBQVQsRUFBZSxDQUFmLEVBQWtCLElBQUlILElBQXRCLEVBQTRCLENBQTVCLEVBQStCQyxPQUEvQixDQUFULENBQVI7QUFDRCxPQUZELE1BRU87QUFDTGxGLFFBQUFBLEtBQUssR0FBRyxJQUFJbUYsSUFBSixDQUFTLElBQVQsRUFBZSxDQUFmLEVBQWtCLElBQUlGLElBQXRCLEVBQTRCLENBQTVCLEVBQStCQyxPQUEvQixDQUFSO0FBQ0Q7O0FBQ0QxRixNQUFBQSxRQUFRLENBQUNRLEtBQUQsQ0FBUjtBQUNELEtBUkQ7QUFTRCxHQVZEO0FBV0Q7O0FBRUQsU0FBU2tDLFlBQVQsQ0FBc0IzQyxNQUF0QixFQUFzQzBDLE1BQXRDLEVBQXVEekMsUUFBdkQsRUFBd0Y7QUFDdEZELEVBQUFBLE1BQU0sQ0FBQ00sV0FBUCxDQUFvQm9GLElBQUQsSUFBVTtBQUMzQjFGLElBQUFBLE1BQU0sQ0FBQ2tCLFlBQVAsQ0FBcUI0RSx1QkFBRCxJQUE2QjtBQUMvQyxZQUFNQyxZQUFZLEdBQUdqQyxJQUFJLENBQUNrQyxLQUFMLENBQVdGLHVCQUF1QixHQUFHdEcsaUJBQXJDLENBQXJCO0FBRUEsVUFBSWlCLEtBQUo7O0FBQ0EsVUFBSWlDLE1BQUosRUFBWTtBQUNWakMsUUFBQUEsS0FBSyxHQUFHLElBQUltRixJQUFKLENBQVNBLElBQUksQ0FBQ0MsR0FBTCxDQUFTLElBQVQsRUFBZSxDQUFmLEVBQWtCLElBQUlILElBQXRCLEVBQTRCLENBQTVCLEVBQStCLENBQS9CLEVBQWtDLENBQWxDLEVBQXFDSyxZQUFyQyxDQUFULENBQVI7QUFDRCxPQUZELE1BRU87QUFDTHRGLFFBQUFBLEtBQUssR0FBRyxJQUFJbUYsSUFBSixDQUFTLElBQVQsRUFBZSxDQUFmLEVBQWtCLElBQUlGLElBQXRCLEVBQTRCLENBQTVCLEVBQStCLENBQS9CLEVBQWtDLENBQWxDLEVBQXFDSyxZQUFyQyxDQUFSO0FBQ0Q7O0FBRUQ5RixNQUFBQSxRQUFRLENBQUNRLEtBQUQsQ0FBUjtBQUNELEtBWEQ7QUFZRCxHQWJEO0FBY0Q7O0FBTUQsU0FBU21DLFFBQVQsQ0FBa0I1QyxNQUFsQixFQUFrQzBCLFVBQWxDLEVBQXNEbUIsS0FBdEQsRUFBcUVILE1BQXJFLEVBQXNGekMsUUFBdEYsRUFBMkk7QUFDekksTUFBSXdELFNBQUo7O0FBQ0EsVUFBUS9CLFVBQVI7QUFDRSxTQUFLLENBQUw7QUFDRStCLE1BQUFBLFNBQVMsR0FBR3pELE1BQU0sQ0FBQ2lHLFlBQW5CO0FBQ0E7O0FBQ0YsU0FBSyxDQUFMO0FBQ0V4QyxNQUFBQSxTQUFTLEdBQUd6RCxNQUFNLENBQUNrQixZQUFuQjtBQUNBOztBQUNGLFNBQUssQ0FBTDtBQUNFdUMsTUFBQUEsU0FBUyxHQUFHekQsTUFBTSxDQUFDa0csWUFBbkI7QUFSSjs7QUFXQXpDLEVBQUFBLFNBQVMsQ0FBRUksSUFBWCxDQUFnQjdELE1BQWhCLEVBQXlCUyxLQUFELElBQW1CO0FBQ3pDLFFBQUlvQyxLQUFLLEdBQUcsQ0FBWixFQUFlO0FBQ2IsV0FBSyxJQUFJc0QsQ0FBQyxHQUFHdEQsS0FBYixFQUFvQnNELENBQUMsR0FBRyxDQUF4QixFQUEyQkEsQ0FBQyxFQUE1QixFQUFnQztBQUM5QjFGLFFBQUFBLEtBQUssSUFBSSxFQUFUO0FBQ0Q7QUFDRjs7QUFFRCxRQUFJMkYsSUFBSjs7QUFDQSxRQUFJMUQsTUFBSixFQUFZO0FBQ1YwRCxNQUFBQSxJQUFJLEdBQUcsSUFBSVIsSUFBSixDQUFTQSxJQUFJLENBQUNDLEdBQUwsQ0FBUyxJQUFULEVBQWUsQ0FBZixFQUFrQixDQUFsQixFQUFxQixDQUFyQixFQUF3QixDQUF4QixFQUEyQixDQUEzQixFQUE4QnBGLEtBQUssR0FBRyxLQUF0QyxDQUFULENBQVA7QUFDRCxLQUZELE1BRU87QUFDTDJGLE1BQUFBLElBQUksR0FBRyxJQUFJUixJQUFKLENBQVMsSUFBVCxFQUFlLENBQWYsRUFBa0IsQ0FBbEIsRUFBcUIsQ0FBckIsRUFBd0IsQ0FBeEIsRUFBMkIsQ0FBM0IsRUFBOEJuRixLQUFLLEdBQUcsS0FBdEMsQ0FBUDtBQUNEOztBQUNENEYsSUFBQUEsTUFBTSxDQUFDQyxjQUFQLENBQXNCRixJQUF0QixFQUE0QixrQkFBNUIsRUFBZ0Q7QUFDOUNHLE1BQUFBLFVBQVUsRUFBRSxLQURrQztBQUU5QzlGLE1BQUFBLEtBQUssRUFBR0EsS0FBSyxHQUFHLEtBQVQsR0FBa0JxRCxJQUFJLENBQUNDLEdBQUwsQ0FBUyxFQUFULEVBQWEsQ0FBYjtBQUZxQixLQUFoRDtBQUlBOUQsSUFBQUEsUUFBUSxDQUFDbUcsSUFBRCxDQUFSO0FBQ0QsR0FsQkQ7QUFtQkQ7O0FBRUQsU0FBU3RELFFBQVQsQ0FBa0I5QyxNQUFsQixFQUFrQzBDLE1BQWxDLEVBQW1EekMsUUFBbkQsRUFBb0Y7QUFDbEZELEVBQUFBLE1BQU0sQ0FBQ2lHLFlBQVAsQ0FBcUJQLElBQUQsSUFBVTtBQUM1QixRQUFJaEQsTUFBSixFQUFZO0FBQ1Z6QyxNQUFBQSxRQUFRLENBQUMsSUFBSTJGLElBQUosQ0FBU0EsSUFBSSxDQUFDQyxHQUFMLENBQVMsSUFBVCxFQUFlLENBQWYsRUFBa0JILElBQUksR0FBRyxNQUF6QixDQUFULENBQUQsQ0FBUjtBQUNELEtBRkQsTUFFTztBQUNMekYsTUFBQUEsUUFBUSxDQUFDLElBQUkyRixJQUFKLENBQVMsSUFBVCxFQUFlLENBQWYsRUFBa0JGLElBQUksR0FBRyxNQUF6QixDQUFELENBQVI7QUFDRDtBQUNGLEdBTkQ7QUFPRDs7QUFFRCxTQUFTM0MsYUFBVCxDQUF1Qi9DLE1BQXZCLEVBQXVDMEIsVUFBdkMsRUFBMkRtQixLQUEzRCxFQUEwRUgsTUFBMUUsRUFBMkZ6QyxRQUEzRixFQUFnSjtBQUM5STJDLEVBQUFBLFFBQVEsQ0FBQzVDLE1BQUQsRUFBUzBCLFVBQVUsR0FBRyxDQUF0QixFQUF5Qm1CLEtBQXpCLEVBQWdDSCxNQUFoQyxFQUF5QzhELElBQUQsSUFBVTtBQUFFO0FBQzFEeEcsSUFBQUEsTUFBTSxDQUFDaUcsWUFBUCxDQUFxQlAsSUFBRCxJQUFVO0FBQzVCLFVBQUlVLElBQUo7O0FBQ0EsVUFBSTFELE1BQUosRUFBWTtBQUNWMEQsUUFBQUEsSUFBSSxHQUFHLElBQUlSLElBQUosQ0FBU0EsSUFBSSxDQUFDQyxHQUFMLENBQVMsSUFBVCxFQUFlLENBQWYsRUFBa0JILElBQUksR0FBRyxNQUF6QixFQUFpQyxDQUFqQyxFQUFvQyxDQUFwQyxFQUF1QyxDQUF2QyxFQUEwQyxDQUFDYyxJQUEzQyxDQUFULENBQVA7QUFDRCxPQUZELE1BRU87QUFDTEosUUFBQUEsSUFBSSxHQUFHLElBQUlSLElBQUosQ0FBUyxJQUFULEVBQWUsQ0FBZixFQUFrQkYsSUFBSSxHQUFHLE1BQXpCLEVBQWlDYyxJQUFJLENBQUNDLFFBQUwsRUFBakMsRUFBa0RELElBQUksQ0FBQ0UsVUFBTCxFQUFsRCxFQUFxRUYsSUFBSSxDQUFDRyxVQUFMLEVBQXJFLEVBQXdGSCxJQUFJLENBQUNJLGVBQUwsRUFBeEYsQ0FBUDtBQUNEOztBQUNEUCxNQUFBQSxNQUFNLENBQUNDLGNBQVAsQ0FBc0JGLElBQXRCLEVBQTRCLGtCQUE1QixFQUFnRDtBQUM5Q0csUUFBQUEsVUFBVSxFQUFFLEtBRGtDO0FBRTlDOUYsUUFBQUEsS0FBSyxFQUFFK0YsSUFBSSxDQUFDSztBQUZrQyxPQUFoRDtBQUlBNUcsTUFBQUEsUUFBUSxDQUFDbUcsSUFBRCxDQUFSO0FBQ0QsS0FaRDtBQWFELEdBZE8sQ0FBUjtBQWVEOztBQUVELFNBQVNwRCxrQkFBVCxDQUE0QmhELE1BQTVCLEVBQTRDMEIsVUFBNUMsRUFBZ0VtQixLQUFoRSxFQUErRTVDLFFBQS9FLEVBQW9JO0FBQ2xJMkMsRUFBQUEsUUFBUSxDQUFDNUMsTUFBRCxFQUFTMEIsVUFBVSxHQUFHLENBQXRCLEVBQXlCbUIsS0FBekIsRUFBZ0MsSUFBaEMsRUFBdUMyRCxJQUFELElBQVU7QUFDdER4RyxJQUFBQSxNQUFNLENBQUNpRyxZQUFQLENBQXFCUCxJQUFELElBQVU7QUFDNUI7QUFDQTFGLE1BQUFBLE1BQU0sQ0FBQ0ksV0FBUCxDQUFtQixNQUFNO0FBQ3ZCLGNBQU1nRyxJQUFJLEdBQUcsSUFBSVIsSUFBSixDQUFTQSxJQUFJLENBQUNDLEdBQUwsQ0FBUyxJQUFULEVBQWUsQ0FBZixFQUFrQkgsSUFBSSxHQUFHLE1BQXpCLEVBQWlDLENBQWpDLEVBQW9DLENBQXBDLEVBQXVDLENBQXZDLEVBQTBDLENBQUNjLElBQTNDLENBQVQsQ0FBYjtBQUNBSCxRQUFBQSxNQUFNLENBQUNDLGNBQVAsQ0FBc0JGLElBQXRCLEVBQTRCLGtCQUE1QixFQUFnRDtBQUM5Q0csVUFBQUEsVUFBVSxFQUFFLEtBRGtDO0FBRTlDOUYsVUFBQUEsS0FBSyxFQUFFK0YsSUFBSSxDQUFDSztBQUZrQyxTQUFoRDtBQUlBNUcsUUFBQUEsUUFBUSxDQUFDbUcsSUFBRCxDQUFSO0FBQ0QsT0FQRDtBQVFELEtBVkQ7QUFXRCxHQVpPLENBQVI7QUFhRDs7ZUFFYy9FLFU7O0FBQ2Z5RixNQUFNLENBQUNDLE9BQVAsR0FBaUIxRixVQUFqQiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBQYXJzZXIgZnJvbSAnLi90b2tlbi9zdHJlYW0tcGFyc2VyJztcbmltcG9ydCB7IE1ldGFkYXRhLCByZWFkQ29sbGF0aW9uIH0gZnJvbSAnLi9tZXRhZGF0YS1wYXJzZXInO1xuaW1wb3J0IHsgSW50ZXJuYWxDb25uZWN0aW9uT3B0aW9ucyB9IGZyb20gJy4vY29ubmVjdGlvbic7XG5pbXBvcnQgeyBUWVBFIH0gZnJvbSAnLi9kYXRhLXR5cGUnO1xuXG5pbXBvcnQgaWNvbnYgZnJvbSAnaWNvbnYtbGl0ZSc7XG5pbXBvcnQgeyBzcHJpbnRmIH0gZnJvbSAnc3ByaW50Zi1qcyc7XG5pbXBvcnQgeyBidWZmZXJUb0xvd2VyQ2FzZUd1aWQsIGJ1ZmZlclRvVXBwZXJDYXNlR3VpZCB9IGZyb20gJy4vZ3VpZC1wYXJzZXInO1xuXG5jb25zdCBOVUxMID0gKDEgPDwgMTYpIC0gMTtcbmNvbnN0IE1BWCA9ICgxIDw8IDE2KSAtIDE7XG5jb25zdCBUSFJFRV9BTkRfQV9USElSRCA9IDMgKyAoMSAvIDMpO1xuY29uc3QgTU9ORVlfRElWSVNPUiA9IDEwMDAwO1xuY29uc3QgUExQX05VTEwgPSBCdWZmZXIuZnJvbShbMHhGRiwgMHhGRiwgMHhGRiwgMHhGRiwgMHhGRiwgMHhGRiwgMHhGRiwgMHhGRl0pO1xuY29uc3QgVU5LTk9XTl9QTFBfTEVOID0gQnVmZmVyLmZyb20oWzB4RkUsIDB4RkYsIDB4RkYsIDB4RkYsIDB4RkYsIDB4RkYsIDB4RkYsIDB4RkZdKTtcbmNvbnN0IERFRkFVTFRfRU5DT0RJTkcgPSAndXRmOCc7XG5cbmZ1bmN0aW9uIHJlYWRUaW55SW50KHBhcnNlcjogUGFyc2VyLCBjYWxsYmFjazogKHZhbHVlOiB1bmtub3duKSA9PiB2b2lkKSB7XG4gIHBhcnNlci5yZWFkVUludDgoY2FsbGJhY2spO1xufVxuXG5mdW5jdGlvbiByZWFkU21hbGxJbnQocGFyc2VyOiBQYXJzZXIsIGNhbGxiYWNrOiAodmFsdWU6IHVua25vd24pID0+IHZvaWQpIHtcbiAgcGFyc2VyLnJlYWRJbnQxNkxFKGNhbGxiYWNrKTtcbn1cblxuZnVuY3Rpb24gcmVhZEludChwYXJzZXI6IFBhcnNlciwgY2FsbGJhY2s6ICh2YWx1ZTogdW5rbm93bikgPT4gdm9pZCkge1xuICBwYXJzZXIucmVhZEludDMyTEUoY2FsbGJhY2spO1xufVxuXG5mdW5jdGlvbiByZWFkQmlnSW50KHBhcnNlcjogUGFyc2VyLCBjYWxsYmFjazogKHZhbHVlOiB1bmtub3duKSA9PiB2b2lkKSB7XG4gIHBhcnNlci5yZWFkQmlnSW50NjRMRSgodmFsdWUpID0+IHtcbiAgICBjYWxsYmFjayh2YWx1ZS50b1N0cmluZygpKTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJlYWRSZWFsKHBhcnNlcjogUGFyc2VyLCBjYWxsYmFjazogKHZhbHVlOiB1bmtub3duKSA9PiB2b2lkKSB7XG4gIHBhcnNlci5yZWFkRmxvYXRMRShjYWxsYmFjayk7XG59XG5cbmZ1bmN0aW9uIHJlYWRGbG9hdChwYXJzZXI6IFBhcnNlciwgY2FsbGJhY2s6ICh2YWx1ZTogdW5rbm93bikgPT4gdm9pZCkge1xuICBwYXJzZXIucmVhZERvdWJsZUxFKGNhbGxiYWNrKTtcbn1cblxuZnVuY3Rpb24gcmVhZFNtYWxsTW9uZXkocGFyc2VyOiBQYXJzZXIsIGNhbGxiYWNrOiAodmFsdWU6IHVua25vd24pID0+IHZvaWQpIHtcbiAgcGFyc2VyLnJlYWRJbnQzMkxFKCh2YWx1ZSkgPT4ge1xuICAgIGNhbGxiYWNrKHZhbHVlIC8gTU9ORVlfRElWSVNPUik7XG4gIH0pO1xufVxuXG5mdW5jdGlvbiByZWFkTW9uZXkocGFyc2VyOiBQYXJzZXIsIGNhbGxiYWNrOiAodmFsdWU6IHVua25vd24pID0+IHZvaWQpIHtcbiAgcGFyc2VyLnJlYWRJbnQzMkxFKChoaWdoKSA9PiB7XG4gICAgcGFyc2VyLnJlYWRVSW50MzJMRSgobG93KSA9PiB7XG4gICAgICBjYWxsYmFjaygobG93ICsgKDB4MTAwMDAwMDAwICogaGlnaCkpIC8gTU9ORVlfRElWSVNPUik7XG4gICAgfSk7XG4gIH0pO1xufVxuXG5mdW5jdGlvbiByZWFkQml0KHBhcnNlcjogUGFyc2VyLCBjYWxsYmFjazogKHZhbHVlOiB1bmtub3duKSA9PiB2b2lkKSB7XG4gIHBhcnNlci5yZWFkVUludDgoKHZhbHVlKSA9PiB7XG4gICAgY2FsbGJhY2soISF2YWx1ZSk7XG4gIH0pO1xufVxuXG5mdW5jdGlvbiB2YWx1ZVBhcnNlKHBhcnNlcjogUGFyc2VyLCBtZXRhZGF0YTogTWV0YWRhdGEsIG9wdGlvbnM6IEludGVybmFsQ29ubmVjdGlvbk9wdGlvbnMsIGNhbGxiYWNrOiAodmFsdWU6IHVua25vd24pID0+IHZvaWQpOiB2b2lkIHtcbiAgY29uc3QgdHlwZSA9IG1ldGFkYXRhLnR5cGU7XG5cbiAgc3dpdGNoICh0eXBlLm5hbWUpIHtcbiAgICBjYXNlICdOdWxsJzpcbiAgICAgIHJldHVybiBjYWxsYmFjayhudWxsKTtcblxuICAgIGNhc2UgJ1RpbnlJbnQnOlxuICAgICAgcmV0dXJuIHJlYWRUaW55SW50KHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgY2FzZSAnU21hbGxJbnQnOlxuICAgICAgcmV0dXJuIHJlYWRTbWFsbEludChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgIGNhc2UgJ0ludCc6XG4gICAgICByZXR1cm4gcmVhZEludChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgIGNhc2UgJ0JpZ0ludCc6XG4gICAgICByZXR1cm4gcmVhZEJpZ0ludChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgIGNhc2UgJ0ludE4nOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDgoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgc3dpdGNoIChkYXRhTGVuZ3RoKSB7XG4gICAgICAgICAgY2FzZSAwOlxuICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuXG4gICAgICAgICAgY2FzZSAxOlxuICAgICAgICAgICAgcmV0dXJuIHJlYWRUaW55SW50KHBhcnNlciwgY2FsbGJhY2spO1xuICAgICAgICAgIGNhc2UgMjpcbiAgICAgICAgICAgIHJldHVybiByZWFkU21hbGxJbnQocGFyc2VyLCBjYWxsYmFjayk7XG4gICAgICAgICAgY2FzZSA0OlxuICAgICAgICAgICAgcmV0dXJuIHJlYWRJbnQocGFyc2VyLCBjYWxsYmFjayk7XG4gICAgICAgICAgY2FzZSA4OlxuICAgICAgICAgICAgcmV0dXJuIHJlYWRCaWdJbnQocGFyc2VyLCBjYWxsYmFjayk7XG5cbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVbnN1cHBvcnRlZCBkYXRhTGVuZ3RoICcgKyBkYXRhTGVuZ3RoICsgJyBmb3IgSW50TicpO1xuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgIGNhc2UgJ1JlYWwnOlxuICAgICAgcmV0dXJuIHJlYWRSZWFsKHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgY2FzZSAnRmxvYXQnOlxuICAgICAgcmV0dXJuIHJlYWRGbG9hdChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgIGNhc2UgJ0Zsb2F0Tic6XG4gICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgoZGF0YUxlbmd0aCkgPT4ge1xuICAgICAgICBzd2l0Y2ggKGRhdGFMZW5ndGgpIHtcbiAgICAgICAgICBjYXNlIDA6XG4gICAgICAgICAgICByZXR1cm4gY2FsbGJhY2sobnVsbCk7XG5cbiAgICAgICAgICBjYXNlIDQ6XG4gICAgICAgICAgICByZXR1cm4gcmVhZFJlYWwocGFyc2VyLCBjYWxsYmFjayk7XG4gICAgICAgICAgY2FzZSA4OlxuICAgICAgICAgICAgcmV0dXJuIHJlYWRGbG9hdChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Vuc3VwcG9ydGVkIGRhdGFMZW5ndGggJyArIGRhdGFMZW5ndGggKyAnIGZvciBGbG9hdE4nKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICBjYXNlICdTbWFsbE1vbmV5JzpcbiAgICAgIHJldHVybiByZWFkU21hbGxNb25leShwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgIGNhc2UgJ01vbmV5JzpcbiAgICAgIHJldHVybiByZWFkTW9uZXkocGFyc2VyLCBjYWxsYmFjayk7XG5cbiAgICBjYXNlICdNb25leU4nOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDgoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgc3dpdGNoIChkYXRhTGVuZ3RoKSB7XG4gICAgICAgICAgY2FzZSAwOlxuICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuXG4gICAgICAgICAgY2FzZSA0OlxuICAgICAgICAgICAgcmV0dXJuIHJlYWRTbWFsbE1vbmV5KHBhcnNlciwgY2FsbGJhY2spO1xuICAgICAgICAgIGNhc2UgODpcbiAgICAgICAgICAgIHJldHVybiByZWFkTW9uZXkocGFyc2VyLCBjYWxsYmFjayk7XG5cbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVbnN1cHBvcnRlZCBkYXRhTGVuZ3RoICcgKyBkYXRhTGVuZ3RoICsgJyBmb3IgTW9uZXlOJyk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgY2FzZSAnQml0JzpcbiAgICAgIHJldHVybiByZWFkQml0KHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgY2FzZSAnQml0Tic6XG4gICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgoZGF0YUxlbmd0aCkgPT4ge1xuICAgICAgICBzd2l0Y2ggKGRhdGFMZW5ndGgpIHtcbiAgICAgICAgICBjYXNlIDA6XG4gICAgICAgICAgICByZXR1cm4gY2FsbGJhY2sobnVsbCk7XG5cbiAgICAgICAgICBjYXNlIDE6XG4gICAgICAgICAgICByZXR1cm4gcmVhZEJpdChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Vuc3VwcG9ydGVkIGRhdGFMZW5ndGggJyArIGRhdGFMZW5ndGggKyAnIGZvciBCaXROJyk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgY2FzZSAnVmFyQ2hhcic6XG4gICAgY2FzZSAnQ2hhcic6XG4gICAgICBjb25zdCBjb2RlcGFnZSA9IG1ldGFkYXRhLmNvbGxhdGlvbiEuY29kZXBhZ2UhO1xuICAgICAgaWYgKG1ldGFkYXRhLmRhdGFMZW5ndGggPT09IE1BWCkge1xuICAgICAgICByZXR1cm4gcmVhZE1heENoYXJzKHBhcnNlciwgY29kZXBhZ2UsIGNhbGxiYWNrKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiBwYXJzZXIucmVhZFVJbnQxNkxFKChkYXRhTGVuZ3RoKSA9PiB7XG4gICAgICAgICAgaWYgKGRhdGFMZW5ndGggPT09IE5VTEwpIHtcbiAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhudWxsKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICByZWFkQ2hhcnMocGFyc2VyLCBkYXRhTGVuZ3RoISwgY29kZXBhZ2UsIGNhbGxiYWNrKTtcbiAgICAgICAgfSk7XG4gICAgICB9XG5cbiAgICBjYXNlICdOVmFyQ2hhcic6XG4gICAgY2FzZSAnTkNoYXInOlxuICAgICAgaWYgKG1ldGFkYXRhLmRhdGFMZW5ndGggPT09IE1BWCkge1xuICAgICAgICByZXR1cm4gcmVhZE1heE5DaGFycyhwYXJzZXIsIGNhbGxiYWNrKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiBwYXJzZXIucmVhZFVJbnQxNkxFKChkYXRhTGVuZ3RoKSA9PiB7XG4gICAgICAgICAgaWYgKGRhdGFMZW5ndGggPT09IE5VTEwpIHtcbiAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhudWxsKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICByZWFkTkNoYXJzKHBhcnNlciwgZGF0YUxlbmd0aCEsIGNhbGxiYWNrKTtcbiAgICAgICAgfSk7XG4gICAgICB9XG5cbiAgICBjYXNlICdWYXJCaW5hcnknOlxuICAgIGNhc2UgJ0JpbmFyeSc6XG4gICAgICBpZiAobWV0YWRhdGEuZGF0YUxlbmd0aCA9PT0gTUFYKSB7XG4gICAgICAgIHJldHVybiByZWFkTWF4QmluYXJ5KHBhcnNlciwgY2FsbGJhY2spO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDE2TEUoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgICBpZiAoZGF0YUxlbmd0aCA9PT0gTlVMTCkge1xuICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIHJlYWRCaW5hcnkocGFyc2VyLCBkYXRhTGVuZ3RoISwgY2FsbGJhY2spO1xuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgIGNhc2UgJ1RleHQnOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDgoKHRleHRQb2ludGVyTGVuZ3RoKSA9PiB7XG4gICAgICAgIGlmICh0ZXh0UG9pbnRlckxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgIHJldHVybiBjYWxsYmFjayhudWxsKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHBhcnNlci5yZWFkQnVmZmVyKHRleHRQb2ludGVyTGVuZ3RoLCAoX3RleHRQb2ludGVyKSA9PiB7XG4gICAgICAgICAgcGFyc2VyLnJlYWRCdWZmZXIoOCwgKF90aW1lc3RhbXApID0+IHtcbiAgICAgICAgICAgIHBhcnNlci5yZWFkVUludDMyTEUoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgICAgICAgcmVhZENoYXJzKHBhcnNlciwgZGF0YUxlbmd0aCEsIG1ldGFkYXRhLmNvbGxhdGlvbiEuY29kZXBhZ2UhLCBjYWxsYmFjayk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG4gICAgICB9KTtcblxuICAgIGNhc2UgJ05UZXh0JzpcbiAgICAgIHJldHVybiBwYXJzZXIucmVhZFVJbnQ4KCh0ZXh0UG9pbnRlckxlbmd0aCkgPT4ge1xuICAgICAgICBpZiAodGV4dFBvaW50ZXJMZW5ndGggPT09IDApIHtcbiAgICAgICAgICByZXR1cm4gY2FsbGJhY2sobnVsbCk7XG4gICAgICAgIH1cblxuICAgICAgICBwYXJzZXIucmVhZEJ1ZmZlcih0ZXh0UG9pbnRlckxlbmd0aCwgKF90ZXh0UG9pbnRlcikgPT4ge1xuICAgICAgICAgIHBhcnNlci5yZWFkQnVmZmVyKDgsIChfdGltZXN0YW1wKSA9PiB7XG4gICAgICAgICAgICBwYXJzZXIucmVhZFVJbnQzMkxFKChkYXRhTGVuZ3RoKSA9PiB7XG4gICAgICAgICAgICAgIHJlYWROQ2hhcnMocGFyc2VyLCBkYXRhTGVuZ3RoISwgY2FsbGJhY2spO1xuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgfSk7XG4gICAgICAgIH0pO1xuICAgICAgfSk7XG5cbiAgICBjYXNlICdJbWFnZSc6XG4gICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgodGV4dFBvaW50ZXJMZW5ndGgpID0+IHtcbiAgICAgICAgaWYgKHRleHRQb2ludGVyTGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuICAgICAgICB9XG5cbiAgICAgICAgcGFyc2VyLnJlYWRCdWZmZXIodGV4dFBvaW50ZXJMZW5ndGgsIChfdGV4dFBvaW50ZXIpID0+IHtcbiAgICAgICAgICBwYXJzZXIucmVhZEJ1ZmZlcig4LCAoX3RpbWVzdGFtcCkgPT4ge1xuICAgICAgICAgICAgcGFyc2VyLnJlYWRVSW50MzJMRSgoZGF0YUxlbmd0aCkgPT4ge1xuICAgICAgICAgICAgICByZWFkQmluYXJ5KHBhcnNlciwgZGF0YUxlbmd0aCEsIGNhbGxiYWNrKTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgIH0pO1xuICAgICAgICB9KTtcbiAgICAgIH0pO1xuXG4gICAgY2FzZSAnWG1sJzpcbiAgICAgIHJldHVybiByZWFkTWF4TkNoYXJzKHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgY2FzZSAnU21hbGxEYXRlVGltZSc6XG4gICAgICByZXR1cm4gcmVhZFNtYWxsRGF0ZVRpbWUocGFyc2VyLCBvcHRpb25zLnVzZVVUQywgY2FsbGJhY2spO1xuXG4gICAgY2FzZSAnRGF0ZVRpbWUnOlxuICAgICAgcmV0dXJuIHJlYWREYXRlVGltZShwYXJzZXIsIG9wdGlvbnMudXNlVVRDLCBjYWxsYmFjayk7XG5cbiAgICBjYXNlICdEYXRlVGltZU4nOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDgoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgc3dpdGNoIChkYXRhTGVuZ3RoKSB7XG4gICAgICAgICAgY2FzZSAwOlxuICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuXG4gICAgICAgICAgY2FzZSA0OlxuICAgICAgICAgICAgcmV0dXJuIHJlYWRTbWFsbERhdGVUaW1lKHBhcnNlciwgb3B0aW9ucy51c2VVVEMsIGNhbGxiYWNrKTtcbiAgICAgICAgICBjYXNlIDg6XG4gICAgICAgICAgICByZXR1cm4gcmVhZERhdGVUaW1lKHBhcnNlciwgb3B0aW9ucy51c2VVVEMsIGNhbGxiYWNrKTtcblxuICAgICAgICAgIGRlZmF1bHQ6XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Vuc3VwcG9ydGVkIGRhdGFMZW5ndGggJyArIGRhdGFMZW5ndGggKyAnIGZvciBEYXRlVGltZU4nKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICBjYXNlICdUaW1lJzpcbiAgICAgIHJldHVybiBwYXJzZXIucmVhZFVJbnQ4KChkYXRhTGVuZ3RoKSA9PiB7XG4gICAgICAgIGlmIChkYXRhTGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHJldHVybiByZWFkVGltZShwYXJzZXIsIGRhdGFMZW5ndGghLCBtZXRhZGF0YS5zY2FsZSEsIG9wdGlvbnMudXNlVVRDLCBjYWxsYmFjayk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgY2FzZSAnRGF0ZSc6XG4gICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgoZGF0YUxlbmd0aCkgPT4ge1xuICAgICAgICBpZiAoZGF0YUxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgIHJldHVybiBjYWxsYmFjayhudWxsKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXR1cm4gcmVhZERhdGUocGFyc2VyLCBvcHRpb25zLnVzZVVUQywgY2FsbGJhY2spO1xuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgIGNhc2UgJ0RhdGVUaW1lMic6XG4gICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgoZGF0YUxlbmd0aCkgPT4ge1xuICAgICAgICBpZiAoZGF0YUxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgIHJldHVybiBjYWxsYmFjayhudWxsKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICByZXR1cm4gcmVhZERhdGVUaW1lMihwYXJzZXIsIGRhdGFMZW5ndGghLCBtZXRhZGF0YS5zY2FsZSEsIG9wdGlvbnMudXNlVVRDLCBjYWxsYmFjayk7XG4gICAgICAgIH1cbiAgICAgIH0pO1xuXG4gICAgY2FzZSAnRGF0ZVRpbWVPZmZzZXQnOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDgoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgaWYgKGRhdGFMZW5ndGggPT09IDApIHtcbiAgICAgICAgICByZXR1cm4gY2FsbGJhY2sobnVsbCk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgcmV0dXJuIHJlYWREYXRlVGltZU9mZnNldChwYXJzZXIsIGRhdGFMZW5ndGghLCBtZXRhZGF0YS5zY2FsZSEsIGNhbGxiYWNrKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICBjYXNlICdOdW1lcmljTic6XG4gICAgY2FzZSAnRGVjaW1hbE4nOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDgoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgaWYgKGRhdGFMZW5ndGggPT09IDApIHtcbiAgICAgICAgICByZXR1cm4gY2FsbGJhY2sobnVsbCk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgcmV0dXJuIHJlYWROdW1lcmljKHBhcnNlciwgZGF0YUxlbmd0aCEsIG1ldGFkYXRhLnByZWNpc2lvbiEsIG1ldGFkYXRhLnNjYWxlISwgY2FsbGJhY2spO1xuICAgICAgICB9XG4gICAgICB9KTtcblxuICAgIGNhc2UgJ1VuaXF1ZUlkZW50aWZpZXInOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDgoKGRhdGFMZW5ndGgpID0+IHtcbiAgICAgICAgc3dpdGNoIChkYXRhTGVuZ3RoKSB7XG4gICAgICAgICAgY2FzZSAwOlxuICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuXG4gICAgICAgICAgY2FzZSAweDEwOlxuICAgICAgICAgICAgcmV0dXJuIHJlYWRVbmlxdWVJZGVudGlmaWVyKHBhcnNlciwgb3B0aW9ucywgY2FsbGJhY2spO1xuXG4gICAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihzcHJpbnRmKCdVbnN1cHBvcnRlZCBndWlkIHNpemUgJWQnLCBkYXRhTGVuZ3RoISAtIDEpKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICBjYXNlICdVRFQnOlxuICAgICAgcmV0dXJuIHJlYWRNYXhCaW5hcnkocGFyc2VyLCBjYWxsYmFjayk7XG5cbiAgICBjYXNlICdWYXJpYW50JzpcbiAgICAgIHJldHVybiBwYXJzZXIucmVhZFVJbnQzMkxFKChkYXRhTGVuZ3RoKSA9PiB7XG4gICAgICAgIGlmIChkYXRhTGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG51bGwpO1xuICAgICAgICB9XG5cbiAgICAgICAgcmVhZFZhcmlhbnQocGFyc2VyLCBvcHRpb25zLCBkYXRhTGVuZ3RoISwgY2FsbGJhY2spO1xuICAgICAgfSk7XG5cbiAgICBkZWZhdWx0OlxuICAgICAgdGhyb3cgbmV3IEVycm9yKHNwcmludGYoJ1VucmVjb2duaXNlZCB0eXBlICVzJywgdHlwZS5uYW1lKSk7XG4gIH1cbn1cblxuZnVuY3Rpb24gcmVhZFVuaXF1ZUlkZW50aWZpZXIocGFyc2VyOiBQYXJzZXIsIG9wdGlvbnM6IEludGVybmFsQ29ubmVjdGlvbk9wdGlvbnMsIGNhbGxiYWNrOiAodmFsdWU6IHVua25vd24pID0+IHZvaWQpIHtcbiAgcGFyc2VyLnJlYWRCdWZmZXIoMHgxMCwgKGRhdGEpID0+IHtcbiAgICBjYWxsYmFjayhvcHRpb25zLmxvd2VyQ2FzZUd1aWRzID8gYnVmZmVyVG9Mb3dlckNhc2VHdWlkKGRhdGEpIDogYnVmZmVyVG9VcHBlckNhc2VHdWlkKGRhdGEpKTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJlYWROdW1lcmljKHBhcnNlcjogUGFyc2VyLCBkYXRhTGVuZ3RoOiBudW1iZXIsIF9wcmVjaXNpb246IG51bWJlciwgc2NhbGU6IG51bWJlciwgY2FsbGJhY2s6ICh2YWx1ZTogdW5rbm93bikgPT4gdm9pZCkge1xuICBwYXJzZXIucmVhZFVJbnQ4KChzaWduKSA9PiB7XG4gICAgc2lnbiA9IHNpZ24gPT09IDEgPyAxIDogLTE7XG5cbiAgICBsZXQgcmVhZFZhbHVlO1xuICAgIGlmIChkYXRhTGVuZ3RoID09PSA1KSB7XG4gICAgICByZWFkVmFsdWUgPSBwYXJzZXIucmVhZFVJbnQzMkxFO1xuICAgIH0gZWxzZSBpZiAoZGF0YUxlbmd0aCA9PT0gOSkge1xuICAgICAgcmVhZFZhbHVlID0gcGFyc2VyLnJlYWRVTnVtZXJpYzY0TEU7XG4gICAgfSBlbHNlIGlmIChkYXRhTGVuZ3RoID09PSAxMykge1xuICAgICAgcmVhZFZhbHVlID0gcGFyc2VyLnJlYWRVTnVtZXJpYzk2TEU7XG4gICAgfSBlbHNlIGlmIChkYXRhTGVuZ3RoID09PSAxNykge1xuICAgICAgcmVhZFZhbHVlID0gcGFyc2VyLnJlYWRVTnVtZXJpYzEyOExFO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3Ioc3ByaW50ZignVW5zdXBwb3J0ZWQgbnVtZXJpYyBkYXRhTGVuZ3RoICVkJywgZGF0YUxlbmd0aCkpO1xuICAgIH1cblxuICAgIHJlYWRWYWx1ZS5jYWxsKHBhcnNlciwgKHZhbHVlKSA9PiB7XG4gICAgICBjYWxsYmFjaygodmFsdWUgKiBzaWduKSAvIE1hdGgucG93KDEwLCBzY2FsZSkpO1xuICAgIH0pO1xuICB9KTtcbn1cblxuZnVuY3Rpb24gcmVhZFZhcmlhbnQocGFyc2VyOiBQYXJzZXIsIG9wdGlvbnM6IEludGVybmFsQ29ubmVjdGlvbk9wdGlvbnMsIGRhdGFMZW5ndGg6IG51bWJlciwgY2FsbGJhY2s6ICh2YWx1ZTogdW5rbm93bikgPT4gdm9pZCkge1xuICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgoYmFzZVR5cGUpID0+IHtcbiAgICBjb25zdCB0eXBlID0gVFlQRVtiYXNlVHlwZV07XG5cbiAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgocHJvcEJ5dGVzKSA9PiB7XG4gICAgICBkYXRhTGVuZ3RoID0gZGF0YUxlbmd0aCAtIHByb3BCeXRlcyAtIDI7XG5cbiAgICAgIHN3aXRjaCAodHlwZS5uYW1lKSB7XG4gICAgICAgIGNhc2UgJ1VuaXF1ZUlkZW50aWZpZXInOlxuICAgICAgICAgIHJldHVybiByZWFkVW5pcXVlSWRlbnRpZmllcihwYXJzZXIsIG9wdGlvbnMsIGNhbGxiYWNrKTtcblxuICAgICAgICBjYXNlICdCaXQnOlxuICAgICAgICAgIHJldHVybiByZWFkQml0KHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgICAgIGNhc2UgJ1RpbnlJbnQnOlxuICAgICAgICAgIHJldHVybiByZWFkVGlueUludChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgICAgICBjYXNlICdTbWFsbEludCc6XG4gICAgICAgICAgcmV0dXJuIHJlYWRTbWFsbEludChwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgICAgICBjYXNlICdJbnQnOlxuICAgICAgICAgIHJldHVybiByZWFkSW50KHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgICAgIGNhc2UgJ0JpZ0ludCc6XG4gICAgICAgICAgcmV0dXJuIHJlYWRCaWdJbnQocGFyc2VyLCBjYWxsYmFjayk7XG5cbiAgICAgICAgY2FzZSAnU21hbGxEYXRlVGltZSc6XG4gICAgICAgICAgcmV0dXJuIHJlYWRTbWFsbERhdGVUaW1lKHBhcnNlciwgb3B0aW9ucy51c2VVVEMsIGNhbGxiYWNrKTtcblxuICAgICAgICBjYXNlICdEYXRlVGltZSc6XG4gICAgICAgICAgcmV0dXJuIHJlYWREYXRlVGltZShwYXJzZXIsIG9wdGlvbnMudXNlVVRDLCBjYWxsYmFjayk7XG5cbiAgICAgICAgY2FzZSAnUmVhbCc6XG4gICAgICAgICAgcmV0dXJuIHJlYWRSZWFsKHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgICAgIGNhc2UgJ0Zsb2F0JzpcbiAgICAgICAgICByZXR1cm4gcmVhZEZsb2F0KHBhcnNlciwgY2FsbGJhY2spO1xuXG4gICAgICAgIGNhc2UgJ1NtYWxsTW9uZXknOlxuICAgICAgICAgIHJldHVybiByZWFkU21hbGxNb25leShwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgICAgICBjYXNlICdNb25leSc6XG4gICAgICAgICAgcmV0dXJuIHJlYWRNb25leShwYXJzZXIsIGNhbGxiYWNrKTtcblxuICAgICAgICBjYXNlICdEYXRlJzpcbiAgICAgICAgICByZXR1cm4gcmVhZERhdGUocGFyc2VyLCBvcHRpb25zLnVzZVVUQywgY2FsbGJhY2spO1xuXG4gICAgICAgIGNhc2UgJ1RpbWUnOlxuICAgICAgICAgIHJldHVybiBwYXJzZXIucmVhZFVJbnQ4KChzY2FsZSkgPT4ge1xuICAgICAgICAgICAgcmV0dXJuIHJlYWRUaW1lKHBhcnNlciwgZGF0YUxlbmd0aCwgc2NhbGUsIG9wdGlvbnMudXNlVVRDLCBjYWxsYmFjayk7XG4gICAgICAgICAgfSk7XG5cbiAgICAgICAgY2FzZSAnRGF0ZVRpbWUyJzpcbiAgICAgICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgoc2NhbGUpID0+IHtcbiAgICAgICAgICAgIHJldHVybiByZWFkRGF0ZVRpbWUyKHBhcnNlciwgZGF0YUxlbmd0aCwgc2NhbGUsIG9wdGlvbnMudXNlVVRDLCBjYWxsYmFjayk7XG4gICAgICAgICAgfSk7XG5cbiAgICAgICAgY2FzZSAnRGF0ZVRpbWVPZmZzZXQnOlxuICAgICAgICAgIHJldHVybiBwYXJzZXIucmVhZFVJbnQ4KChzY2FsZSkgPT4ge1xuICAgICAgICAgICAgcmV0dXJuIHJlYWREYXRlVGltZU9mZnNldChwYXJzZXIsIGRhdGFMZW5ndGgsIHNjYWxlLCBjYWxsYmFjayk7XG4gICAgICAgICAgfSk7XG5cbiAgICAgICAgY2FzZSAnVmFyQmluYXJ5JzpcbiAgICAgICAgY2FzZSAnQmluYXJ5JzpcbiAgICAgICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50MTZMRSgoX21heExlbmd0aCkgPT4ge1xuICAgICAgICAgICAgcmVhZEJpbmFyeShwYXJzZXIsIGRhdGFMZW5ndGgsIGNhbGxiYWNrKTtcbiAgICAgICAgICB9KTtcblxuICAgICAgICBjYXNlICdOdW1lcmljTic6XG4gICAgICAgIGNhc2UgJ0RlY2ltYWxOJzpcbiAgICAgICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50OCgocHJlY2lzaW9uKSA9PiB7XG4gICAgICAgICAgICBwYXJzZXIucmVhZFVJbnQ4KChzY2FsZSkgPT4ge1xuICAgICAgICAgICAgICByZWFkTnVtZXJpYyhwYXJzZXIsIGRhdGFMZW5ndGgsIHByZWNpc2lvbiwgc2NhbGUsIGNhbGxiYWNrKTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgIH0pO1xuXG4gICAgICAgIGNhc2UgJ1ZhckNoYXInOlxuICAgICAgICBjYXNlICdDaGFyJzpcbiAgICAgICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50MTZMRSgoX21heExlbmd0aCkgPT4ge1xuICAgICAgICAgICAgcmVhZENvbGxhdGlvbihwYXJzZXIsIChjb2xsYXRpb24pID0+IHtcbiAgICAgICAgICAgICAgcmVhZENoYXJzKHBhcnNlciwgZGF0YUxlbmd0aCwgY29sbGF0aW9uLmNvZGVwYWdlISwgY2FsbGJhY2spO1xuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgfSk7XG5cbiAgICAgICAgY2FzZSAnTlZhckNoYXInOlxuICAgICAgICBjYXNlICdOQ2hhcic6XG4gICAgICAgICAgcmV0dXJuIHBhcnNlci5yZWFkVUludDE2TEUoKF9tYXhMZW5ndGgpID0+IHtcbiAgICAgICAgICAgIHJlYWRDb2xsYXRpb24ocGFyc2VyLCAoX2NvbGxhdGlvbikgPT4ge1xuICAgICAgICAgICAgICByZWFkTkNoYXJzKHBhcnNlciwgZGF0YUxlbmd0aCwgY2FsbGJhY2spO1xuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgfSk7XG5cbiAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ludmFsaWQgdHlwZSEnKTtcbiAgICAgIH1cbiAgICB9KTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJlYWRCaW5hcnkocGFyc2VyOiBQYXJzZXIsIGRhdGFMZW5ndGg6IG51bWJlciwgY2FsbGJhY2s6ICh2YWx1ZTogdW5rbm93bikgPT4gdm9pZCkge1xuICByZXR1cm4gcGFyc2VyLnJlYWRCdWZmZXIoZGF0YUxlbmd0aCwgY2FsbGJhY2spO1xufVxuXG5mdW5jdGlvbiByZWFkQ2hhcnMocGFyc2VyOiBQYXJzZXIsIGRhdGFMZW5ndGg6IG51bWJlciwgY29kZXBhZ2U6IHN0cmluZywgY2FsbGJhY2s6ICh2YWx1ZTogdW5rbm93bikgPT4gdm9pZCkge1xuICBpZiAoY29kZXBhZ2UgPT0gbnVsbCkge1xuICAgIGNvZGVwYWdlID0gREVGQVVMVF9FTkNPRElORztcbiAgfVxuXG4gIHJldHVybiBwYXJzZXIucmVhZEJ1ZmZlcihkYXRhTGVuZ3RoLCAoZGF0YSkgPT4ge1xuICAgIGNhbGxiYWNrKGljb252LmRlY29kZShkYXRhLCBjb2RlcGFnZSkpO1xuICB9KTtcbn1cblxuZnVuY3Rpb24gcmVhZE5DaGFycyhwYXJzZXI6IFBhcnNlciwgZGF0YUxlbmd0aDogbnVtYmVyLCBjYWxsYmFjazogKHZhbHVlOiB1bmtub3duKSA9PiB2b2lkKSB7XG4gIHBhcnNlci5yZWFkQnVmZmVyKGRhdGFMZW5ndGgsIChkYXRhKSA9PiB7XG4gICAgY2FsbGJhY2soZGF0YS50b1N0cmluZygndWNzMicpKTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJlYWRNYXhCaW5hcnkocGFyc2VyOiBQYXJzZXIsIGNhbGxiYWNrOiAodmFsdWU6IHVua25vd24pID0+IHZvaWQpIHtcbiAgcmV0dXJuIHJlYWRNYXgocGFyc2VyLCBjYWxsYmFjayk7XG59XG5cbmZ1bmN0aW9uIHJlYWRNYXhDaGFycyhwYXJzZXI6IFBhcnNlciwgY29kZXBhZ2U6IHN0cmluZywgY2FsbGJhY2s6ICh2YWx1ZTogdW5rbm93bikgPT4gdm9pZCkge1xuICBpZiAoY29kZXBhZ2UgPT0gbnVsbCkge1xuICAgIGNvZGVwYWdlID0gREVGQVVMVF9FTkNPRElORztcbiAgfVxuXG4gIHJlYWRNYXgocGFyc2VyLCAoZGF0YSkgPT4ge1xuICAgIGlmIChkYXRhKSB7XG4gICAgICBjYWxsYmFjayhpY29udi5kZWNvZGUoZGF0YSwgY29kZXBhZ2UpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY2FsbGJhY2sobnVsbCk7XG4gICAgfVxuICB9KTtcbn1cblxuZnVuY3Rpb24gcmVhZE1heE5DaGFycyhwYXJzZXI6IFBhcnNlciwgY2FsbGJhY2s6ICh2YWx1ZTogc3RyaW5nIHwgbnVsbCkgPT4gdm9pZCkge1xuICByZWFkTWF4KHBhcnNlciwgKGRhdGEpID0+IHtcbiAgICBpZiAoZGF0YSkge1xuICAgICAgY2FsbGJhY2soZGF0YS50b1N0cmluZygndWNzMicpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY2FsbGJhY2sobnVsbCk7XG4gICAgfVxuICB9KTtcbn1cblxuZnVuY3Rpb24gcmVhZE1heChwYXJzZXI6IFBhcnNlciwgY2FsbGJhY2s6ICh2YWx1ZTogbnVsbCB8IEJ1ZmZlcikgPT4gdm9pZCkge1xuICBwYXJzZXIucmVhZEJ1ZmZlcig4LCAodHlwZSkgPT4ge1xuICAgIGlmICh0eXBlLmVxdWFscyhQTFBfTlVMTCkpIHtcbiAgICAgIHJldHVybiBjYWxsYmFjayhudWxsKTtcbiAgICB9IGVsc2UgaWYgKHR5cGUuZXF1YWxzKFVOS05PV05fUExQX0xFTikpIHtcbiAgICAgIHJldHVybiByZWFkTWF4VW5rbm93bkxlbmd0aChwYXJzZXIsIGNhbGxiYWNrKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgbG93ID0gdHlwZS5yZWFkVUludDMyTEUoMCk7XG4gICAgICBjb25zdCBoaWdoID0gdHlwZS5yZWFkVUludDMyTEUoNCk7XG5cbiAgICAgIGlmIChoaWdoID49ICgyIDw8ICg1MyAtIDMyKSkpIHtcbiAgICAgICAgY29uc29sZS53YXJuKCdSZWFkIFVJbnQ2NExFID4gNTMgYml0cyA6IGhpZ2g9JyArIGhpZ2ggKyAnLCBsb3c9JyArIGxvdyk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGV4cGVjdGVkTGVuZ3RoID0gbG93ICsgKDB4MTAwMDAwMDAwICogaGlnaCk7XG4gICAgICByZXR1cm4gcmVhZE1heEtub3duTGVuZ3RoKHBhcnNlciwgZXhwZWN0ZWRMZW5ndGgsIGNhbGxiYWNrKTtcbiAgICB9XG4gIH0pO1xufVxuXG5mdW5jdGlvbiByZWFkTWF4S25vd25MZW5ndGgocGFyc2VyOiBQYXJzZXIsIHRvdGFsTGVuZ3RoOiBudW1iZXIsIGNhbGxiYWNrOiAodmFsdWU6IG51bGwgfCBCdWZmZXIpID0+IHZvaWQpIHtcbiAgY29uc3QgZGF0YSA9IEJ1ZmZlci5hbGxvYyh0b3RhbExlbmd0aCwgMCk7XG5cbiAgbGV0IG9mZnNldCA9IDA7XG4gIGZ1bmN0aW9uIG5leHQoZG9uZTogYW55KSB7XG4gICAgcGFyc2VyLnJlYWRVSW50MzJMRSgoY2h1bmtMZW5ndGgpID0+IHtcbiAgICAgIGlmICghY2h1bmtMZW5ndGgpIHtcbiAgICAgICAgcmV0dXJuIGRvbmUoKTtcbiAgICAgIH1cblxuICAgICAgcGFyc2VyLnJlYWRCdWZmZXIoY2h1bmtMZW5ndGgsIChjaHVuaykgPT4ge1xuICAgICAgICBjaHVuay5jb3B5KGRhdGEsIG9mZnNldCk7XG4gICAgICAgIG9mZnNldCArPSBjaHVua0xlbmd0aDtcblxuICAgICAgICBuZXh0KGRvbmUpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH1cblxuICBuZXh0KCgpID0+IHtcbiAgICBpZiAob2Zmc2V0ICE9PSB0b3RhbExlbmd0aCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdQYXJ0aWFsbHkgTGVuZ3RoLXByZWZpeGVkIEJ5dGVzIHVubWF0Y2hlZCBsZW5ndGhzIDogZXhwZWN0ZWQgJyArIHRvdGFsTGVuZ3RoICsgJywgYnV0IGdvdCAnICsgb2Zmc2V0ICsgJyBieXRlcycpO1xuICAgIH1cblxuICAgIGNhbGxiYWNrKGRhdGEpO1xuICB9KTtcbn1cblxuZnVuY3Rpb24gcmVhZE1heFVua25vd25MZW5ndGgocGFyc2VyOiBQYXJzZXIsIGNhbGxiYWNrOiAodmFsdWU6IG51bGwgfCBCdWZmZXIpID0+IHZvaWQpIHtcbiAgY29uc3QgY2h1bmtzOiBCdWZmZXJbXSA9IFtdO1xuXG4gIGxldCBsZW5ndGggPSAwO1xuICBmdW5jdGlvbiBuZXh0KGRvbmU6IGFueSkge1xuICAgIHBhcnNlci5yZWFkVUludDMyTEUoKGNodW5rTGVuZ3RoKSA9PiB7XG4gICAgICBpZiAoIWNodW5rTGVuZ3RoKSB7XG4gICAgICAgIHJldHVybiBkb25lKCk7XG4gICAgICB9XG5cbiAgICAgIHBhcnNlci5yZWFkQnVmZmVyKGNodW5rTGVuZ3RoLCAoY2h1bmspID0+IHtcbiAgICAgICAgY2h1bmtzLnB1c2goY2h1bmspO1xuICAgICAgICBsZW5ndGggKz0gY2h1bmtMZW5ndGg7XG5cbiAgICAgICAgbmV4dChkb25lKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9XG5cbiAgbmV4dCgoKSA9PiB7XG4gICAgY2FsbGJhY2soQnVmZmVyLmNvbmNhdChjaHVua3MsIGxlbmd0aCkpO1xuICB9KTtcbn1cblxuZnVuY3Rpb24gcmVhZFNtYWxsRGF0ZVRpbWUocGFyc2VyOiBQYXJzZXIsIHVzZVVUQzogYm9vbGVhbiwgY2FsbGJhY2s6ICh2YWx1ZTogRGF0ZSkgPT4gdm9pZCkge1xuICBwYXJzZXIucmVhZFVJbnQxNkxFKChkYXlzKSA9PiB7XG4gICAgcGFyc2VyLnJlYWRVSW50MTZMRSgobWludXRlcykgPT4ge1xuICAgICAgbGV0IHZhbHVlO1xuICAgICAgaWYgKHVzZVVUQykge1xuICAgICAgICB2YWx1ZSA9IG5ldyBEYXRlKERhdGUuVVRDKDE5MDAsIDAsIDEgKyBkYXlzLCAwLCBtaW51dGVzKSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB2YWx1ZSA9IG5ldyBEYXRlKDE5MDAsIDAsIDEgKyBkYXlzLCAwLCBtaW51dGVzKTtcbiAgICAgIH1cbiAgICAgIGNhbGxiYWNrKHZhbHVlKTtcbiAgICB9KTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJlYWREYXRlVGltZShwYXJzZXI6IFBhcnNlciwgdXNlVVRDOiBib29sZWFuLCBjYWxsYmFjazogKHZhbHVlOiBEYXRlKSA9PiB2b2lkKSB7XG4gIHBhcnNlci5yZWFkSW50MzJMRSgoZGF5cykgPT4ge1xuICAgIHBhcnNlci5yZWFkVUludDMyTEUoKHRocmVlSHVuZHJlZHRoc09mU2Vjb25kKSA9PiB7XG4gICAgICBjb25zdCBtaWxsaXNlY29uZHMgPSBNYXRoLnJvdW5kKHRocmVlSHVuZHJlZHRoc09mU2Vjb25kICogVEhSRUVfQU5EX0FfVEhJUkQpO1xuXG4gICAgICBsZXQgdmFsdWU7XG4gICAgICBpZiAodXNlVVRDKSB7XG4gICAgICAgIHZhbHVlID0gbmV3IERhdGUoRGF0ZS5VVEMoMTkwMCwgMCwgMSArIGRheXMsIDAsIDAsIDAsIG1pbGxpc2Vjb25kcykpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdmFsdWUgPSBuZXcgRGF0ZSgxOTAwLCAwLCAxICsgZGF5cywgMCwgMCwgMCwgbWlsbGlzZWNvbmRzKTtcbiAgICAgIH1cblxuICAgICAgY2FsbGJhY2sodmFsdWUpO1xuICAgIH0pO1xuICB9KTtcbn1cblxuaW50ZXJmYWNlIERhdGVXaXRoTmFub3NlY29uZHNEZWx0YSBleHRlbmRzIERhdGUge1xuICBuYW5vc2Vjb25kc0RlbHRhOiBudW1iZXI7XG59XG5cbmZ1bmN0aW9uIHJlYWRUaW1lKHBhcnNlcjogUGFyc2VyLCBkYXRhTGVuZ3RoOiBudW1iZXIsIHNjYWxlOiBudW1iZXIsIHVzZVVUQzogYm9vbGVhbiwgY2FsbGJhY2s6ICh2YWx1ZTogRGF0ZVdpdGhOYW5vc2Vjb25kc0RlbHRhKSA9PiB2b2lkKSB7XG4gIGxldCByZWFkVmFsdWU6IGFueTtcbiAgc3dpdGNoIChkYXRhTGVuZ3RoKSB7XG4gICAgY2FzZSAzOlxuICAgICAgcmVhZFZhbHVlID0gcGFyc2VyLnJlYWRVSW50MjRMRTtcbiAgICAgIGJyZWFrO1xuICAgIGNhc2UgNDpcbiAgICAgIHJlYWRWYWx1ZSA9IHBhcnNlci5yZWFkVUludDMyTEU7XG4gICAgICBicmVhaztcbiAgICBjYXNlIDU6XG4gICAgICByZWFkVmFsdWUgPSBwYXJzZXIucmVhZFVJbnQ0MExFO1xuICB9XG5cbiAgcmVhZFZhbHVlIS5jYWxsKHBhcnNlciwgKHZhbHVlOiBudW1iZXIpID0+IHtcbiAgICBpZiAoc2NhbGUgPCA3KSB7XG4gICAgICBmb3IgKGxldCBpID0gc2NhbGU7IGkgPCA3OyBpKyspIHtcbiAgICAgICAgdmFsdWUgKj0gMTA7XG4gICAgICB9XG4gICAgfVxuXG4gICAgbGV0IGRhdGU7XG4gICAgaWYgKHVzZVVUQykge1xuICAgICAgZGF0ZSA9IG5ldyBEYXRlKERhdGUuVVRDKDE5NzAsIDAsIDEsIDAsIDAsIDAsIHZhbHVlIC8gMTAwMDApKSBhcyBEYXRlV2l0aE5hbm9zZWNvbmRzRGVsdGE7XG4gICAgfSBlbHNlIHtcbiAgICAgIGRhdGUgPSBuZXcgRGF0ZSgxOTcwLCAwLCAxLCAwLCAwLCAwLCB2YWx1ZSAvIDEwMDAwKSBhcyBEYXRlV2l0aE5hbm9zZWNvbmRzRGVsdGE7XG4gICAgfVxuICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eShkYXRlLCAnbmFub3NlY29uZHNEZWx0YScsIHtcbiAgICAgIGVudW1lcmFibGU6IGZhbHNlLFxuICAgICAgdmFsdWU6ICh2YWx1ZSAlIDEwMDAwKSAvIE1hdGgucG93KDEwLCA3KVxuICAgIH0pO1xuICAgIGNhbGxiYWNrKGRhdGUpO1xuICB9KTtcbn1cblxuZnVuY3Rpb24gcmVhZERhdGUocGFyc2VyOiBQYXJzZXIsIHVzZVVUQzogYm9vbGVhbiwgY2FsbGJhY2s6ICh2YWx1ZTogRGF0ZSkgPT4gdm9pZCkge1xuICBwYXJzZXIucmVhZFVJbnQyNExFKChkYXlzKSA9PiB7XG4gICAgaWYgKHVzZVVUQykge1xuICAgICAgY2FsbGJhY2sobmV3IERhdGUoRGF0ZS5VVEMoMjAwMCwgMCwgZGF5cyAtIDczMDExOCkpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY2FsbGJhY2sobmV3IERhdGUoMjAwMCwgMCwgZGF5cyAtIDczMDExOCkpO1xuICAgIH1cbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJlYWREYXRlVGltZTIocGFyc2VyOiBQYXJzZXIsIGRhdGFMZW5ndGg6IG51bWJlciwgc2NhbGU6IG51bWJlciwgdXNlVVRDOiBib29sZWFuLCBjYWxsYmFjazogKHZhbHVlOiBEYXRlV2l0aE5hbm9zZWNvbmRzRGVsdGEpID0+IHZvaWQpIHtcbiAgcmVhZFRpbWUocGFyc2VyLCBkYXRhTGVuZ3RoIC0gMywgc2NhbGUsIHVzZVVUQywgKHRpbWUpID0+IHsgLy8gVE9ETzogJ2lucHV0JyBpcyAndGltZScsIGJ1dCBUeXBlU2NyaXB0IGNhbm5vdCBmaW5kIFwidGltZS5uYW5vc2Vjb25kc0RlbHRhXCI7XG4gICAgcGFyc2VyLnJlYWRVSW50MjRMRSgoZGF5cykgPT4ge1xuICAgICAgbGV0IGRhdGU7XG4gICAgICBpZiAodXNlVVRDKSB7XG4gICAgICAgIGRhdGUgPSBuZXcgRGF0ZShEYXRlLlVUQygyMDAwLCAwLCBkYXlzIC0gNzMwMTE4LCAwLCAwLCAwLCArdGltZSkpIGFzIERhdGVXaXRoTmFub3NlY29uZHNEZWx0YTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGRhdGUgPSBuZXcgRGF0ZSgyMDAwLCAwLCBkYXlzIC0gNzMwMTE4LCB0aW1lLmdldEhvdXJzKCksIHRpbWUuZ2V0TWludXRlcygpLCB0aW1lLmdldFNlY29uZHMoKSwgdGltZS5nZXRNaWxsaXNlY29uZHMoKSkgYXMgRGF0ZVdpdGhOYW5vc2Vjb25kc0RlbHRhO1xuICAgICAgfVxuICAgICAgT2JqZWN0LmRlZmluZVByb3BlcnR5KGRhdGUsICduYW5vc2Vjb25kc0RlbHRhJywge1xuICAgICAgICBlbnVtZXJhYmxlOiBmYWxzZSxcbiAgICAgICAgdmFsdWU6IHRpbWUubmFub3NlY29uZHNEZWx0YVxuICAgICAgfSk7XG4gICAgICBjYWxsYmFjayhkYXRlKTtcbiAgICB9KTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHJlYWREYXRlVGltZU9mZnNldChwYXJzZXI6IFBhcnNlciwgZGF0YUxlbmd0aDogbnVtYmVyLCBzY2FsZTogbnVtYmVyLCBjYWxsYmFjazogKHZhbHVlOiBEYXRlV2l0aE5hbm9zZWNvbmRzRGVsdGEpID0+IHZvaWQpIHtcbiAgcmVhZFRpbWUocGFyc2VyLCBkYXRhTGVuZ3RoIC0gNSwgc2NhbGUsIHRydWUsICh0aW1lKSA9PiB7XG4gICAgcGFyc2VyLnJlYWRVSW50MjRMRSgoZGF5cykgPT4ge1xuICAgICAgLy8gb2Zmc2V0XG4gICAgICBwYXJzZXIucmVhZEludDE2TEUoKCkgPT4ge1xuICAgICAgICBjb25zdCBkYXRlID0gbmV3IERhdGUoRGF0ZS5VVEMoMjAwMCwgMCwgZGF5cyAtIDczMDExOCwgMCwgMCwgMCwgK3RpbWUpKSBhcyBEYXRlV2l0aE5hbm9zZWNvbmRzRGVsdGE7XG4gICAgICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eShkYXRlLCAnbmFub3NlY29uZHNEZWx0YScsIHtcbiAgICAgICAgICBlbnVtZXJhYmxlOiBmYWxzZSxcbiAgICAgICAgICB2YWx1ZTogdGltZS5uYW5vc2Vjb25kc0RlbHRhXG4gICAgICAgIH0pO1xuICAgICAgICBjYWxsYmFjayhkYXRlKTtcbiAgICAgIH0pO1xuICAgIH0pO1xuICB9KTtcbn1cblxuZXhwb3J0IGRlZmF1bHQgdmFsdWVQYXJzZTtcbm1vZHVsZS5leHBvcnRzID0gdmFsdWVQYXJzZTtcbiJdfQ==