"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _jsbi = _interopRequireDefault(require("jsbi"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const SHIFT_LEFT_32 = (1 << 16) * (1 << 16);
const SHIFT_RIGHT_32 = 1 / SHIFT_LEFT_32;
const UNKNOWN_PLP_LEN = Buffer.from([0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff]);
const ZERO_LENGTH_BUFFER = Buffer.alloc(0);

/**
  A Buffer-like class that tracks position.

  As values are written, the position advances by the size of the written data.
  When writing, automatically allocates new buffers if there's not enough space.
 */
class WritableTrackingBuffer {
  constructor(initialSize, encoding, doubleSizeGrowth) {
    this.initialSize = void 0;
    this.encoding = void 0;
    this.doubleSizeGrowth = void 0;
    this.buffer = void 0;
    this.compositeBuffer = void 0;
    this.position = void 0;
    this.initialSize = initialSize;
    this.encoding = encoding || 'ucs2';
    this.doubleSizeGrowth = doubleSizeGrowth || false;
    this.buffer = Buffer.alloc(this.initialSize, 0);
    this.compositeBuffer = ZERO_LENGTH_BUFFER;
    this.position = 0;
  }

  get data() {
    this.newBuffer(0);
    return this.compositeBuffer;
  }

  copyFrom(buffer) {
    const length = buffer.length;
    this.makeRoomFor(length);
    buffer.copy(this.buffer, this.position);
    this.position += length;
  }

  makeRoomFor(requiredLength) {
    if (this.buffer.length - this.position < requiredLength) {
      if (this.doubleSizeGrowth) {
        let size = Math.max(128, this.buffer.length * 2);

        while (size < requiredLength) {
          size *= 2;
        }

        this.newBuffer(size);
      } else {
        this.newBuffer(requiredLength);
      }
    }
  }

  newBuffer(size) {
    const buffer = this.buffer.slice(0, this.position);
    this.compositeBuffer = Buffer.concat([this.compositeBuffer, buffer]);
    this.buffer = size === 0 ? ZERO_LENGTH_BUFFER : Buffer.alloc(size, 0);
    this.position = 0;
  }

  writeUInt8(value) {
    const length = 1;
    this.makeRoomFor(length);
    this.buffer.writeUInt8(value, this.position);
    this.position += length;
  }

  writeUInt16LE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeUInt16LE(value, this.position);
    this.position += length;
  }

  writeUShort(value) {
    this.writeUInt16LE(value);
  }

  writeUInt16BE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeUInt16BE(value, this.position);
    this.position += length;
  }

  writeUInt24LE(value) {
    const length = 3;
    this.makeRoomFor(length);
    this.buffer[this.position + 2] = value >>> 16 & 0xff;
    this.buffer[this.position + 1] = value >>> 8 & 0xff;
    this.buffer[this.position] = value & 0xff;
    this.position += length;
  }

  writeUInt32LE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeUInt32LE(value, this.position);
    this.position += length;
  }

  writeBigInt64LE(value) {
    this.writeBigU_Int64LE(value);
  }

  writeBigU_Int64LE(value) {
    this.makeRoomFor(8);

    let lo = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(value, _jsbi.default.BigInt(0xffffffff)));

    this.buffer[this.position++] = lo;
    lo = lo >> 8;
    this.buffer[this.position++] = lo;
    lo = lo >> 8;
    this.buffer[this.position++] = lo;
    lo = lo >> 8;
    this.buffer[this.position++] = lo;

    let hi = _jsbi.default.toNumber(_jsbi.default.bitwiseAnd(_jsbi.default.signedRightShift(value, _jsbi.default.BigInt(32)), _jsbi.default.BigInt(0xffffffff)));

    this.buffer[this.position++] = hi;
    hi = hi >> 8;
    this.buffer[this.position++] = hi;
    hi = hi >> 8;
    this.buffer[this.position++] = hi;
    hi = hi >> 8;
    this.buffer[this.position++] = hi;
  }

  writeInt64LE(value) {
    this.writeBigInt64LE(_jsbi.default.BigInt(value));
  }

  writeUInt32BE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeUInt32BE(value, this.position);
    this.position += length;
  }

  writeUInt40LE(value) {
    // inspired by https://github.com/dpw/node-buffer-more-ints
    this.writeInt32LE(value & -1);
    this.writeUInt8(Math.floor(value * SHIFT_RIGHT_32));
  }

  writeUInt64LE(value) {
    this.writeBigUInt64LE(_jsbi.default.BigInt(value));
  }

  writeBigUInt64LE(value) {
    this.writeBigU_Int64LE(value);
  }

  writeInt8(value) {
    const length = 1;
    this.makeRoomFor(length);
    this.buffer.writeInt8(value, this.position);
    this.position += length;
  }

  writeInt16LE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeInt16LE(value, this.position);
    this.position += length;
  }

  writeInt16BE(value) {
    const length = 2;
    this.makeRoomFor(length);
    this.buffer.writeInt16BE(value, this.position);
    this.position += length;
  }

  writeInt32LE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeInt32LE(value, this.position);
    this.position += length;
  }

  writeInt32BE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeInt32BE(value, this.position);
    this.position += length;
  }

  writeFloatLE(value) {
    const length = 4;
    this.makeRoomFor(length);
    this.buffer.writeFloatLE(value, this.position);
    this.position += length;
  }

  writeDoubleLE(value) {
    const length = 8;
    this.makeRoomFor(length);
    this.buffer.writeDoubleLE(value, this.position);
    this.position += length;
  }

  writeString(value, encoding) {
    if (encoding == null) {
      encoding = this.encoding;
    }

    const length = Buffer.byteLength(value, encoding);
    this.makeRoomFor(length); // $FlowFixMe https://github.com/facebook/flow/pull/5398

    this.buffer.write(value, this.position, encoding);
    this.position += length;
  }

  writeBVarchar(value, encoding) {
    this.writeUInt8(value.length);
    this.writeString(value, encoding);
  }

  writeUsVarchar(value, encoding) {
    this.writeUInt16LE(value.length);
    this.writeString(value, encoding);
  } // TODO: Figure out what types are passed in other than `Buffer`


  writeUsVarbyte(value, encoding) {
    if (encoding == null) {
      encoding = this.encoding;
    }

    let length;

    if (value instanceof Buffer) {
      length = value.length;
    } else {
      value = value.toString();
      length = Buffer.byteLength(value, encoding);
    }

    this.writeUInt16LE(length);

    if (value instanceof Buffer) {
      this.writeBuffer(value);
    } else {
      this.makeRoomFor(length); // $FlowFixMe https://github.com/facebook/flow/pull/5398

      this.buffer.write(value, this.position, encoding);
      this.position += length;
    }
  }

  writePLPBody(value, encoding) {
    if (encoding == null) {
      encoding = this.encoding;
    }

    let length;

    if (value instanceof Buffer) {
      length = value.length;
    } else {
      value = value.toString();
      length = Buffer.byteLength(value, encoding);
    } // Length of all chunks.
    // this.writeUInt64LE(length);
    // unknown seems to work better here - might revisit later.


    this.writeBuffer(UNKNOWN_PLP_LEN); // In the UNKNOWN_PLP_LEN case, the data is represented as a series of zero or more chunks.

    if (length > 0) {
      // One chunk.
      this.writeUInt32LE(length);

      if (value instanceof Buffer) {
        this.writeBuffer(value);
      } else {
        this.makeRoomFor(length);
        this.buffer.write(value, this.position, encoding);
        this.position += length;
      }
    } // PLP_TERMINATOR (no more chunks).


    this.writeUInt32LE(0);
  }

  writeBuffer(value) {
    const length = value.length;
    this.makeRoomFor(length);
    value.copy(this.buffer, this.position);
    this.position += length;
  }

  writeMoney(value) {
    this.writeInt32LE(Math.floor(value * SHIFT_RIGHT_32));
    this.writeInt32LE(value & -1);
  }

}

var _default = WritableTrackingBuffer;
exports.default = _default;
module.exports = WritableTrackingBuffer;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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