"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AttentionTokenHandler = exports.RequestTokenHandler = exports.Login7TokenHandler = exports.InitialSqlTokenHandler = exports.TokenHandler = exports.UnexpectedTokenError = void 0;

var _errors = require("../errors");

class UnexpectedTokenError extends Error {
  constructor(handler, token) {
    super('Unexpected token `' + token.name + '` in `' + handler.constructor.name + '`');
  }

}

exports.UnexpectedTokenError = UnexpectedTokenError;

class TokenHandler {
  onInfoMessage(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onErrorMessage(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onSSPI(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onDatabaseChange(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onLanguageChange(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onCharsetChange(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onSqlCollationChange(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onRoutingChange(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onPacketSizeChange(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onResetConnection(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onBeginTransaction(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onCommitTransaction(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onRollbackTransaction(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onFedAuthInfo(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onFeatureExtAck(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onLoginAck(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onColMetadata(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onOrder(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onRow(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onReturnStatus(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onReturnValue(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onDoneProc(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onDoneInProc(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onDone(token) {
    throw new UnexpectedTokenError(this, token);
  }

  onDatabaseMirroringPartner(token) {
    throw new UnexpectedTokenError(this, token);
  }

}
/**
 * A handler for tokens received in the response message to the initial SQL Batch request
 * that sets up different connection settings.
 */


exports.TokenHandler = TokenHandler;

class InitialSqlTokenHandler extends TokenHandler {
  constructor(connection) {
    super();
    this.connection = void 0;
    this.connection = connection;
  }

  onInfoMessage(token) {
    this.connection.emit('infoMessage', token);
  }

  onErrorMessage(token) {
    this.connection.emit('errorMessage', token);
  }

  onDatabaseChange(token) {
    this.connection.emit('databaseChange', token.newValue);
  }

  onLanguageChange(token) {
    this.connection.emit('languageChange', token.newValue);
  }

  onCharsetChange(token) {
    this.connection.emit('charsetChange', token.newValue);
  }

  onSqlCollationChange(token) {
    this.connection.databaseCollation = token.newValue;
  }

  onPacketSizeChange(token) {
    this.connection.messageIo.packetSize(token.newValue);
  }

  onBeginTransaction(token) {
    this.connection.transactionDescriptors.push(token.newValue);
    this.connection.inTransaction = true;
  }

  onCommitTransaction(token) {
    this.connection.transactionDescriptors.length = 1;
    this.connection.inTransaction = false;
  }

  onRollbackTransaction(token) {
    this.connection.transactionDescriptors.length = 1; // An outermost transaction was rolled back. Reset the transaction counter

    this.connection.inTransaction = false;
    this.connection.emit('rollbackTransaction');
  }

  onColMetadata(token) {
    this.connection.emit('error', new Error("Received 'columnMetadata' when no sqlRequest is in progress"));
    this.connection.close();
  }

  onOrder(token) {
    this.connection.emit('error', new Error("Received 'order' when no sqlRequest is in progress"));
    this.connection.close();
  }

  onRow(token) {
    this.connection.emit('error', new Error("Received 'row' when no sqlRequest is in progress"));
    this.connection.close();
  }

  onReturnStatus(token) {// Do nothing
  }

  onReturnValue(token) {// Do nothing
  }

  onDoneProc(token) {// Do nothing
  }

  onDoneInProc(token) {// Do nothing
  }

  onDone(token) {// Do nothing
  }

  onResetConnection(token) {
    this.connection.emit('resetConnection');
  }

}
/**
 * A handler for tokens received in the response message to a Login7 message.
 */


exports.InitialSqlTokenHandler = InitialSqlTokenHandler;

class Login7TokenHandler extends TokenHandler {
  constructor(connection) {
    super();
    this.connection = void 0;
    this.fedAuthInfoToken = void 0;
    this.routingData = void 0;
    this.loginAckReceived = false;
    this.connection = connection;
  }

  onInfoMessage(token) {
    this.connection.emit('infoMessage', token);
  }

  onErrorMessage(token) {
    this.connection.emit('errorMessage', token);
    const error = new _errors.ConnectionError(token.message, 'ELOGIN');
    const isLoginErrorTransient = this.connection.transientErrorLookup.isTransientError(token.number);

    if (isLoginErrorTransient && this.connection.curTransientRetryCount !== this.connection.config.options.maxRetriesOnTransientErrors) {
      error.isTransient = true;
    }

    this.connection.loginError = error;
  }

  onSSPI(token) {
    if (token.ntlmpacket) {
      this.connection.ntlmpacket = token.ntlmpacket;
      this.connection.ntlmpacketBuffer = token.ntlmpacketBuffer;
    }
  }

  onDatabaseChange(token) {
    this.connection.emit('databaseChange', token.newValue);
  }

  onLanguageChange(token) {
    this.connection.emit('languageChange', token.newValue);
  }

  onCharsetChange(token) {
    this.connection.emit('charsetChange', token.newValue);
  }

  onSqlCollationChange(token) {
    this.connection.databaseCollation = token.newValue;
  }

  onFedAuthInfo(token) {
    this.fedAuthInfoToken = token;
  }

  onFeatureExtAck(token) {
    const {
      authentication
    } = this.connection.config;

    if (authentication.type === 'azure-active-directory-password' || authentication.type === 'azure-active-directory-access-token' || authentication.type === 'azure-active-directory-msi-vm' || authentication.type === 'azure-active-directory-msi-app-service' || authentication.type === 'azure-active-directory-service-principal-secret') {
      if (token.fedAuth === undefined) {
        this.connection.loginError = new _errors.ConnectionError('Did not receive Active Directory authentication acknowledgement');
      } else if (token.fedAuth.length !== 0) {
        this.connection.loginError = new _errors.ConnectionError(`Active Directory authentication acknowledgment for ${authentication.type} authentication method includes extra data`);
      }
    } else if (token.fedAuth === undefined && token.utf8Support === undefined) {
      this.connection.loginError = new _errors.ConnectionError('Received acknowledgement for unknown feature');
    } else if (token.fedAuth) {
      this.connection.loginError = new _errors.ConnectionError('Did not request Active Directory authentication, but received the acknowledgment');
    }
  }

  onLoginAck(token) {
    if (!token.tdsVersion) {
      // unsupported TDS version
      this.connection.loginError = new _errors.ConnectionError('Server responded with unknown TDS version.', 'ETDS');
      return;
    }

    if (!token.interface) {
      // unsupported interface
      this.connection.loginError = new _errors.ConnectionError('Server responded with unsupported interface.', 'EINTERFACENOTSUPP');
      return;
    } // use negotiated version


    this.connection.config.options.tdsVersion = token.tdsVersion;
    this.loginAckReceived = true;
  }

  onRoutingChange(token) {
    // Removes instance name attached to the redirect url. E.g., redirect.db.net\instance1 --> redirect.db.net
    const [server] = token.newValue.server.split('\\');
    this.routingData = {
      server,
      port: token.newValue.port
    };
  }

  onDoneInProc(token) {// Do nothing
  }

  onDone(token) {// Do nothing
  }

  onPacketSizeChange(token) {
    this.connection.messageIo.packetSize(token.newValue);
  }

  onDatabaseMirroringPartner(token) {// Do nothing
  }

}
/**
 * A handler for tokens received in the response message to a RPC Request,
 * a SQL Batch Request, a Bulk Load BCP Request or a Transaction Manager Request.
 */


exports.Login7TokenHandler = Login7TokenHandler;

class RequestTokenHandler extends TokenHandler {
  constructor(connection, request) {
    super();
    this.connection = void 0;
    this.request = void 0;
    this.connection = connection;
    this.request = request;
  }

  onInfoMessage(token) {
    this.connection.emit('infoMessage', token);
  }

  onErrorMessage(token) {
    this.connection.emit('errorMessage', token);

    if (!this.request.canceled) {
      const error = new _errors.RequestError(token.message, 'EREQUEST');
      error.number = token.number;
      error.state = token.state;
      error.class = token.class;
      error.serverName = token.serverName;
      error.procName = token.procName;
      error.lineNumber = token.lineNumber;
      this.request.error = error;
    }
  }

  onDatabaseChange(token) {
    this.connection.emit('databaseChange', token.newValue);
  }

  onLanguageChange(token) {
    this.connection.emit('languageChange', token.newValue);
  }

  onCharsetChange(token) {
    this.connection.emit('charsetChange', token.newValue);
  }

  onSqlCollationChange(token) {
    this.connection.databaseCollation = token.newValue;
  }

  onPacketSizeChange(token) {
    this.connection.messageIo.packetSize(token.newValue);
  }

  onBeginTransaction(token) {
    this.connection.transactionDescriptors.push(token.newValue);
    this.connection.inTransaction = true;
  }

  onCommitTransaction(token) {
    this.connection.transactionDescriptors.length = 1;
    this.connection.inTransaction = false;
  }

  onRollbackTransaction(token) {
    this.connection.transactionDescriptors.length = 1; // An outermost transaction was rolled back. Reset the transaction counter

    this.connection.inTransaction = false;
    this.connection.emit('rollbackTransaction');
  }

  onColMetadata(token) {
    if (!this.request.canceled) {
      if (this.connection.config.options.useColumnNames) {
        const columns = Object.create(null);

        for (let j = 0, len = token.columns.length; j < len; j++) {
          const col = token.columns[j];

          if (columns[col.colName] == null) {
            columns[col.colName] = col;
          }
        }

        this.request.emit('columnMetadata', columns);
      } else {
        this.request.emit('columnMetadata', token.columns);
      }
    }
  }

  onOrder(token) {
    if (!this.request.canceled) {
      this.request.emit('order', token.orderColumns);
    }
  }

  onRow(token) {
    if (!this.request.canceled) {
      if (this.connection.config.options.rowCollectionOnRequestCompletion) {
        this.request.rows.push(token.columns);
      }

      if (this.connection.config.options.rowCollectionOnDone) {
        this.request.rst.push(token.columns);
      }

      this.request.emit('row', token.columns);
    }
  }

  onReturnStatus(token) {
    if (!this.request.canceled) {
      // Keep value for passing in 'doneProc' event.
      this.connection.procReturnStatusValue = token.value;
    }
  }

  onReturnValue(token) {
    if (!this.request.canceled) {
      this.request.emit('returnValue', token.paramName, token.value, token.metadata);
    }
  }

  onDoneProc(token) {
    if (!this.request.canceled) {
      if (token.sqlError && !this.request.error) {
        // check if the DONE_ERROR flags was set, but an ERROR token was not sent.
        this.request.error = new _errors.RequestError('An unknown error has occurred.', 'UNKNOWN');
      }

      this.request.emit('doneProc', token.rowCount, token.more, this.connection.procReturnStatusValue, this.request.rst);
      this.connection.procReturnStatusValue = undefined;

      if (token.rowCount !== undefined) {
        this.request.rowCount += token.rowCount;
      }

      if (this.connection.config.options.rowCollectionOnDone) {
        this.request.rst = [];
      }
    }
  }

  onDoneInProc(token) {
    if (!this.request.canceled) {
      this.request.emit('doneInProc', token.rowCount, token.more, this.request.rst);

      if (token.rowCount !== undefined) {
        this.request.rowCount += token.rowCount;
      }

      if (this.connection.config.options.rowCollectionOnDone) {
        this.request.rst = [];
      }
    }
  }

  onDone(token) {
    if (!this.request.canceled) {
      if (token.sqlError && !this.request.error) {
        // check if the DONE_ERROR flags was set, but an ERROR token was not sent.
        this.request.error = new _errors.RequestError('An unknown error has occurred.', 'UNKNOWN');
      }

      this.request.emit('done', token.rowCount, token.more, this.request.rst);

      if (token.rowCount !== undefined) {
        this.request.rowCount += token.rowCount;
      }

      if (this.connection.config.options.rowCollectionOnDone) {
        this.request.rst = [];
      }
    }
  }

  onResetConnection(token) {
    this.connection.emit('resetConnection');
  }

}
/**
 * A handler for the attention acknowledgement message.
 *
 * This message only contains a `DONE` token that acknowledges
 * that the attention message was received by the server.
 */


exports.RequestTokenHandler = RequestTokenHandler;

class AttentionTokenHandler extends TokenHandler {
  /**
   * Returns whether an attention acknowledgement was received.
   */
  constructor(connection, request) {
    super();
    this.connection = void 0;
    this.request = void 0;
    this.attentionReceived = void 0;
    this.connection = connection;
    this.request = request;
    this.attentionReceived = false;
  }

  onDone(token) {
    if (token.attention) {
      this.attentionReceived = true;
    }
  }

}

exports.AttentionTokenHandler = AttentionTokenHandler;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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