"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _sprintfJs = require("sprintf-js");

var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const optionBufferSize = 20;
const TOKEN = {
  VERSION: 0x00,
  ENCRYPTION: 0x01,
  INSTOPT: 0x02,
  THREADID: 0x03,
  MARS: 0x04,
  FEDAUTHREQUIRED: 0x06,
  TERMINATOR: 0xFF
};
const ENCRYPT = {
  OFF: 0x00,
  ON: 0x01,
  NOT_SUP: 0x02,
  REQ: 0x03
};
const encryptByValue = {};

for (const name in ENCRYPT) {
  const value = ENCRYPT[name];
  encryptByValue[value] = name;
}

const MARS = {
  OFF: 0x00,
  ON: 0x01
};
const marsByValue = {};

for (const name in MARS) {
  const value = MARS[name];
  marsByValue[value] = name;
}

/*
  s2.2.6.4
 */
class PreloginPayload {
  constructor(bufferOrOptions = {
    encrypt: false,
    version: {
      major: 0,
      minor: 0,
      build: 0,
      subbuild: 0
    }
  }) {
    this.data = void 0;
    this.options = void 0;
    this.version = void 0;
    this.encryption = void 0;
    this.encryptionString = void 0;
    this.instance = void 0;
    this.threadId = void 0;
    this.mars = void 0;
    this.marsString = void 0;
    this.fedAuthRequired = void 0;

    if (bufferOrOptions instanceof Buffer) {
      this.data = bufferOrOptions;
      this.options = {
        encrypt: false,
        version: {
          major: 0,
          minor: 0,
          build: 0,
          subbuild: 0
        }
      };
    } else {
      this.options = bufferOrOptions;
      this.createOptions();
    }

    this.extractOptions();
  }

  createOptions() {
    const options = [this.createVersionOption(), this.createEncryptionOption(), this.createInstanceOption(), this.createThreadIdOption(), this.createMarsOption(), this.createFedAuthOption()];
    let length = 0;

    for (let i = 0, len = options.length; i < len; i++) {
      const option = options[i];
      length += 5 + option.data.length;
    }

    length++; // terminator

    this.data = Buffer.alloc(length, 0);
    let optionOffset = 0;
    let optionDataOffset = 5 * options.length + 1;

    for (let j = 0, len = options.length; j < len; j++) {
      const option = options[j];
      this.data.writeUInt8(option.token, optionOffset + 0);
      this.data.writeUInt16BE(optionDataOffset, optionOffset + 1);
      this.data.writeUInt16BE(option.data.length, optionOffset + 3);
      optionOffset += 5;
      option.data.copy(this.data, optionDataOffset);
      optionDataOffset += option.data.length;
    }

    this.data.writeUInt8(TOKEN.TERMINATOR, optionOffset);
  }

  createVersionOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(this.options.version.major);
    buffer.writeUInt8(this.options.version.minor);
    buffer.writeUInt16BE(this.options.version.build);
    buffer.writeUInt16BE(this.options.version.subbuild);
    return {
      token: TOKEN.VERSION,
      data: buffer.data
    };
  }

  createEncryptionOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);

    if (this.options.encrypt) {
      buffer.writeUInt8(ENCRYPT.ON);
    } else {
      buffer.writeUInt8(ENCRYPT.NOT_SUP);
    }

    return {
      token: TOKEN.ENCRYPTION,
      data: buffer.data
    };
  }

  createInstanceOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(0x00);
    return {
      token: TOKEN.INSTOPT,
      data: buffer.data
    };
  }

  createThreadIdOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt32BE(0x00);
    return {
      token: TOKEN.THREADID,
      data: buffer.data
    };
  }

  createMarsOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(MARS.OFF);
    return {
      token: TOKEN.MARS,
      data: buffer.data
    };
  }

  createFedAuthOption() {
    const buffer = new _writableTrackingBuffer.default(optionBufferSize);
    buffer.writeUInt8(0x01);
    return {
      token: TOKEN.FEDAUTHREQUIRED,
      data: buffer.data
    };
  }

  extractOptions() {
    let offset = 0;

    while (this.data[offset] !== TOKEN.TERMINATOR) {
      let dataOffset = this.data.readUInt16BE(offset + 1);
      const dataLength = this.data.readUInt16BE(offset + 3);

      switch (this.data[offset]) {
        case TOKEN.VERSION:
          this.extractVersion(dataOffset);
          break;

        case TOKEN.ENCRYPTION:
          this.extractEncryption(dataOffset);
          break;

        case TOKEN.INSTOPT:
          this.extractInstance(dataOffset);
          break;

        case TOKEN.THREADID:
          if (dataLength > 0) {
            this.extractThreadId(dataOffset);
          }

          break;

        case TOKEN.MARS:
          this.extractMars(dataOffset);
          break;

        case TOKEN.FEDAUTHREQUIRED:
          this.extractFedAuth(dataOffset);
          break;
      }

      offset += 5;
      dataOffset += dataLength;
    }
  }

  extractVersion(offset) {
    this.version = {
      major: this.data.readUInt8(offset + 0),
      minor: this.data.readUInt8(offset + 1),
      build: this.data.readUInt16BE(offset + 2),
      subbuild: this.data.readUInt16BE(offset + 4)
    };
  }

  extractEncryption(offset) {
    this.encryption = this.data.readUInt8(offset);
    this.encryptionString = encryptByValue[this.encryption];
  }

  extractInstance(offset) {
    this.instance = this.data.readUInt8(offset);
  }

  extractThreadId(offset) {
    this.threadId = this.data.readUInt32BE(offset);
  }

  extractMars(offset) {
    this.mars = this.data.readUInt8(offset);
    this.marsString = marsByValue[this.mars];
  }

  extractFedAuth(offset) {
    this.fedAuthRequired = this.data.readUInt8(offset);
  }

  toString(indent = '') {
    return indent + 'PreLogin - ' + (0, _sprintfJs.sprintf)('version:%d.%d.%d.%d, encryption:0x%02X(%s), instopt:0x%02X, threadId:0x%08X, mars:0x%02X(%s)', this.version.major, this.version.minor, this.version.build, this.version.subbuild, this.encryption ? this.encryption : 0, this.encryptionString ? this.encryptionString : '', this.instance ? this.instance : 0, this.threadId ? this.threadId : 0, this.mars ? this.mars : 0, this.marsString ? this.marsString : '');
  }

}

var _default = PreloginPayload;
exports.default = _default;
module.exports = PreloginPayload;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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