"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readCollation = readCollation;
exports.default = void 0;

var _collation = require("./collation");

var _dataType = require("./data-type");

var _sprintfJs = require("sprintf-js");

function readCollation(parser, callback) {
  // s2.2.5.1.2
  parser.readBuffer(5, collationData => {
    callback(_collation.Collation.fromBuffer(collationData));
  });
}

function readSchema(parser, callback) {
  // s2.2.5.5.3
  parser.readUInt8(schemaPresent => {
    if (schemaPresent === 0x01) {
      parser.readBVarChar(dbname => {
        parser.readBVarChar(owningSchema => {
          parser.readUsVarChar(xmlSchemaCollection => {
            callback({
              dbname: dbname,
              owningSchema: owningSchema,
              xmlSchemaCollection: xmlSchemaCollection
            });
          });
        });
      });
    } else {
      callback(undefined);
    }
  });
}

function readUDTInfo(parser, callback) {
  parser.readUInt16LE(maxByteSize => {
    parser.readBVarChar(dbname => {
      parser.readBVarChar(owningSchema => {
        parser.readBVarChar(typeName => {
          parser.readUsVarChar(assemblyName => {
            callback({
              maxByteSize: maxByteSize,
              dbname: dbname,
              owningSchema: owningSchema,
              typeName: typeName,
              assemblyName: assemblyName
            });
          });
        });
      });
    });
  });
}

function metadataParse(parser, options, callback) {
  (options.tdsVersion < '7_2' ? parser.readUInt16LE : parser.readUInt32LE).call(parser, userType => {
    parser.readUInt16LE(flags => {
      parser.readUInt8(typeNumber => {
        const type = _dataType.TYPE[typeNumber];

        if (!type) {
          throw new Error((0, _sprintfJs.sprintf)('Unrecognised data type 0x%02X', typeNumber));
        }

        switch (type.name) {
          case 'Null':
          case 'TinyInt':
          case 'SmallInt':
          case 'Int':
          case 'BigInt':
          case 'Real':
          case 'Float':
          case 'SmallMoney':
          case 'Money':
          case 'Bit':
          case 'SmallDateTime':
          case 'DateTime':
          case 'Date':
            return callback({
              userType: userType,
              flags: flags,
              type: type,
              collation: undefined,
              precision: undefined,
              scale: undefined,
              dataLength: undefined,
              schema: undefined,
              udtInfo: undefined
            });

          case 'IntN':
          case 'FloatN':
          case 'MoneyN':
          case 'BitN':
          case 'UniqueIdentifier':
          case 'DateTimeN':
            return parser.readUInt8(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'Variant':
            return parser.readUInt32LE(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'VarChar':
          case 'Char':
          case 'NVarChar':
          case 'NChar':
            return parser.readUInt16LE(dataLength => {
              readCollation(parser, collation => {
                callback({
                  userType: userType,
                  flags: flags,
                  type: type,
                  collation: collation,
                  precision: undefined,
                  scale: undefined,
                  dataLength: dataLength,
                  schema: undefined,
                  udtInfo: undefined
                });
              });
            });

          case 'Text':
          case 'NText':
            return parser.readUInt32LE(dataLength => {
              readCollation(parser, collation => {
                callback({
                  userType: userType,
                  flags: flags,
                  type: type,
                  collation: collation,
                  precision: undefined,
                  scale: undefined,
                  dataLength: dataLength,
                  schema: undefined,
                  udtInfo: undefined
                });
              });
            });

          case 'VarBinary':
          case 'Binary':
            return parser.readUInt16LE(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'Image':
            return parser.readUInt32LE(dataLength => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: dataLength,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'Xml':
            return readSchema(parser, schema => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: undefined,
                schema: schema,
                udtInfo: undefined
              });
            });

          case 'Time':
          case 'DateTime2':
          case 'DateTimeOffset':
            return parser.readUInt8(scale => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: scale,
                dataLength: undefined,
                schema: undefined,
                udtInfo: undefined
              });
            });

          case 'NumericN':
          case 'DecimalN':
            return parser.readUInt8(dataLength => {
              parser.readUInt8(precision => {
                parser.readUInt8(scale => {
                  callback({
                    userType: userType,
                    flags: flags,
                    type: type,
                    collation: undefined,
                    precision: precision,
                    scale: scale,
                    dataLength: dataLength,
                    schema: undefined,
                    udtInfo: undefined
                  });
                });
              });
            });

          case 'UDT':
            return readUDTInfo(parser, udtInfo => {
              callback({
                userType: userType,
                flags: flags,
                type: type,
                collation: undefined,
                precision: undefined,
                scale: undefined,
                dataLength: undefined,
                schema: undefined,
                udtInfo: udtInfo
              });
            });

          default:
            throw new Error((0, _sprintfJs.sprintf)('Unrecognised type %s', type.name));
        }
      });
    });
  });
}

var _default = metadataParse;
exports.default = _default;
module.exports = metadataParse;
module.exports.readCollation = readCollation;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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