"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _sprintfJs = require("sprintf-js");

var _tdsVersions = require("./tds-versions");

const FLAGS_1 = {
  ENDIAN_LITTLE: 0x00,
  ENDIAN_BIG: 0x01,
  CHARSET_ASCII: 0x00,
  CHARSET_EBCDIC: 0x02,
  FLOAT_IEEE_754: 0x00,
  FLOAT_VAX: 0x04,
  FLOAT_ND5000: 0x08,
  BCP_DUMPLOAD_ON: 0x00,
  BCP_DUMPLOAD_OFF: 0x10,
  USE_DB_ON: 0x00,
  USE_DB_OFF: 0x20,
  INIT_DB_WARN: 0x00,
  INIT_DB_FATAL: 0x40,
  SET_LANG_WARN_OFF: 0x00,
  SET_LANG_WARN_ON: 0x80
};
const FLAGS_2 = {
  INIT_LANG_WARN: 0x00,
  INIT_LANG_FATAL: 0x01,
  ODBC_OFF: 0x00,
  ODBC_ON: 0x02,
  F_TRAN_BOUNDARY: 0x04,
  F_CACHE_CONNECT: 0x08,
  USER_NORMAL: 0x00,
  USER_SERVER: 0x10,
  USER_REMUSER: 0x20,
  USER_SQLREPL: 0x40,
  INTEGRATED_SECURITY_OFF: 0x00,
  INTEGRATED_SECURITY_ON: 0x80
};
const TYPE_FLAGS = {
  SQL_DFLT: 0x00,
  SQL_TSQL: 0x08,
  OLEDB_OFF: 0x00,
  OLEDB_ON: 0x10,
  READ_WRITE_INTENT: 0x00,
  READ_ONLY_INTENT: 0x20
};
const FLAGS_3 = {
  CHANGE_PASSWORD_NO: 0x00,
  CHANGE_PASSWORD_YES: 0x01,
  BINARY_XML: 0x02,
  SPAWN_USER_INSTANCE: 0x04,
  UNKNOWN_COLLATION_HANDLING: 0x08,
  EXTENSION_USED: 0x10
};
const FEDAUTH_OPTIONS = {
  FEATURE_ID: 0x02,
  LIBRARY_SECURITYTOKEN: 0x01,
  LIBRARY_ADAL: 0x02,
  FEDAUTH_YES_ECHO: 0x01,
  FEDAUTH_NO_ECHO: 0x00,
  ADAL_WORKFLOW_USER_PASS: 0x01,
  ADAL_WORKFLOW_INTEGRATED: 0x02
};
const FEATURE_EXT_TERMINATOR = 0xFF;

/*
  s2.2.6.3
 */
class Login7Payload {
  constructor({
    tdsVersion,
    packetSize,
    clientProgVer,
    clientPid,
    connectionId,
    clientTimeZone,
    clientLcid
  }) {
    this.tdsVersion = void 0;
    this.packetSize = void 0;
    this.clientProgVer = void 0;
    this.clientPid = void 0;
    this.connectionId = void 0;
    this.clientTimeZone = void 0;
    this.clientLcid = void 0;
    this.readOnlyIntent = void 0;
    this.initDbFatal = void 0;
    this.userName = void 0;
    this.password = void 0;
    this.serverName = void 0;
    this.appName = void 0;
    this.hostname = void 0;
    this.libraryName = void 0;
    this.language = void 0;
    this.database = void 0;
    this.clientId = void 0;
    this.sspi = void 0;
    this.attachDbFile = void 0;
    this.changePassword = void 0;
    this.fedAuth = void 0;
    this.tdsVersion = tdsVersion;
    this.packetSize = packetSize;
    this.clientProgVer = clientProgVer;
    this.clientPid = clientPid;
    this.connectionId = connectionId;
    this.clientTimeZone = clientTimeZone;
    this.clientLcid = clientLcid;
    this.readOnlyIntent = false;
    this.initDbFatal = false;
    this.fedAuth = undefined;
    this.userName = undefined;
    this.password = undefined;
    this.serverName = undefined;
    this.appName = undefined;
    this.hostname = undefined;
    this.libraryName = undefined;
    this.language = undefined;
    this.database = undefined;
    this.clientId = undefined;
    this.sspi = undefined;
    this.attachDbFile = undefined;
    this.changePassword = undefined;
  }

  toBuffer() {
    const fixedData = Buffer.alloc(94);
    const buffers = [fixedData];
    let offset = 0;
    let dataOffset = fixedData.length; // Length: 4-byte

    offset = fixedData.writeUInt32LE(0, offset); // TDSVersion: 4-byte

    offset = fixedData.writeUInt32LE(this.tdsVersion, offset); // PacketSize: 4-byte

    offset = fixedData.writeUInt32LE(this.packetSize, offset); // ClientProgVer: 4-byte

    offset = fixedData.writeUInt32LE(this.clientProgVer, offset); // ClientPID: 4-byte

    offset = fixedData.writeUInt32LE(this.clientPid, offset); // ConnectionID: 4-byte

    offset = fixedData.writeUInt32LE(this.connectionId, offset); // OptionFlags1: 1-byte

    offset = fixedData.writeUInt8(this.buildOptionFlags1(), offset); // OptionFlags2: 1-byte

    offset = fixedData.writeUInt8(this.buildOptionFlags2(), offset); // TypeFlags: 1-byte

    offset = fixedData.writeUInt8(this.buildTypeFlags(), offset); // OptionFlags3: 1-byte

    offset = fixedData.writeUInt8(this.buildOptionFlags3(), offset); // ClientTimZone: 4-byte

    offset = fixedData.writeInt32LE(this.clientTimeZone, offset); // ClientLCID: 4-byte

    offset = fixedData.writeUInt32LE(this.clientLcid, offset); // ibHostName: 2-byte

    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchHostName: 2-byte

    if (this.hostname) {
      const buffer = Buffer.from(this.hostname, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(dataOffset, offset);
    } // ibUserName: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchUserName: 2-byte

    if (this.userName) {
      const buffer = Buffer.from(this.userName, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ibPassword: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchPassword: 2-byte

    if (this.password) {
      const buffer = Buffer.from(this.password, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(this.scramblePassword(buffer));
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ibAppName: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchAppName: 2-byte

    if (this.appName) {
      const buffer = Buffer.from(this.appName, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ibServerName: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchServerName: 2-byte

    if (this.serverName) {
      const buffer = Buffer.from(this.serverName, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // (ibUnused / ibExtension): 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // (cchUnused / cbExtension): 2-byte

    const extensions = this.buildFeatureExt();
    offset = fixedData.writeUInt16LE(4, offset);
    const extensionOffset = Buffer.alloc(4);
    extensionOffset.writeUInt32LE(dataOffset += 4, 0);
    dataOffset += extensions.length;
    buffers.push(extensionOffset, extensions); // ibCltIntName: 2-byte

    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchCltIntName: 2-byte

    if (this.libraryName) {
      const buffer = Buffer.from(this.libraryName, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ibLanguage: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchLanguage: 2-byte

    if (this.language) {
      const buffer = Buffer.from(this.language, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ibDatabase: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchDatabase: 2-byte

    if (this.database) {
      const buffer = Buffer.from(this.database, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ClientID: 6-byte


    if (this.clientId) {
      this.clientId.copy(fixedData, offset, 0, 6);
    }

    offset += 6; // ibSSPI: 2-byte

    offset = fixedData.writeUInt16LE(dataOffset, offset); // cbSSPI: 2-byte

    if (this.sspi) {
      if (this.sspi.length > 65535) {
        offset = fixedData.writeUInt16LE(65535, offset);
      } else {
        offset = fixedData.writeUInt16LE(this.sspi.length, offset);
      }

      buffers.push(this.sspi);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ibAtchDBFile: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchAtchDBFile: 2-byte

    if (this.attachDbFile) {
      const buffer = Buffer.from(this.attachDbFile, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // ibChangePassword: 2-byte


    offset = fixedData.writeUInt16LE(dataOffset, offset); // cchChangePassword: 2-byte

    if (this.changePassword) {
      const buffer = Buffer.from(this.changePassword, 'ucs2');
      offset = fixedData.writeUInt16LE(buffer.length / 2, offset);
      dataOffset += buffer.length;
      buffers.push(buffer);
    } else {
      offset = fixedData.writeUInt16LE(0, offset);
    } // cbSSPILong: 4-byte


    if (this.sspi && this.sspi.length > 65535) {
      fixedData.writeUInt32LE(this.sspi.length, offset);
    } else {
      fixedData.writeUInt32LE(0, offset);
    }

    const data = Buffer.concat(buffers);
    data.writeUInt32LE(data.length, 0);
    return data;
  }

  buildOptionFlags1() {
    let flags1 = FLAGS_1.ENDIAN_LITTLE | FLAGS_1.CHARSET_ASCII | FLAGS_1.FLOAT_IEEE_754 | FLAGS_1.BCP_DUMPLOAD_OFF | FLAGS_1.USE_DB_OFF | FLAGS_1.SET_LANG_WARN_ON;

    if (this.initDbFatal) {
      flags1 |= FLAGS_1.INIT_DB_FATAL;
    } else {
      flags1 |= FLAGS_1.INIT_DB_WARN;
    }

    return flags1;
  }

  buildFeatureExt() {
    const buffers = [];
    const fedAuth = this.fedAuth;

    if (fedAuth) {
      switch (fedAuth.type) {
        case 'ADAL':
          const buffer = Buffer.alloc(7);
          buffer.writeUInt8(FEDAUTH_OPTIONS.FEATURE_ID, 0);
          buffer.writeUInt32LE(2, 1);
          buffer.writeUInt8(FEDAUTH_OPTIONS.LIBRARY_ADAL << 1 | (fedAuth.echo ? FEDAUTH_OPTIONS.FEDAUTH_YES_ECHO : FEDAUTH_OPTIONS.FEDAUTH_NO_ECHO), 5);
          buffer.writeUInt8(fedAuth.workflow === 'integrated' ? 0x02 : FEDAUTH_OPTIONS.ADAL_WORKFLOW_USER_PASS, 6);
          buffers.push(buffer);
          break;

        case 'SECURITYTOKEN':
          const token = Buffer.from(fedAuth.fedAuthToken, 'ucs2');
          const buf = Buffer.alloc(10);
          let offset = 0;
          offset = buf.writeUInt8(FEDAUTH_OPTIONS.FEATURE_ID, offset);
          offset = buf.writeUInt32LE(token.length + 4 + 1, offset);
          offset = buf.writeUInt8(FEDAUTH_OPTIONS.LIBRARY_SECURITYTOKEN << 1 | (fedAuth.echo ? FEDAUTH_OPTIONS.FEDAUTH_YES_ECHO : FEDAUTH_OPTIONS.FEDAUTH_NO_ECHO), offset);
          buf.writeInt32LE(token.length, offset);
          buffers.push(buf);
          buffers.push(token);
          break;
      }
    }

    if (this.tdsVersion >= _tdsVersions.versions['7_4']) {
      // Signal UTF-8 support: Value 0x0A, bit 0 must be set to 1. Added in TDS 7.4.
      const UTF8_SUPPORT_FEATURE_ID = 0x0a;
      const UTF8_SUPPORT_CLIENT_SUPPORTS_UTF8 = 0x01;
      const buf = Buffer.alloc(6);
      buf.writeUInt8(UTF8_SUPPORT_FEATURE_ID, 0);
      buf.writeUInt32LE(1, 1);
      buf.writeUInt8(UTF8_SUPPORT_CLIENT_SUPPORTS_UTF8, 5);
      buffers.push(buf);
    }

    buffers.push(Buffer.from([FEATURE_EXT_TERMINATOR]));
    return Buffer.concat(buffers);
  }

  buildOptionFlags2() {
    let flags2 = FLAGS_2.INIT_LANG_WARN | FLAGS_2.ODBC_OFF | FLAGS_2.USER_NORMAL;

    if (this.sspi) {
      flags2 |= FLAGS_2.INTEGRATED_SECURITY_ON;
    } else {
      flags2 |= FLAGS_2.INTEGRATED_SECURITY_OFF;
    }

    return flags2;
  }

  buildTypeFlags() {
    let typeFlags = TYPE_FLAGS.SQL_DFLT | TYPE_FLAGS.OLEDB_OFF;

    if (this.readOnlyIntent) {
      typeFlags |= TYPE_FLAGS.READ_ONLY_INTENT;
    } else {
      typeFlags |= TYPE_FLAGS.READ_WRITE_INTENT;
    }

    return typeFlags;
  }

  buildOptionFlags3() {
    return FLAGS_3.CHANGE_PASSWORD_NO | FLAGS_3.UNKNOWN_COLLATION_HANDLING | FLAGS_3.EXTENSION_USED;
  }

  scramblePassword(password) {
    for (let b = 0, len = password.length; b < len; b++) {
      let byte = password[b];
      const lowNibble = byte & 0x0f;
      const highNibble = byte >> 4;
      byte = lowNibble << 4 | highNibble;
      byte = byte ^ 0xa5;
      password[b] = byte;
    }

    return password;
  }

  toString(indent = '') {
    return indent + 'Login7 - ' + (0, _sprintfJs.sprintf)('TDS:0x%08X, PacketSize:0x%08X, ClientProgVer:0x%08X, ClientPID:0x%08X, ConnectionID:0x%08X', this.tdsVersion, this.packetSize, this.clientProgVer, this.clientPid, this.connectionId) + '\n' + indent + '         ' + (0, _sprintfJs.sprintf)('Flags1:0x%02X, Flags2:0x%02X, TypeFlags:0x%02X, Flags3:0x%02X, ClientTimezone:%d, ClientLCID:0x%08X', this.buildOptionFlags1(), this.buildOptionFlags2(), this.buildTypeFlags(), this.buildOptionFlags3(), this.clientTimeZone, this.clientLcid) + '\n' + indent + '         ' + (0, _sprintfJs.sprintf)("Hostname:'%s', Username:'%s', Password:'%s', AppName:'%s', ServerName:'%s', LibraryName:'%s'", this.hostname, this.userName, this.password, this.appName, this.serverName, this.libraryName) + '\n' + indent + '         ' + (0, _sprintfJs.sprintf)("Language:'%s', Database:'%s', SSPI:'%s', AttachDbFile:'%s', ChangePassword:'%s'", this.language, this.database, this.sspi, this.attachDbFile, this.changePassword);
  }

}

var _default = Login7Payload;
exports.default = _default;
module.exports = Login7Payload;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy9sb2dpbjctcGF5bG9hZC50cyJdLCJuYW1lcyI6WyJGTEFHU18xIiwiRU5ESUFOX0xJVFRMRSIsIkVORElBTl9CSUciLCJDSEFSU0VUX0FTQ0lJIiwiQ0hBUlNFVF9FQkNESUMiLCJGTE9BVF9JRUVFXzc1NCIsIkZMT0FUX1ZBWCIsIkZMT0FUX05ENTAwMCIsIkJDUF9EVU1QTE9BRF9PTiIsIkJDUF9EVU1QTE9BRF9PRkYiLCJVU0VfREJfT04iLCJVU0VfREJfT0ZGIiwiSU5JVF9EQl9XQVJOIiwiSU5JVF9EQl9GQVRBTCIsIlNFVF9MQU5HX1dBUk5fT0ZGIiwiU0VUX0xBTkdfV0FSTl9PTiIsIkZMQUdTXzIiLCJJTklUX0xBTkdfV0FSTiIsIklOSVRfTEFOR19GQVRBTCIsIk9EQkNfT0ZGIiwiT0RCQ19PTiIsIkZfVFJBTl9CT1VOREFSWSIsIkZfQ0FDSEVfQ09OTkVDVCIsIlVTRVJfTk9STUFMIiwiVVNFUl9TRVJWRVIiLCJVU0VSX1JFTVVTRVIiLCJVU0VSX1NRTFJFUEwiLCJJTlRFR1JBVEVEX1NFQ1VSSVRZX09GRiIsIklOVEVHUkFURURfU0VDVVJJVFlfT04iLCJUWVBFX0ZMQUdTIiwiU1FMX0RGTFQiLCJTUUxfVFNRTCIsIk9MRURCX09GRiIsIk9MRURCX09OIiwiUkVBRF9XUklURV9JTlRFTlQiLCJSRUFEX09OTFlfSU5URU5UIiwiRkxBR1NfMyIsIkNIQU5HRV9QQVNTV09SRF9OTyIsIkNIQU5HRV9QQVNTV09SRF9ZRVMiLCJCSU5BUllfWE1MIiwiU1BBV05fVVNFUl9JTlNUQU5DRSIsIlVOS05PV05fQ09MTEFUSU9OX0hBTkRMSU5HIiwiRVhURU5TSU9OX1VTRUQiLCJGRURBVVRIX09QVElPTlMiLCJGRUFUVVJFX0lEIiwiTElCUkFSWV9TRUNVUklUWVRPS0VOIiwiTElCUkFSWV9BREFMIiwiRkVEQVVUSF9ZRVNfRUNITyIsIkZFREFVVEhfTk9fRUNITyIsIkFEQUxfV09SS0ZMT1dfVVNFUl9QQVNTIiwiQURBTF9XT1JLRkxPV19JTlRFR1JBVEVEIiwiRkVBVFVSRV9FWFRfVEVSTUlOQVRPUiIsIkxvZ2luN1BheWxvYWQiLCJjb25zdHJ1Y3RvciIsInRkc1ZlcnNpb24iLCJwYWNrZXRTaXplIiwiY2xpZW50UHJvZ1ZlciIsImNsaWVudFBpZCIsImNvbm5lY3Rpb25JZCIsImNsaWVudFRpbWVab25lIiwiY2xpZW50TGNpZCIsInJlYWRPbmx5SW50ZW50IiwiaW5pdERiRmF0YWwiLCJ1c2VyTmFtZSIsInBhc3N3b3JkIiwic2VydmVyTmFtZSIsImFwcE5hbWUiLCJob3N0bmFtZSIsImxpYnJhcnlOYW1lIiwibGFuZ3VhZ2UiLCJkYXRhYmFzZSIsImNsaWVudElkIiwic3NwaSIsImF0dGFjaERiRmlsZSIsImNoYW5nZVBhc3N3b3JkIiwiZmVkQXV0aCIsInVuZGVmaW5lZCIsInRvQnVmZmVyIiwiZml4ZWREYXRhIiwiQnVmZmVyIiwiYWxsb2MiLCJidWZmZXJzIiwib2Zmc2V0IiwiZGF0YU9mZnNldCIsImxlbmd0aCIsIndyaXRlVUludDMyTEUiLCJ3cml0ZVVJbnQ4IiwiYnVpbGRPcHRpb25GbGFnczEiLCJidWlsZE9wdGlvbkZsYWdzMiIsImJ1aWxkVHlwZUZsYWdzIiwiYnVpbGRPcHRpb25GbGFnczMiLCJ3cml0ZUludDMyTEUiLCJ3cml0ZVVJbnQxNkxFIiwiYnVmZmVyIiwiZnJvbSIsInB1c2giLCJzY3JhbWJsZVBhc3N3b3JkIiwiZXh0ZW5zaW9ucyIsImJ1aWxkRmVhdHVyZUV4dCIsImV4dGVuc2lvbk9mZnNldCIsImNvcHkiLCJkYXRhIiwiY29uY2F0IiwiZmxhZ3MxIiwidHlwZSIsImVjaG8iLCJ3b3JrZmxvdyIsInRva2VuIiwiZmVkQXV0aFRva2VuIiwiYnVmIiwidmVyc2lvbnMiLCJVVEY4X1NVUFBPUlRfRkVBVFVSRV9JRCIsIlVURjhfU1VQUE9SVF9DTElFTlRfU1VQUE9SVFNfVVRGOCIsImZsYWdzMiIsInR5cGVGbGFncyIsImIiLCJsZW4iLCJieXRlIiwibG93TmliYmxlIiwiaGlnaE5pYmJsZSIsInRvU3RyaW5nIiwiaW5kZW50IiwibW9kdWxlIiwiZXhwb3J0cyJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUVBLE1BQU1BLE9BQU8sR0FBRztBQUNkQyxFQUFBQSxhQUFhLEVBQUUsSUFERDtBQUVkQyxFQUFBQSxVQUFVLEVBQUUsSUFGRTtBQUdkQyxFQUFBQSxhQUFhLEVBQUUsSUFIRDtBQUlkQyxFQUFBQSxjQUFjLEVBQUUsSUFKRjtBQUtkQyxFQUFBQSxjQUFjLEVBQUUsSUFMRjtBQU1kQyxFQUFBQSxTQUFTLEVBQUUsSUFORztBQU9kQyxFQUFBQSxZQUFZLEVBQUUsSUFQQTtBQVFkQyxFQUFBQSxlQUFlLEVBQUUsSUFSSDtBQVNkQyxFQUFBQSxnQkFBZ0IsRUFBRSxJQVRKO0FBVWRDLEVBQUFBLFNBQVMsRUFBRSxJQVZHO0FBV2RDLEVBQUFBLFVBQVUsRUFBRSxJQVhFO0FBWWRDLEVBQUFBLFlBQVksRUFBRSxJQVpBO0FBYWRDLEVBQUFBLGFBQWEsRUFBRSxJQWJEO0FBY2RDLEVBQUFBLGlCQUFpQixFQUFFLElBZEw7QUFlZEMsRUFBQUEsZ0JBQWdCLEVBQUU7QUFmSixDQUFoQjtBQWtCQSxNQUFNQyxPQUFPLEdBQUc7QUFDZEMsRUFBQUEsY0FBYyxFQUFFLElBREY7QUFFZEMsRUFBQUEsZUFBZSxFQUFFLElBRkg7QUFHZEMsRUFBQUEsUUFBUSxFQUFFLElBSEk7QUFJZEMsRUFBQUEsT0FBTyxFQUFFLElBSks7QUFLZEMsRUFBQUEsZUFBZSxFQUFFLElBTEg7QUFNZEMsRUFBQUEsZUFBZSxFQUFFLElBTkg7QUFPZEMsRUFBQUEsV0FBVyxFQUFFLElBUEM7QUFRZEMsRUFBQUEsV0FBVyxFQUFFLElBUkM7QUFTZEMsRUFBQUEsWUFBWSxFQUFFLElBVEE7QUFVZEMsRUFBQUEsWUFBWSxFQUFFLElBVkE7QUFXZEMsRUFBQUEsdUJBQXVCLEVBQUUsSUFYWDtBQVlkQyxFQUFBQSxzQkFBc0IsRUFBRTtBQVpWLENBQWhCO0FBZUEsTUFBTUMsVUFBVSxHQUFHO0FBQ2pCQyxFQUFBQSxRQUFRLEVBQUUsSUFETztBQUVqQkMsRUFBQUEsUUFBUSxFQUFFLElBRk87QUFHakJDLEVBQUFBLFNBQVMsRUFBRSxJQUhNO0FBSWpCQyxFQUFBQSxRQUFRLEVBQUUsSUFKTztBQUtqQkMsRUFBQUEsaUJBQWlCLEVBQUUsSUFMRjtBQU1qQkMsRUFBQUEsZ0JBQWdCLEVBQUU7QUFORCxDQUFuQjtBQVNBLE1BQU1DLE9BQU8sR0FBRztBQUNkQyxFQUFBQSxrQkFBa0IsRUFBRSxJQUROO0FBRWRDLEVBQUFBLG1CQUFtQixFQUFFLElBRlA7QUFHZEMsRUFBQUEsVUFBVSxFQUFFLElBSEU7QUFJZEMsRUFBQUEsbUJBQW1CLEVBQUUsSUFKUDtBQUtkQyxFQUFBQSwwQkFBMEIsRUFBRSxJQUxkO0FBTWRDLEVBQUFBLGNBQWMsRUFBRTtBQU5GLENBQWhCO0FBU0EsTUFBTUMsZUFBZSxHQUFHO0FBQ3RCQyxFQUFBQSxVQUFVLEVBQUUsSUFEVTtBQUV0QkMsRUFBQUEscUJBQXFCLEVBQUUsSUFGRDtBQUd0QkMsRUFBQUEsWUFBWSxFQUFFLElBSFE7QUFJdEJDLEVBQUFBLGdCQUFnQixFQUFFLElBSkk7QUFLdEJDLEVBQUFBLGVBQWUsRUFBRSxJQUxLO0FBTXRCQyxFQUFBQSx1QkFBdUIsRUFBRSxJQU5IO0FBT3RCQyxFQUFBQSx3QkFBd0IsRUFBRTtBQVBKLENBQXhCO0FBVUEsTUFBTUMsc0JBQXNCLEdBQUcsSUFBL0I7O0FBWUE7QUFDQTtBQUNBO0FBQ0EsTUFBTUMsYUFBTixDQUFvQjtBQTJCbEJDLEVBQUFBLFdBQVcsQ0FBQztBQUFFQyxJQUFBQSxVQUFGO0FBQWNDLElBQUFBLFVBQWQ7QUFBMEJDLElBQUFBLGFBQTFCO0FBQXlDQyxJQUFBQSxTQUF6QztBQUFvREMsSUFBQUEsWUFBcEQ7QUFBa0VDLElBQUFBLGNBQWxFO0FBQWtGQyxJQUFBQTtBQUFsRixHQUFELEVBQTBHO0FBQUEsU0ExQnJITixVQTBCcUg7QUFBQSxTQXpCckhDLFVBeUJxSDtBQUFBLFNBeEJySEMsYUF3QnFIO0FBQUEsU0F2QnJIQyxTQXVCcUg7QUFBQSxTQXRCckhDLFlBc0JxSDtBQUFBLFNBckJySEMsY0FxQnFIO0FBQUEsU0FwQnJIQyxVQW9CcUg7QUFBQSxTQWxCckhDLGNBa0JxSDtBQUFBLFNBakJySEMsV0FpQnFIO0FBQUEsU0FmckhDLFFBZXFIO0FBQUEsU0FkckhDLFFBY3FIO0FBQUEsU0FickhDLFVBYXFIO0FBQUEsU0FackhDLE9BWXFIO0FBQUEsU0FYckhDLFFBV3FIO0FBQUEsU0FWckhDLFdBVXFIO0FBQUEsU0FUckhDLFFBU3FIO0FBQUEsU0FSckhDLFFBUXFIO0FBQUEsU0FQckhDLFFBT3FIO0FBQUEsU0FOckhDLElBTXFIO0FBQUEsU0FMckhDLFlBS3FIO0FBQUEsU0FKckhDLGNBSXFIO0FBQUEsU0FGckhDLE9BRXFIO0FBQ25ILFNBQUtyQixVQUFMLEdBQWtCQSxVQUFsQjtBQUNBLFNBQUtDLFVBQUwsR0FBa0JBLFVBQWxCO0FBQ0EsU0FBS0MsYUFBTCxHQUFxQkEsYUFBckI7QUFDQSxTQUFLQyxTQUFMLEdBQWlCQSxTQUFqQjtBQUNBLFNBQUtDLFlBQUwsR0FBb0JBLFlBQXBCO0FBQ0EsU0FBS0MsY0FBTCxHQUFzQkEsY0FBdEI7QUFDQSxTQUFLQyxVQUFMLEdBQWtCQSxVQUFsQjtBQUVBLFNBQUtDLGNBQUwsR0FBc0IsS0FBdEI7QUFDQSxTQUFLQyxXQUFMLEdBQW1CLEtBQW5CO0FBRUEsU0FBS2EsT0FBTCxHQUFlQyxTQUFmO0FBRUEsU0FBS2IsUUFBTCxHQUFnQmEsU0FBaEI7QUFDQSxTQUFLWixRQUFMLEdBQWdCWSxTQUFoQjtBQUNBLFNBQUtYLFVBQUwsR0FBa0JXLFNBQWxCO0FBQ0EsU0FBS1YsT0FBTCxHQUFlVSxTQUFmO0FBQ0EsU0FBS1QsUUFBTCxHQUFnQlMsU0FBaEI7QUFDQSxTQUFLUixXQUFMLEdBQW1CUSxTQUFuQjtBQUNBLFNBQUtQLFFBQUwsR0FBZ0JPLFNBQWhCO0FBQ0EsU0FBS04sUUFBTCxHQUFnQk0sU0FBaEI7QUFDQSxTQUFLTCxRQUFMLEdBQWdCSyxTQUFoQjtBQUNBLFNBQUtKLElBQUwsR0FBWUksU0FBWjtBQUNBLFNBQUtILFlBQUwsR0FBb0JHLFNBQXBCO0FBQ0EsU0FBS0YsY0FBTCxHQUFzQkUsU0FBdEI7QUFDRDs7QUFFREMsRUFBQUEsUUFBUSxHQUFHO0FBQ1QsVUFBTUMsU0FBUyxHQUFHQyxNQUFNLENBQUNDLEtBQVAsQ0FBYSxFQUFiLENBQWxCO0FBQ0EsVUFBTUMsT0FBTyxHQUFHLENBQUNILFNBQUQsQ0FBaEI7QUFFQSxRQUFJSSxNQUFNLEdBQUcsQ0FBYjtBQUNBLFFBQUlDLFVBQVUsR0FBR0wsU0FBUyxDQUFDTSxNQUEzQixDQUxTLENBT1Q7O0FBQ0FGLElBQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDTyxhQUFWLENBQXdCLENBQXhCLEVBQTJCSCxNQUEzQixDQUFULENBUlMsQ0FVVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNPLGFBQVYsQ0FBd0IsS0FBSy9CLFVBQTdCLEVBQXlDNEIsTUFBekMsQ0FBVCxDQVhTLENBYVQ7O0FBQ0FBLElBQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDTyxhQUFWLENBQXdCLEtBQUs5QixVQUE3QixFQUF5QzJCLE1BQXpDLENBQVQsQ0FkUyxDQWdCVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNPLGFBQVYsQ0FBd0IsS0FBSzdCLGFBQTdCLEVBQTRDMEIsTUFBNUMsQ0FBVCxDQWpCUyxDQW1CVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNPLGFBQVYsQ0FBd0IsS0FBSzVCLFNBQTdCLEVBQXdDeUIsTUFBeEMsQ0FBVCxDQXBCUyxDQXNCVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNPLGFBQVYsQ0FBd0IsS0FBSzNCLFlBQTdCLEVBQTJDd0IsTUFBM0MsQ0FBVCxDQXZCUyxDQXlCVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNRLFVBQVYsQ0FBcUIsS0FBS0MsaUJBQUwsRUFBckIsRUFBK0NMLE1BQS9DLENBQVQsQ0ExQlMsQ0E0QlQ7O0FBQ0FBLElBQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDUSxVQUFWLENBQXFCLEtBQUtFLGlCQUFMLEVBQXJCLEVBQStDTixNQUEvQyxDQUFULENBN0JTLENBK0JUOztBQUNBQSxJQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ1EsVUFBVixDQUFxQixLQUFLRyxjQUFMLEVBQXJCLEVBQTRDUCxNQUE1QyxDQUFULENBaENTLENBa0NUOztBQUNBQSxJQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ1EsVUFBVixDQUFxQixLQUFLSSxpQkFBTCxFQUFyQixFQUErQ1IsTUFBL0MsQ0FBVCxDQW5DUyxDQXFDVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNhLFlBQVYsQ0FBdUIsS0FBS2hDLGNBQTVCLEVBQTRDdUIsTUFBNUMsQ0FBVCxDQXRDUyxDQXdDVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNPLGFBQVYsQ0FBd0IsS0FBS3pCLFVBQTdCLEVBQXlDc0IsTUFBekMsQ0FBVCxDQXpDUyxDQTJDVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBNUNTLENBOENUOztBQUNBLFFBQUksS0FBS2YsUUFBVCxFQUFtQjtBQUNqQixZQUFNMEIsTUFBTSxHQUFHZCxNQUFNLENBQUNlLElBQVAsQ0FBWSxLQUFLM0IsUUFBakIsRUFBMkIsTUFBM0IsQ0FBZjtBQUVBZSxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QlQsVUFBeEIsRUFBb0NELE1BQXBDLENBQVQ7QUFDRCxLQXhEUSxDQTBEVDs7O0FBQ0FBLElBQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDYyxhQUFWLENBQXdCVCxVQUF4QixFQUFvQ0QsTUFBcEMsQ0FBVCxDQTNEUyxDQTZEVDs7QUFDQSxRQUFJLEtBQUtuQixRQUFULEVBQW1CO0FBQ2pCLFlBQU04QixNQUFNLEdBQUdkLE1BQU0sQ0FBQ2UsSUFBUCxDQUFZLEtBQUsvQixRQUFqQixFQUEyQixNQUEzQixDQUFmO0FBRUFtQixNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBdkVRLENBeUVUOzs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBMUVTLENBNEVUOztBQUNBLFFBQUksS0FBS2xCLFFBQVQsRUFBbUI7QUFDakIsWUFBTTZCLE1BQU0sR0FBR2QsTUFBTSxDQUFDZSxJQUFQLENBQVksS0FBSzlCLFFBQWpCLEVBQTJCLE1BQTNCLENBQWY7QUFFQWtCLE1BQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDYyxhQUFWLENBQXdCQyxNQUFNLENBQUNULE1BQVAsR0FBZ0IsQ0FBeEMsRUFBMkNGLE1BQTNDLENBQVQ7QUFDQUMsTUFBQUEsVUFBVSxJQUFJVSxNQUFNLENBQUNULE1BQXJCO0FBRUFILE1BQUFBLE9BQU8sQ0FBQ2MsSUFBUixDQUFhLEtBQUtDLGdCQUFMLENBQXNCSCxNQUF0QixDQUFiO0FBQ0QsS0FQRCxNQU9PO0FBQ0xYLE1BQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDYyxhQUFWLENBQXdCLENBQXhCLEVBQTJCVixNQUEzQixDQUFUO0FBQ0QsS0F0RlEsQ0F3RlQ7OztBQUNBQSxJQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QlQsVUFBeEIsRUFBb0NELE1BQXBDLENBQVQsQ0F6RlMsQ0EyRlQ7O0FBQ0EsUUFBSSxLQUFLaEIsT0FBVCxFQUFrQjtBQUNoQixZQUFNMkIsTUFBTSxHQUFHZCxNQUFNLENBQUNlLElBQVAsQ0FBWSxLQUFLNUIsT0FBakIsRUFBMEIsTUFBMUIsQ0FBZjtBQUVBZ0IsTUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JDLE1BQU0sQ0FBQ1QsTUFBUCxHQUFnQixDQUF4QyxFQUEyQ0YsTUFBM0MsQ0FBVDtBQUNBQyxNQUFBQSxVQUFVLElBQUlVLE1BQU0sQ0FBQ1QsTUFBckI7QUFFQUgsTUFBQUEsT0FBTyxDQUFDYyxJQUFSLENBQWFGLE1BQWI7QUFDRCxLQVBELE1BT087QUFDTFgsTUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0IsQ0FBeEIsRUFBMkJWLE1BQTNCLENBQVQ7QUFDRCxLQXJHUSxDQXVHVDs7O0FBQ0FBLElBQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDYyxhQUFWLENBQXdCVCxVQUF4QixFQUFvQ0QsTUFBcEMsQ0FBVCxDQXhHUyxDQTBHVDs7QUFDQSxRQUFJLEtBQUtqQixVQUFULEVBQXFCO0FBQ25CLFlBQU00QixNQUFNLEdBQUdkLE1BQU0sQ0FBQ2UsSUFBUCxDQUFZLEtBQUs3QixVQUFqQixFQUE2QixNQUE3QixDQUFmO0FBRUFpQixNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBcEhRLENBc0hUOzs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBdkhTLENBeUhUOztBQUNBLFVBQU1lLFVBQVUsR0FBRyxLQUFLQyxlQUFMLEVBQW5CO0FBQ0FoQixJQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNBLFVBQU1pQixlQUFlLEdBQUdwQixNQUFNLENBQUNDLEtBQVAsQ0FBYSxDQUFiLENBQXhCO0FBQ0FtQixJQUFBQSxlQUFlLENBQUNkLGFBQWhCLENBQThCRixVQUFVLElBQUksQ0FBNUMsRUFBK0MsQ0FBL0M7QUFDQUEsSUFBQUEsVUFBVSxJQUFJYyxVQUFVLENBQUNiLE1BQXpCO0FBQ0FILElBQUFBLE9BQU8sQ0FBQ2MsSUFBUixDQUFhSSxlQUFiLEVBQThCRixVQUE5QixFQS9IUyxDQWlJVDs7QUFDQWYsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBbElTLENBb0lUOztBQUNBLFFBQUksS0FBS2QsV0FBVCxFQUFzQjtBQUNwQixZQUFNeUIsTUFBTSxHQUFHZCxNQUFNLENBQUNlLElBQVAsQ0FBWSxLQUFLMUIsV0FBakIsRUFBOEIsTUFBOUIsQ0FBZjtBQUVBYyxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBOUlRLENBZ0pUOzs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBakpTLENBbUpUOztBQUNBLFFBQUksS0FBS2IsUUFBVCxFQUFtQjtBQUNqQixZQUFNd0IsTUFBTSxHQUFHZCxNQUFNLENBQUNlLElBQVAsQ0FBWSxLQUFLekIsUUFBakIsRUFBMkIsTUFBM0IsQ0FBZjtBQUVBYSxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBN0pRLENBK0pUOzs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBaEtTLENBa0tUOztBQUNBLFFBQUksS0FBS1osUUFBVCxFQUFtQjtBQUNqQixZQUFNdUIsTUFBTSxHQUFHZCxNQUFNLENBQUNlLElBQVAsQ0FBWSxLQUFLeEIsUUFBakIsRUFBMkIsTUFBM0IsQ0FBZjtBQUVBWSxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBNUtRLENBOEtUOzs7QUFDQSxRQUFJLEtBQUtYLFFBQVQsRUFBbUI7QUFDakIsV0FBS0EsUUFBTCxDQUFjNkIsSUFBZCxDQUFtQnRCLFNBQW5CLEVBQThCSSxNQUE5QixFQUFzQyxDQUF0QyxFQUF5QyxDQUF6QztBQUNEOztBQUNEQSxJQUFBQSxNQUFNLElBQUksQ0FBVixDQWxMUyxDQW9MVDs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBckxTLENBdUxUOztBQUNBLFFBQUksS0FBS1YsSUFBVCxFQUFlO0FBQ2IsVUFBSSxLQUFLQSxJQUFMLENBQVVZLE1BQVYsR0FBbUIsS0FBdkIsRUFBOEI7QUFDNUJGLFFBQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDYyxhQUFWLENBQXdCLEtBQXhCLEVBQStCVixNQUEvQixDQUFUO0FBQ0QsT0FGRCxNQUVPO0FBQ0xBLFFBQUFBLE1BQU0sR0FBR0osU0FBUyxDQUFDYyxhQUFWLENBQXdCLEtBQUtwQixJQUFMLENBQVVZLE1BQWxDLEVBQTBDRixNQUExQyxDQUFUO0FBQ0Q7O0FBRURELE1BQUFBLE9BQU8sQ0FBQ2MsSUFBUixDQUFhLEtBQUt2QixJQUFsQjtBQUNELEtBUkQsTUFRTztBQUNMVSxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBbE1RLENBb01UOzs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBck1TLENBdU1UOztBQUNBLFFBQUksS0FBS1QsWUFBVCxFQUF1QjtBQUNyQixZQUFNb0IsTUFBTSxHQUFHZCxNQUFNLENBQUNlLElBQVAsQ0FBWSxLQUFLckIsWUFBakIsRUFBK0IsTUFBL0IsQ0FBZjtBQUVBUyxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBak5RLENBbU5UOzs7QUFDQUEsSUFBQUEsTUFBTSxHQUFHSixTQUFTLENBQUNjLGFBQVYsQ0FBd0JULFVBQXhCLEVBQW9DRCxNQUFwQyxDQUFULENBcE5TLENBc05UOztBQUNBLFFBQUksS0FBS1IsY0FBVCxFQUF5QjtBQUN2QixZQUFNbUIsTUFBTSxHQUFHZCxNQUFNLENBQUNlLElBQVAsQ0FBWSxLQUFLcEIsY0FBakIsRUFBaUMsTUFBakMsQ0FBZjtBQUVBUSxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QkMsTUFBTSxDQUFDVCxNQUFQLEdBQWdCLENBQXhDLEVBQTJDRixNQUEzQyxDQUFUO0FBQ0FDLE1BQUFBLFVBQVUsSUFBSVUsTUFBTSxDQUFDVCxNQUFyQjtBQUVBSCxNQUFBQSxPQUFPLENBQUNjLElBQVIsQ0FBYUYsTUFBYjtBQUNELEtBUEQsTUFPTztBQUNMWCxNQUFBQSxNQUFNLEdBQUdKLFNBQVMsQ0FBQ2MsYUFBVixDQUF3QixDQUF4QixFQUEyQlYsTUFBM0IsQ0FBVDtBQUNELEtBaE9RLENBa09UOzs7QUFDQSxRQUFJLEtBQUtWLElBQUwsSUFBYSxLQUFLQSxJQUFMLENBQVVZLE1BQVYsR0FBbUIsS0FBcEMsRUFBMkM7QUFDekNOLE1BQUFBLFNBQVMsQ0FBQ08sYUFBVixDQUF3QixLQUFLYixJQUFMLENBQVVZLE1BQWxDLEVBQTBDRixNQUExQztBQUNELEtBRkQsTUFFTztBQUNMSixNQUFBQSxTQUFTLENBQUNPLGFBQVYsQ0FBd0IsQ0FBeEIsRUFBMkJILE1BQTNCO0FBQ0Q7O0FBRUQsVUFBTW1CLElBQUksR0FBR3RCLE1BQU0sQ0FBQ3VCLE1BQVAsQ0FBY3JCLE9BQWQsQ0FBYjtBQUNBb0IsSUFBQUEsSUFBSSxDQUFDaEIsYUFBTCxDQUFtQmdCLElBQUksQ0FBQ2pCLE1BQXhCLEVBQWdDLENBQWhDO0FBQ0EsV0FBT2lCLElBQVA7QUFDRDs7QUFFRGQsRUFBQUEsaUJBQWlCLEdBQUc7QUFDbEIsUUFBSWdCLE1BQU0sR0FBR3ZHLE9BQU8sQ0FBQ0MsYUFBUixHQUF3QkQsT0FBTyxDQUFDRyxhQUFoQyxHQUFnREgsT0FBTyxDQUFDSyxjQUF4RCxHQUF5RUwsT0FBTyxDQUFDUyxnQkFBakYsR0FBb0dULE9BQU8sQ0FBQ1csVUFBNUcsR0FBeUhYLE9BQU8sQ0FBQ2UsZ0JBQTlJOztBQUNBLFFBQUksS0FBSytDLFdBQVQsRUFBc0I7QUFDcEJ5QyxNQUFBQSxNQUFNLElBQUl2RyxPQUFPLENBQUNhLGFBQWxCO0FBQ0QsS0FGRCxNQUVPO0FBQ0wwRixNQUFBQSxNQUFNLElBQUl2RyxPQUFPLENBQUNZLFlBQWxCO0FBQ0Q7O0FBQ0QsV0FBTzJGLE1BQVA7QUFDRDs7QUFFREwsRUFBQUEsZUFBZSxHQUFHO0FBQ2hCLFVBQU1qQixPQUFPLEdBQUcsRUFBaEI7QUFFQSxVQUFNTixPQUFPLEdBQUcsS0FBS0EsT0FBckI7O0FBQ0EsUUFBSUEsT0FBSixFQUFhO0FBQ1gsY0FBUUEsT0FBTyxDQUFDNkIsSUFBaEI7QUFDRSxhQUFLLE1BQUw7QUFDRSxnQkFBTVgsTUFBTSxHQUFHZCxNQUFNLENBQUNDLEtBQVAsQ0FBYSxDQUFiLENBQWY7QUFDQWEsVUFBQUEsTUFBTSxDQUFDUCxVQUFQLENBQWtCM0MsZUFBZSxDQUFDQyxVQUFsQyxFQUE4QyxDQUE5QztBQUNBaUQsVUFBQUEsTUFBTSxDQUFDUixhQUFQLENBQXFCLENBQXJCLEVBQXdCLENBQXhCO0FBQ0FRLFVBQUFBLE1BQU0sQ0FBQ1AsVUFBUCxDQUFtQjNDLGVBQWUsQ0FBQ0csWUFBaEIsSUFBZ0MsQ0FBakMsSUFBdUM2QixPQUFPLENBQUM4QixJQUFSLEdBQWU5RCxlQUFlLENBQUNJLGdCQUEvQixHQUFrREosZUFBZSxDQUFDSyxlQUF6RyxDQUFsQixFQUE2SSxDQUE3STtBQUNBNkMsVUFBQUEsTUFBTSxDQUFDUCxVQUFQLENBQWtCWCxPQUFPLENBQUMrQixRQUFSLEtBQXFCLFlBQXJCLEdBQW9DLElBQXBDLEdBQTJDL0QsZUFBZSxDQUFDTSx1QkFBN0UsRUFBc0csQ0FBdEc7QUFDQWdDLFVBQUFBLE9BQU8sQ0FBQ2MsSUFBUixDQUFhRixNQUFiO0FBQ0E7O0FBRUYsYUFBSyxlQUFMO0FBQ0UsZ0JBQU1jLEtBQUssR0FBRzVCLE1BQU0sQ0FBQ2UsSUFBUCxDQUFZbkIsT0FBTyxDQUFDaUMsWUFBcEIsRUFBa0MsTUFBbEMsQ0FBZDtBQUNBLGdCQUFNQyxHQUFHLEdBQUc5QixNQUFNLENBQUNDLEtBQVAsQ0FBYSxFQUFiLENBQVo7QUFFQSxjQUFJRSxNQUFNLEdBQUcsQ0FBYjtBQUNBQSxVQUFBQSxNQUFNLEdBQUcyQixHQUFHLENBQUN2QixVQUFKLENBQWUzQyxlQUFlLENBQUNDLFVBQS9CLEVBQTJDc0MsTUFBM0MsQ0FBVDtBQUNBQSxVQUFBQSxNQUFNLEdBQUcyQixHQUFHLENBQUN4QixhQUFKLENBQWtCc0IsS0FBSyxDQUFDdkIsTUFBTixHQUFlLENBQWYsR0FBbUIsQ0FBckMsRUFBd0NGLE1BQXhDLENBQVQ7QUFDQUEsVUFBQUEsTUFBTSxHQUFHMkIsR0FBRyxDQUFDdkIsVUFBSixDQUFnQjNDLGVBQWUsQ0FBQ0UscUJBQWhCLElBQXlDLENBQTFDLElBQWdEOEIsT0FBTyxDQUFDOEIsSUFBUixHQUFlOUQsZUFBZSxDQUFDSSxnQkFBL0IsR0FBa0RKLGVBQWUsQ0FBQ0ssZUFBbEgsQ0FBZixFQUFtSmtDLE1BQW5KLENBQVQ7QUFDQTJCLFVBQUFBLEdBQUcsQ0FBQ2xCLFlBQUosQ0FBaUJnQixLQUFLLENBQUN2QixNQUF2QixFQUErQkYsTUFBL0I7QUFFQUQsVUFBQUEsT0FBTyxDQUFDYyxJQUFSLENBQWFjLEdBQWI7QUFDQTVCLFVBQUFBLE9BQU8sQ0FBQ2MsSUFBUixDQUFhWSxLQUFiO0FBRUE7QUF2Qko7QUF5QkQ7O0FBRUQsUUFBSSxLQUFLckQsVUFBTCxJQUFtQndELHNCQUFTLEtBQVQsQ0FBdkIsRUFBd0M7QUFDdEM7QUFDQSxZQUFNQyx1QkFBdUIsR0FBRyxJQUFoQztBQUNBLFlBQU1DLGlDQUFpQyxHQUFHLElBQTFDO0FBQ0EsWUFBTUgsR0FBRyxHQUFHOUIsTUFBTSxDQUFDQyxLQUFQLENBQWEsQ0FBYixDQUFaO0FBQ0E2QixNQUFBQSxHQUFHLENBQUN2QixVQUFKLENBQWV5Qix1QkFBZixFQUF3QyxDQUF4QztBQUNBRixNQUFBQSxHQUFHLENBQUN4QixhQUFKLENBQWtCLENBQWxCLEVBQXFCLENBQXJCO0FBQ0F3QixNQUFBQSxHQUFHLENBQUN2QixVQUFKLENBQWUwQixpQ0FBZixFQUFrRCxDQUFsRDtBQUNBL0IsTUFBQUEsT0FBTyxDQUFDYyxJQUFSLENBQWFjLEdBQWI7QUFDRDs7QUFFRDVCLElBQUFBLE9BQU8sQ0FBQ2MsSUFBUixDQUFhaEIsTUFBTSxDQUFDZSxJQUFQLENBQVksQ0FBQzNDLHNCQUFELENBQVosQ0FBYjtBQUVBLFdBQU80QixNQUFNLENBQUN1QixNQUFQLENBQWNyQixPQUFkLENBQVA7QUFDRDs7QUFFRE8sRUFBQUEsaUJBQWlCLEdBQUc7QUFDbEIsUUFBSXlCLE1BQU0sR0FBR2pHLE9BQU8sQ0FBQ0MsY0FBUixHQUF5QkQsT0FBTyxDQUFDRyxRQUFqQyxHQUE0Q0gsT0FBTyxDQUFDTyxXQUFqRTs7QUFDQSxRQUFJLEtBQUtpRCxJQUFULEVBQWU7QUFDYnlDLE1BQUFBLE1BQU0sSUFBSWpHLE9BQU8sQ0FBQ1ksc0JBQWxCO0FBQ0QsS0FGRCxNQUVPO0FBQ0xxRixNQUFBQSxNQUFNLElBQUlqRyxPQUFPLENBQUNXLHVCQUFsQjtBQUNEOztBQUNELFdBQU9zRixNQUFQO0FBQ0Q7O0FBRUR4QixFQUFBQSxjQUFjLEdBQUc7QUFDZixRQUFJeUIsU0FBUyxHQUFHckYsVUFBVSxDQUFDQyxRQUFYLEdBQXNCRCxVQUFVLENBQUNHLFNBQWpEOztBQUNBLFFBQUksS0FBSzZCLGNBQVQsRUFBeUI7QUFDdkJxRCxNQUFBQSxTQUFTLElBQUlyRixVQUFVLENBQUNNLGdCQUF4QjtBQUNELEtBRkQsTUFFTztBQUNMK0UsTUFBQUEsU0FBUyxJQUFJckYsVUFBVSxDQUFDSyxpQkFBeEI7QUFDRDs7QUFDRCxXQUFPZ0YsU0FBUDtBQUNEOztBQUVEeEIsRUFBQUEsaUJBQWlCLEdBQUc7QUFDbEIsV0FBT3RELE9BQU8sQ0FBQ0Msa0JBQVIsR0FBNkJELE9BQU8sQ0FBQ0ssMEJBQXJDLEdBQWtFTCxPQUFPLENBQUNNLGNBQWpGO0FBQ0Q7O0FBRURzRCxFQUFBQSxnQkFBZ0IsQ0FBQ2hDLFFBQUQsRUFBbUI7QUFDakMsU0FBSyxJQUFJbUQsQ0FBQyxHQUFHLENBQVIsRUFBV0MsR0FBRyxHQUFHcEQsUUFBUSxDQUFDb0IsTUFBL0IsRUFBdUMrQixDQUFDLEdBQUdDLEdBQTNDLEVBQWdERCxDQUFDLEVBQWpELEVBQXFEO0FBQ25ELFVBQUlFLElBQUksR0FBR3JELFFBQVEsQ0FBQ21ELENBQUQsQ0FBbkI7QUFDQSxZQUFNRyxTQUFTLEdBQUdELElBQUksR0FBRyxJQUF6QjtBQUNBLFlBQU1FLFVBQVUsR0FBR0YsSUFBSSxJQUFJLENBQTNCO0FBQ0FBLE1BQUFBLElBQUksR0FBSUMsU0FBUyxJQUFJLENBQWQsR0FBbUJDLFVBQTFCO0FBQ0FGLE1BQUFBLElBQUksR0FBR0EsSUFBSSxHQUFHLElBQWQ7QUFDQXJELE1BQUFBLFFBQVEsQ0FBQ21ELENBQUQsQ0FBUixHQUFjRSxJQUFkO0FBQ0Q7O0FBQ0QsV0FBT3JELFFBQVA7QUFDRDs7QUFFRHdELEVBQUFBLFFBQVEsQ0FBQ0MsTUFBTSxHQUFHLEVBQVYsRUFBYztBQUNwQixXQUFPQSxNQUFNLEdBQUcsV0FBVCxHQUNMLHdCQUFRLDRGQUFSLEVBQ1EsS0FBS25FLFVBRGIsRUFDeUIsS0FBS0MsVUFEOUIsRUFDMEMsS0FBS0MsYUFEL0MsRUFDOEQsS0FBS0MsU0FEbkUsRUFDOEUsS0FBS0MsWUFEbkYsQ0FESyxHQUdELElBSEMsR0FHTStELE1BSE4sR0FHZSxXQUhmLEdBSUwsd0JBQVEscUdBQVIsRUFDUSxLQUFLbEMsaUJBQUwsRUFEUixFQUNrQyxLQUFLQyxpQkFBTCxFQURsQyxFQUM0RCxLQUFLQyxjQUFMLEVBRDVELEVBQ21GLEtBQUtDLGlCQUFMLEVBRG5GLEVBQzZHLEtBQUsvQixjQURsSCxFQUNrSSxLQUFLQyxVQUR2SSxDQUpLLEdBTUQsSUFOQyxHQU1NNkQsTUFOTixHQU1lLFdBTmYsR0FPTCx3QkFBUSw4RkFBUixFQUNRLEtBQUt0RCxRQURiLEVBQ3VCLEtBQUtKLFFBRDVCLEVBQ3NDLEtBQUtDLFFBRDNDLEVBQ3FELEtBQUtFLE9BRDFELEVBQ21FLEtBQUtELFVBRHhFLEVBQ29GLEtBQUtHLFdBRHpGLENBUEssR0FTRCxJQVRDLEdBU01xRCxNQVROLEdBU2UsV0FUZixHQVVMLHdCQUFRLGlGQUFSLEVBQ1EsS0FBS3BELFFBRGIsRUFDdUIsS0FBS0MsUUFENUIsRUFDc0MsS0FBS0UsSUFEM0MsRUFDaUQsS0FBS0MsWUFEdEQsRUFDb0UsS0FBS0MsY0FEekUsQ0FWRjtBQWFEOztBQWpaaUI7O2VBb1pMdEIsYTs7QUFDZnNFLE1BQU0sQ0FBQ0MsT0FBUCxHQUFpQnZFLGFBQWpCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgc3ByaW50ZiB9IGZyb20gJ3NwcmludGYtanMnO1xuaW1wb3J0IHsgdmVyc2lvbnMgfSBmcm9tICcuL3Rkcy12ZXJzaW9ucyc7XG5cbmNvbnN0IEZMQUdTXzEgPSB7XG4gIEVORElBTl9MSVRUTEU6IDB4MDAsXG4gIEVORElBTl9CSUc6IDB4MDEsXG4gIENIQVJTRVRfQVNDSUk6IDB4MDAsXG4gIENIQVJTRVRfRUJDRElDOiAweDAyLFxuICBGTE9BVF9JRUVFXzc1NDogMHgwMCxcbiAgRkxPQVRfVkFYOiAweDA0LFxuICBGTE9BVF9ORDUwMDA6IDB4MDgsXG4gIEJDUF9EVU1QTE9BRF9PTjogMHgwMCxcbiAgQkNQX0RVTVBMT0FEX09GRjogMHgxMCxcbiAgVVNFX0RCX09OOiAweDAwLFxuICBVU0VfREJfT0ZGOiAweDIwLFxuICBJTklUX0RCX1dBUk46IDB4MDAsXG4gIElOSVRfREJfRkFUQUw6IDB4NDAsXG4gIFNFVF9MQU5HX1dBUk5fT0ZGOiAweDAwLFxuICBTRVRfTEFOR19XQVJOX09OOiAweDgwXG59O1xuXG5jb25zdCBGTEFHU18yID0ge1xuICBJTklUX0xBTkdfV0FSTjogMHgwMCxcbiAgSU5JVF9MQU5HX0ZBVEFMOiAweDAxLFxuICBPREJDX09GRjogMHgwMCxcbiAgT0RCQ19PTjogMHgwMixcbiAgRl9UUkFOX0JPVU5EQVJZOiAweDA0LFxuICBGX0NBQ0hFX0NPTk5FQ1Q6IDB4MDgsXG4gIFVTRVJfTk9STUFMOiAweDAwLFxuICBVU0VSX1NFUlZFUjogMHgxMCxcbiAgVVNFUl9SRU1VU0VSOiAweDIwLFxuICBVU0VSX1NRTFJFUEw6IDB4NDAsXG4gIElOVEVHUkFURURfU0VDVVJJVFlfT0ZGOiAweDAwLFxuICBJTlRFR1JBVEVEX1NFQ1VSSVRZX09OOiAweDgwXG59O1xuXG5jb25zdCBUWVBFX0ZMQUdTID0ge1xuICBTUUxfREZMVDogMHgwMCxcbiAgU1FMX1RTUUw6IDB4MDgsXG4gIE9MRURCX09GRjogMHgwMCxcbiAgT0xFREJfT046IDB4MTAsXG4gIFJFQURfV1JJVEVfSU5URU5UOiAweDAwLFxuICBSRUFEX09OTFlfSU5URU5UOiAweDIwXG59O1xuXG5jb25zdCBGTEFHU18zID0ge1xuICBDSEFOR0VfUEFTU1dPUkRfTk86IDB4MDAsXG4gIENIQU5HRV9QQVNTV09SRF9ZRVM6IDB4MDEsXG4gIEJJTkFSWV9YTUw6IDB4MDIsXG4gIFNQQVdOX1VTRVJfSU5TVEFOQ0U6IDB4MDQsXG4gIFVOS05PV05fQ09MTEFUSU9OX0hBTkRMSU5HOiAweDA4LFxuICBFWFRFTlNJT05fVVNFRDogMHgxMFxufTtcblxuY29uc3QgRkVEQVVUSF9PUFRJT05TID0ge1xuICBGRUFUVVJFX0lEOiAweDAyLFxuICBMSUJSQVJZX1NFQ1VSSVRZVE9LRU46IDB4MDEsXG4gIExJQlJBUllfQURBTDogMHgwMixcbiAgRkVEQVVUSF9ZRVNfRUNITzogMHgwMSxcbiAgRkVEQVVUSF9OT19FQ0hPOiAweDAwLFxuICBBREFMX1dPUktGTE9XX1VTRVJfUEFTUzogMHgwMSxcbiAgQURBTF9XT1JLRkxPV19JTlRFR1JBVEVEOiAweDAyXG59O1xuXG5jb25zdCBGRUFUVVJFX0VYVF9URVJNSU5BVE9SID0gMHhGRjtcblxuaW50ZXJmYWNlIE9wdGlvbnMge1xuICB0ZHNWZXJzaW9uOiBudW1iZXI7XG4gIHBhY2tldFNpemU6IG51bWJlcjtcbiAgY2xpZW50UHJvZ1ZlcjogbnVtYmVyO1xuICBjbGllbnRQaWQ6IG51bWJlcjtcbiAgY29ubmVjdGlvbklkOiBudW1iZXI7XG4gIGNsaWVudFRpbWVab25lOiBudW1iZXI7XG4gIGNsaWVudExjaWQ6IG51bWJlcjtcbn1cblxuLypcbiAgczIuMi42LjNcbiAqL1xuY2xhc3MgTG9naW43UGF5bG9hZCB7XG4gIHRkc1ZlcnNpb246IG51bWJlcjtcbiAgcGFja2V0U2l6ZTogbnVtYmVyO1xuICBjbGllbnRQcm9nVmVyOiBudW1iZXI7XG4gIGNsaWVudFBpZDogbnVtYmVyO1xuICBjb25uZWN0aW9uSWQ6IG51bWJlcjtcbiAgY2xpZW50VGltZVpvbmU6IG51bWJlcjtcbiAgY2xpZW50TGNpZDogbnVtYmVyO1xuXG4gIHJlYWRPbmx5SW50ZW50OiBib29sZWFuO1xuICBpbml0RGJGYXRhbDogYm9vbGVhbjtcblxuICB1c2VyTmFtZTogc3RyaW5nIHwgdW5kZWZpbmVkO1xuICBwYXNzd29yZDogc3RyaW5nIHwgdW5kZWZpbmVkO1xuICBzZXJ2ZXJOYW1lOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gIGFwcE5hbWU6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgaG9zdG5hbWU6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgbGlicmFyeU5hbWU6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgbGFuZ3VhZ2U6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgZGF0YWJhc2U6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgY2xpZW50SWQ6IEJ1ZmZlciB8IHVuZGVmaW5lZDtcbiAgc3NwaTogQnVmZmVyIHwgdW5kZWZpbmVkO1xuICBhdHRhY2hEYkZpbGU6IHN0cmluZyB8IHVuZGVmaW5lZDtcbiAgY2hhbmdlUGFzc3dvcmQ6IHN0cmluZyB8IHVuZGVmaW5lZDtcblxuICBmZWRBdXRoOiB7IHR5cGU6ICdBREFMJywgZWNobzogYm9vbGVhbiwgd29ya2Zsb3c6ICdkZWZhdWx0JyB8ICdpbnRlZ3JhdGVkJyB9IHwgeyB0eXBlOiAnU0VDVVJJVFlUT0tFTicsIGVjaG86IGJvb2xlYW4sIGZlZEF1dGhUb2tlbjogc3RyaW5nIH0gfCB1bmRlZmluZWQ7XG5cbiAgY29uc3RydWN0b3IoeyB0ZHNWZXJzaW9uLCBwYWNrZXRTaXplLCBjbGllbnRQcm9nVmVyLCBjbGllbnRQaWQsIGNvbm5lY3Rpb25JZCwgY2xpZW50VGltZVpvbmUsIGNsaWVudExjaWQgfTogT3B0aW9ucykge1xuICAgIHRoaXMudGRzVmVyc2lvbiA9IHRkc1ZlcnNpb247XG4gICAgdGhpcy5wYWNrZXRTaXplID0gcGFja2V0U2l6ZTtcbiAgICB0aGlzLmNsaWVudFByb2dWZXIgPSBjbGllbnRQcm9nVmVyO1xuICAgIHRoaXMuY2xpZW50UGlkID0gY2xpZW50UGlkO1xuICAgIHRoaXMuY29ubmVjdGlvbklkID0gY29ubmVjdGlvbklkO1xuICAgIHRoaXMuY2xpZW50VGltZVpvbmUgPSBjbGllbnRUaW1lWm9uZTtcbiAgICB0aGlzLmNsaWVudExjaWQgPSBjbGllbnRMY2lkO1xuXG4gICAgdGhpcy5yZWFkT25seUludGVudCA9IGZhbHNlO1xuICAgIHRoaXMuaW5pdERiRmF0YWwgPSBmYWxzZTtcblxuICAgIHRoaXMuZmVkQXV0aCA9IHVuZGVmaW5lZDtcblxuICAgIHRoaXMudXNlck5hbWUgPSB1bmRlZmluZWQ7XG4gICAgdGhpcy5wYXNzd29yZCA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLnNlcnZlck5hbWUgPSB1bmRlZmluZWQ7XG4gICAgdGhpcy5hcHBOYW1lID0gdW5kZWZpbmVkO1xuICAgIHRoaXMuaG9zdG5hbWUgPSB1bmRlZmluZWQ7XG4gICAgdGhpcy5saWJyYXJ5TmFtZSA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLmxhbmd1YWdlID0gdW5kZWZpbmVkO1xuICAgIHRoaXMuZGF0YWJhc2UgPSB1bmRlZmluZWQ7XG4gICAgdGhpcy5jbGllbnRJZCA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLnNzcGkgPSB1bmRlZmluZWQ7XG4gICAgdGhpcy5hdHRhY2hEYkZpbGUgPSB1bmRlZmluZWQ7XG4gICAgdGhpcy5jaGFuZ2VQYXNzd29yZCA9IHVuZGVmaW5lZDtcbiAgfVxuXG4gIHRvQnVmZmVyKCkge1xuICAgIGNvbnN0IGZpeGVkRGF0YSA9IEJ1ZmZlci5hbGxvYyg5NCk7XG4gICAgY29uc3QgYnVmZmVycyA9IFtmaXhlZERhdGFdO1xuXG4gICAgbGV0IG9mZnNldCA9IDA7XG4gICAgbGV0IGRhdGFPZmZzZXQgPSBmaXhlZERhdGEubGVuZ3RoO1xuXG4gICAgLy8gTGVuZ3RoOiA0LWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MzJMRSgwLCBvZmZzZXQpO1xuXG4gICAgLy8gVERTVmVyc2lvbjogNC1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDMyTEUodGhpcy50ZHNWZXJzaW9uLCBvZmZzZXQpO1xuXG4gICAgLy8gUGFja2V0U2l6ZTogNC1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDMyTEUodGhpcy5wYWNrZXRTaXplLCBvZmZzZXQpO1xuXG4gICAgLy8gQ2xpZW50UHJvZ1ZlcjogNC1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDMyTEUodGhpcy5jbGllbnRQcm9nVmVyLCBvZmZzZXQpO1xuXG4gICAgLy8gQ2xpZW50UElEOiA0LWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MzJMRSh0aGlzLmNsaWVudFBpZCwgb2Zmc2V0KTtcblxuICAgIC8vIENvbm5lY3Rpb25JRDogNC1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDMyTEUodGhpcy5jb25uZWN0aW9uSWQsIG9mZnNldCk7XG5cbiAgICAvLyBPcHRpb25GbGFnczE6IDEtYnl0ZVxuICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQ4KHRoaXMuYnVpbGRPcHRpb25GbGFnczEoKSwgb2Zmc2V0KTtcblxuICAgIC8vIE9wdGlvbkZsYWdzMjogMS1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDgodGhpcy5idWlsZE9wdGlvbkZsYWdzMigpLCBvZmZzZXQpO1xuXG4gICAgLy8gVHlwZUZsYWdzOiAxLWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50OCh0aGlzLmJ1aWxkVHlwZUZsYWdzKCksIG9mZnNldCk7XG5cbiAgICAvLyBPcHRpb25GbGFnczM6IDEtYnl0ZVxuICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQ4KHRoaXMuYnVpbGRPcHRpb25GbGFnczMoKSwgb2Zmc2V0KTtcblxuICAgIC8vIENsaWVudFRpbVpvbmU6IDQtYnl0ZVxuICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZUludDMyTEUodGhpcy5jbGllbnRUaW1lWm9uZSwgb2Zmc2V0KTtcblxuICAgIC8vIENsaWVudExDSUQ6IDQtYnl0ZVxuICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQzMkxFKHRoaXMuY2xpZW50TGNpZCwgb2Zmc2V0KTtcblxuICAgIC8vIGliSG9zdE5hbWU6IDItYnl0ZVxuICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKGRhdGFPZmZzZXQsIG9mZnNldCk7XG5cbiAgICAvLyBjY2hIb3N0TmFtZTogMi1ieXRlXG4gICAgaWYgKHRoaXMuaG9zdG5hbWUpIHtcbiAgICAgIGNvbnN0IGJ1ZmZlciA9IEJ1ZmZlci5mcm9tKHRoaXMuaG9zdG5hbWUsICd1Y3MyJyk7XG5cbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKGJ1ZmZlci5sZW5ndGggLyAyLCBvZmZzZXQpO1xuICAgICAgZGF0YU9mZnNldCArPSBidWZmZXIubGVuZ3RoO1xuXG4gICAgICBidWZmZXJzLnB1c2goYnVmZmVyKTtcbiAgICB9IGVsc2Uge1xuICAgICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoZGF0YU9mZnNldCwgb2Zmc2V0KTtcbiAgICB9XG5cbiAgICAvLyBpYlVzZXJOYW1lOiAyLWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShkYXRhT2Zmc2V0LCBvZmZzZXQpO1xuXG4gICAgLy8gY2NoVXNlck5hbWU6IDItYnl0ZVxuICAgIGlmICh0aGlzLnVzZXJOYW1lKSB7XG4gICAgICBjb25zdCBidWZmZXIgPSBCdWZmZXIuZnJvbSh0aGlzLnVzZXJOYW1lLCAndWNzMicpO1xuXG4gICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShidWZmZXIubGVuZ3RoIC8gMiwgb2Zmc2V0KTtcbiAgICAgIGRhdGFPZmZzZXQgKz0gYnVmZmVyLmxlbmd0aDtcblxuICAgICAgYnVmZmVycy5wdXNoKGJ1ZmZlcik7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKDAsIG9mZnNldCk7XG4gICAgfVxuXG4gICAgLy8gaWJQYXNzd29yZDogMi1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoZGF0YU9mZnNldCwgb2Zmc2V0KTtcblxuICAgIC8vIGNjaFBhc3N3b3JkOiAyLWJ5dGVcbiAgICBpZiAodGhpcy5wYXNzd29yZCkge1xuICAgICAgY29uc3QgYnVmZmVyID0gQnVmZmVyLmZyb20odGhpcy5wYXNzd29yZCwgJ3VjczInKTtcblxuICAgICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoYnVmZmVyLmxlbmd0aCAvIDIsIG9mZnNldCk7XG4gICAgICBkYXRhT2Zmc2V0ICs9IGJ1ZmZlci5sZW5ndGg7XG5cbiAgICAgIGJ1ZmZlcnMucHVzaCh0aGlzLnNjcmFtYmxlUGFzc3dvcmQoYnVmZmVyKSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKDAsIG9mZnNldCk7XG4gICAgfVxuXG4gICAgLy8gaWJBcHBOYW1lOiAyLWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShkYXRhT2Zmc2V0LCBvZmZzZXQpO1xuXG4gICAgLy8gY2NoQXBwTmFtZTogMi1ieXRlXG4gICAgaWYgKHRoaXMuYXBwTmFtZSkge1xuICAgICAgY29uc3QgYnVmZmVyID0gQnVmZmVyLmZyb20odGhpcy5hcHBOYW1lLCAndWNzMicpO1xuXG4gICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShidWZmZXIubGVuZ3RoIC8gMiwgb2Zmc2V0KTtcbiAgICAgIGRhdGFPZmZzZXQgKz0gYnVmZmVyLmxlbmd0aDtcblxuICAgICAgYnVmZmVycy5wdXNoKGJ1ZmZlcik7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKDAsIG9mZnNldCk7XG4gICAgfVxuXG4gICAgLy8gaWJTZXJ2ZXJOYW1lOiAyLWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShkYXRhT2Zmc2V0LCBvZmZzZXQpO1xuXG4gICAgLy8gY2NoU2VydmVyTmFtZTogMi1ieXRlXG4gICAgaWYgKHRoaXMuc2VydmVyTmFtZSkge1xuICAgICAgY29uc3QgYnVmZmVyID0gQnVmZmVyLmZyb20odGhpcy5zZXJ2ZXJOYW1lLCAndWNzMicpO1xuXG4gICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShidWZmZXIubGVuZ3RoIC8gMiwgb2Zmc2V0KTtcbiAgICAgIGRhdGFPZmZzZXQgKz0gYnVmZmVyLmxlbmd0aDtcblxuICAgICAgYnVmZmVycy5wdXNoKGJ1ZmZlcik7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKDAsIG9mZnNldCk7XG4gICAgfVxuXG4gICAgLy8gKGliVW51c2VkIC8gaWJFeHRlbnNpb24pOiAyLWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShkYXRhT2Zmc2V0LCBvZmZzZXQpO1xuXG4gICAgLy8gKGNjaFVudXNlZCAvIGNiRXh0ZW5zaW9uKTogMi1ieXRlXG4gICAgY29uc3QgZXh0ZW5zaW9ucyA9IHRoaXMuYnVpbGRGZWF0dXJlRXh0KCk7XG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoNCwgb2Zmc2V0KTtcbiAgICBjb25zdCBleHRlbnNpb25PZmZzZXQgPSBCdWZmZXIuYWxsb2MoNCk7XG4gICAgZXh0ZW5zaW9uT2Zmc2V0LndyaXRlVUludDMyTEUoZGF0YU9mZnNldCArPSA0LCAwKTtcbiAgICBkYXRhT2Zmc2V0ICs9IGV4dGVuc2lvbnMubGVuZ3RoO1xuICAgIGJ1ZmZlcnMucHVzaChleHRlbnNpb25PZmZzZXQsIGV4dGVuc2lvbnMpO1xuXG4gICAgLy8gaWJDbHRJbnROYW1lOiAyLWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShkYXRhT2Zmc2V0LCBvZmZzZXQpO1xuXG4gICAgLy8gY2NoQ2x0SW50TmFtZTogMi1ieXRlXG4gICAgaWYgKHRoaXMubGlicmFyeU5hbWUpIHtcbiAgICAgIGNvbnN0IGJ1ZmZlciA9IEJ1ZmZlci5mcm9tKHRoaXMubGlicmFyeU5hbWUsICd1Y3MyJyk7XG5cbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKGJ1ZmZlci5sZW5ndGggLyAyLCBvZmZzZXQpO1xuICAgICAgZGF0YU9mZnNldCArPSBidWZmZXIubGVuZ3RoO1xuXG4gICAgICBidWZmZXJzLnB1c2goYnVmZmVyKTtcbiAgICB9IGVsc2Uge1xuICAgICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoMCwgb2Zmc2V0KTtcbiAgICB9XG5cbiAgICAvLyBpYkxhbmd1YWdlOiAyLWJ5dGVcbiAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShkYXRhT2Zmc2V0LCBvZmZzZXQpO1xuXG4gICAgLy8gY2NoTGFuZ3VhZ2U6IDItYnl0ZVxuICAgIGlmICh0aGlzLmxhbmd1YWdlKSB7XG4gICAgICBjb25zdCBidWZmZXIgPSBCdWZmZXIuZnJvbSh0aGlzLmxhbmd1YWdlLCAndWNzMicpO1xuXG4gICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShidWZmZXIubGVuZ3RoIC8gMiwgb2Zmc2V0KTtcbiAgICAgIGRhdGFPZmZzZXQgKz0gYnVmZmVyLmxlbmd0aDtcblxuICAgICAgYnVmZmVycy5wdXNoKGJ1ZmZlcik7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKDAsIG9mZnNldCk7XG4gICAgfVxuXG4gICAgLy8gaWJEYXRhYmFzZTogMi1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoZGF0YU9mZnNldCwgb2Zmc2V0KTtcblxuICAgIC8vIGNjaERhdGFiYXNlOiAyLWJ5dGVcbiAgICBpZiAodGhpcy5kYXRhYmFzZSkge1xuICAgICAgY29uc3QgYnVmZmVyID0gQnVmZmVyLmZyb20odGhpcy5kYXRhYmFzZSwgJ3VjczInKTtcblxuICAgICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoYnVmZmVyLmxlbmd0aCAvIDIsIG9mZnNldCk7XG4gICAgICBkYXRhT2Zmc2V0ICs9IGJ1ZmZlci5sZW5ndGg7XG5cbiAgICAgIGJ1ZmZlcnMucHVzaChidWZmZXIpO1xuICAgIH0gZWxzZSB7XG4gICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRSgwLCBvZmZzZXQpO1xuICAgIH1cblxuICAgIC8vIENsaWVudElEOiA2LWJ5dGVcbiAgICBpZiAodGhpcy5jbGllbnRJZCkge1xuICAgICAgdGhpcy5jbGllbnRJZC5jb3B5KGZpeGVkRGF0YSwgb2Zmc2V0LCAwLCA2KTtcbiAgICB9XG4gICAgb2Zmc2V0ICs9IDY7XG5cbiAgICAvLyBpYlNTUEk6IDItYnl0ZVxuICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKGRhdGFPZmZzZXQsIG9mZnNldCk7XG5cbiAgICAvLyBjYlNTUEk6IDItYnl0ZVxuICAgIGlmICh0aGlzLnNzcGkpIHtcbiAgICAgIGlmICh0aGlzLnNzcGkubGVuZ3RoID4gNjU1MzUpIHtcbiAgICAgICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoNjU1MzUsIG9mZnNldCk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRSh0aGlzLnNzcGkubGVuZ3RoLCBvZmZzZXQpO1xuICAgICAgfVxuXG4gICAgICBidWZmZXJzLnB1c2godGhpcy5zc3BpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoMCwgb2Zmc2V0KTtcbiAgICB9XG5cbiAgICAvLyBpYkF0Y2hEQkZpbGU6IDItYnl0ZVxuICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKGRhdGFPZmZzZXQsIG9mZnNldCk7XG5cbiAgICAvLyBjY2hBdGNoREJGaWxlOiAyLWJ5dGVcbiAgICBpZiAodGhpcy5hdHRhY2hEYkZpbGUpIHtcbiAgICAgIGNvbnN0IGJ1ZmZlciA9IEJ1ZmZlci5mcm9tKHRoaXMuYXR0YWNoRGJGaWxlLCAndWNzMicpO1xuXG4gICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRShidWZmZXIubGVuZ3RoIC8gMiwgb2Zmc2V0KTtcbiAgICAgIGRhdGFPZmZzZXQgKz0gYnVmZmVyLmxlbmd0aDtcblxuICAgICAgYnVmZmVycy5wdXNoKGJ1ZmZlcik7XG4gICAgfSBlbHNlIHtcbiAgICAgIG9mZnNldCA9IGZpeGVkRGF0YS53cml0ZVVJbnQxNkxFKDAsIG9mZnNldCk7XG4gICAgfVxuXG4gICAgLy8gaWJDaGFuZ2VQYXNzd29yZDogMi1ieXRlXG4gICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoZGF0YU9mZnNldCwgb2Zmc2V0KTtcblxuICAgIC8vIGNjaENoYW5nZVBhc3N3b3JkOiAyLWJ5dGVcbiAgICBpZiAodGhpcy5jaGFuZ2VQYXNzd29yZCkge1xuICAgICAgY29uc3QgYnVmZmVyID0gQnVmZmVyLmZyb20odGhpcy5jaGFuZ2VQYXNzd29yZCwgJ3VjczInKTtcblxuICAgICAgb2Zmc2V0ID0gZml4ZWREYXRhLndyaXRlVUludDE2TEUoYnVmZmVyLmxlbmd0aCAvIDIsIG9mZnNldCk7XG4gICAgICBkYXRhT2Zmc2V0ICs9IGJ1ZmZlci5sZW5ndGg7XG5cbiAgICAgIGJ1ZmZlcnMucHVzaChidWZmZXIpO1xuICAgIH0gZWxzZSB7XG4gICAgICBvZmZzZXQgPSBmaXhlZERhdGEud3JpdGVVSW50MTZMRSgwLCBvZmZzZXQpO1xuICAgIH1cblxuICAgIC8vIGNiU1NQSUxvbmc6IDQtYnl0ZVxuICAgIGlmICh0aGlzLnNzcGkgJiYgdGhpcy5zc3BpLmxlbmd0aCA+IDY1NTM1KSB7XG4gICAgICBmaXhlZERhdGEud3JpdGVVSW50MzJMRSh0aGlzLnNzcGkubGVuZ3RoLCBvZmZzZXQpO1xuICAgIH0gZWxzZSB7XG4gICAgICBmaXhlZERhdGEud3JpdGVVSW50MzJMRSgwLCBvZmZzZXQpO1xuICAgIH1cblxuICAgIGNvbnN0IGRhdGEgPSBCdWZmZXIuY29uY2F0KGJ1ZmZlcnMpO1xuICAgIGRhdGEud3JpdGVVSW50MzJMRShkYXRhLmxlbmd0aCwgMCk7XG4gICAgcmV0dXJuIGRhdGE7XG4gIH1cblxuICBidWlsZE9wdGlvbkZsYWdzMSgpIHtcbiAgICBsZXQgZmxhZ3MxID0gRkxBR1NfMS5FTkRJQU5fTElUVExFIHwgRkxBR1NfMS5DSEFSU0VUX0FTQ0lJIHwgRkxBR1NfMS5GTE9BVF9JRUVFXzc1NCB8IEZMQUdTXzEuQkNQX0RVTVBMT0FEX09GRiB8IEZMQUdTXzEuVVNFX0RCX09GRiB8IEZMQUdTXzEuU0VUX0xBTkdfV0FSTl9PTjtcbiAgICBpZiAodGhpcy5pbml0RGJGYXRhbCkge1xuICAgICAgZmxhZ3MxIHw9IEZMQUdTXzEuSU5JVF9EQl9GQVRBTDtcbiAgICB9IGVsc2Uge1xuICAgICAgZmxhZ3MxIHw9IEZMQUdTXzEuSU5JVF9EQl9XQVJOO1xuICAgIH1cbiAgICByZXR1cm4gZmxhZ3MxO1xuICB9XG5cbiAgYnVpbGRGZWF0dXJlRXh0KCkge1xuICAgIGNvbnN0IGJ1ZmZlcnMgPSBbXTtcblxuICAgIGNvbnN0IGZlZEF1dGggPSB0aGlzLmZlZEF1dGg7XG4gICAgaWYgKGZlZEF1dGgpIHtcbiAgICAgIHN3aXRjaCAoZmVkQXV0aC50eXBlKSB7XG4gICAgICAgIGNhc2UgJ0FEQUwnOlxuICAgICAgICAgIGNvbnN0IGJ1ZmZlciA9IEJ1ZmZlci5hbGxvYyg3KTtcbiAgICAgICAgICBidWZmZXIud3JpdGVVSW50OChGRURBVVRIX09QVElPTlMuRkVBVFVSRV9JRCwgMCk7XG4gICAgICAgICAgYnVmZmVyLndyaXRlVUludDMyTEUoMiwgMSk7XG4gICAgICAgICAgYnVmZmVyLndyaXRlVUludDgoKEZFREFVVEhfT1BUSU9OUy5MSUJSQVJZX0FEQUwgPDwgMSkgfCAoZmVkQXV0aC5lY2hvID8gRkVEQVVUSF9PUFRJT05TLkZFREFVVEhfWUVTX0VDSE8gOiBGRURBVVRIX09QVElPTlMuRkVEQVVUSF9OT19FQ0hPKSwgNSk7XG4gICAgICAgICAgYnVmZmVyLndyaXRlVUludDgoZmVkQXV0aC53b3JrZmxvdyA9PT0gJ2ludGVncmF0ZWQnID8gMHgwMiA6IEZFREFVVEhfT1BUSU9OUy5BREFMX1dPUktGTE9XX1VTRVJfUEFTUywgNik7XG4gICAgICAgICAgYnVmZmVycy5wdXNoKGJ1ZmZlcik7XG4gICAgICAgICAgYnJlYWs7XG5cbiAgICAgICAgY2FzZSAnU0VDVVJJVFlUT0tFTic6XG4gICAgICAgICAgY29uc3QgdG9rZW4gPSBCdWZmZXIuZnJvbShmZWRBdXRoLmZlZEF1dGhUb2tlbiwgJ3VjczInKTtcbiAgICAgICAgICBjb25zdCBidWYgPSBCdWZmZXIuYWxsb2MoMTApO1xuXG4gICAgICAgICAgbGV0IG9mZnNldCA9IDA7XG4gICAgICAgICAgb2Zmc2V0ID0gYnVmLndyaXRlVUludDgoRkVEQVVUSF9PUFRJT05TLkZFQVRVUkVfSUQsIG9mZnNldCk7XG4gICAgICAgICAgb2Zmc2V0ID0gYnVmLndyaXRlVUludDMyTEUodG9rZW4ubGVuZ3RoICsgNCArIDEsIG9mZnNldCk7XG4gICAgICAgICAgb2Zmc2V0ID0gYnVmLndyaXRlVUludDgoKEZFREFVVEhfT1BUSU9OUy5MSUJSQVJZX1NFQ1VSSVRZVE9LRU4gPDwgMSkgfCAoZmVkQXV0aC5lY2hvID8gRkVEQVVUSF9PUFRJT05TLkZFREFVVEhfWUVTX0VDSE8gOiBGRURBVVRIX09QVElPTlMuRkVEQVVUSF9OT19FQ0hPKSwgb2Zmc2V0KTtcbiAgICAgICAgICBidWYud3JpdGVJbnQzMkxFKHRva2VuLmxlbmd0aCwgb2Zmc2V0KTtcblxuICAgICAgICAgIGJ1ZmZlcnMucHVzaChidWYpO1xuICAgICAgICAgIGJ1ZmZlcnMucHVzaCh0b2tlbik7XG5cbiAgICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAodGhpcy50ZHNWZXJzaW9uID49IHZlcnNpb25zWyc3XzQnXSkge1xuICAgICAgLy8gU2lnbmFsIFVURi04IHN1cHBvcnQ6IFZhbHVlIDB4MEEsIGJpdCAwIG11c3QgYmUgc2V0IHRvIDEuIEFkZGVkIGluIFREUyA3LjQuXG4gICAgICBjb25zdCBVVEY4X1NVUFBPUlRfRkVBVFVSRV9JRCA9IDB4MGE7XG4gICAgICBjb25zdCBVVEY4X1NVUFBPUlRfQ0xJRU5UX1NVUFBPUlRTX1VURjggPSAweDAxO1xuICAgICAgY29uc3QgYnVmID0gQnVmZmVyLmFsbG9jKDYpO1xuICAgICAgYnVmLndyaXRlVUludDgoVVRGOF9TVVBQT1JUX0ZFQVRVUkVfSUQsIDApO1xuICAgICAgYnVmLndyaXRlVUludDMyTEUoMSwgMSk7XG4gICAgICBidWYud3JpdGVVSW50OChVVEY4X1NVUFBPUlRfQ0xJRU5UX1NVUFBPUlRTX1VURjgsIDUpO1xuICAgICAgYnVmZmVycy5wdXNoKGJ1Zik7XG4gICAgfVxuXG4gICAgYnVmZmVycy5wdXNoKEJ1ZmZlci5mcm9tKFtGRUFUVVJFX0VYVF9URVJNSU5BVE9SXSkpO1xuXG4gICAgcmV0dXJuIEJ1ZmZlci5jb25jYXQoYnVmZmVycyk7XG4gIH1cblxuICBidWlsZE9wdGlvbkZsYWdzMigpIHtcbiAgICBsZXQgZmxhZ3MyID0gRkxBR1NfMi5JTklUX0xBTkdfV0FSTiB8IEZMQUdTXzIuT0RCQ19PRkYgfCBGTEFHU18yLlVTRVJfTk9STUFMO1xuICAgIGlmICh0aGlzLnNzcGkpIHtcbiAgICAgIGZsYWdzMiB8PSBGTEFHU18yLklOVEVHUkFURURfU0VDVVJJVFlfT047XG4gICAgfSBlbHNlIHtcbiAgICAgIGZsYWdzMiB8PSBGTEFHU18yLklOVEVHUkFURURfU0VDVVJJVFlfT0ZGO1xuICAgIH1cbiAgICByZXR1cm4gZmxhZ3MyO1xuICB9XG5cbiAgYnVpbGRUeXBlRmxhZ3MoKSB7XG4gICAgbGV0IHR5cGVGbGFncyA9IFRZUEVfRkxBR1MuU1FMX0RGTFQgfCBUWVBFX0ZMQUdTLk9MRURCX09GRjtcbiAgICBpZiAodGhpcy5yZWFkT25seUludGVudCkge1xuICAgICAgdHlwZUZsYWdzIHw9IFRZUEVfRkxBR1MuUkVBRF9PTkxZX0lOVEVOVDtcbiAgICB9IGVsc2Uge1xuICAgICAgdHlwZUZsYWdzIHw9IFRZUEVfRkxBR1MuUkVBRF9XUklURV9JTlRFTlQ7XG4gICAgfVxuICAgIHJldHVybiB0eXBlRmxhZ3M7XG4gIH1cblxuICBidWlsZE9wdGlvbkZsYWdzMygpIHtcbiAgICByZXR1cm4gRkxBR1NfMy5DSEFOR0VfUEFTU1dPUkRfTk8gfCBGTEFHU18zLlVOS05PV05fQ09MTEFUSU9OX0hBTkRMSU5HIHwgRkxBR1NfMy5FWFRFTlNJT05fVVNFRDtcbiAgfVxuXG4gIHNjcmFtYmxlUGFzc3dvcmQocGFzc3dvcmQ6IEJ1ZmZlcikge1xuICAgIGZvciAobGV0IGIgPSAwLCBsZW4gPSBwYXNzd29yZC5sZW5ndGg7IGIgPCBsZW47IGIrKykge1xuICAgICAgbGV0IGJ5dGUgPSBwYXNzd29yZFtiXTtcbiAgICAgIGNvbnN0IGxvd05pYmJsZSA9IGJ5dGUgJiAweDBmO1xuICAgICAgY29uc3QgaGlnaE5pYmJsZSA9IGJ5dGUgPj4gNDtcbiAgICAgIGJ5dGUgPSAobG93TmliYmxlIDw8IDQpIHwgaGlnaE5pYmJsZTtcbiAgICAgIGJ5dGUgPSBieXRlIF4gMHhhNTtcbiAgICAgIHBhc3N3b3JkW2JdID0gYnl0ZTtcbiAgICB9XG4gICAgcmV0dXJuIHBhc3N3b3JkO1xuICB9XG5cbiAgdG9TdHJpbmcoaW5kZW50ID0gJycpIHtcbiAgICByZXR1cm4gaW5kZW50ICsgJ0xvZ2luNyAtICcgK1xuICAgICAgc3ByaW50ZignVERTOjB4JTA4WCwgUGFja2V0U2l6ZToweCUwOFgsIENsaWVudFByb2dWZXI6MHglMDhYLCBDbGllbnRQSUQ6MHglMDhYLCBDb25uZWN0aW9uSUQ6MHglMDhYJyxcbiAgICAgICAgICAgICAgdGhpcy50ZHNWZXJzaW9uLCB0aGlzLnBhY2tldFNpemUsIHRoaXMuY2xpZW50UHJvZ1ZlciwgdGhpcy5jbGllbnRQaWQsIHRoaXMuY29ubmVjdGlvbklkXG4gICAgICApICsgJ1xcbicgKyBpbmRlbnQgKyAnICAgICAgICAgJyArXG4gICAgICBzcHJpbnRmKCdGbGFnczE6MHglMDJYLCBGbGFnczI6MHglMDJYLCBUeXBlRmxhZ3M6MHglMDJYLCBGbGFnczM6MHglMDJYLCBDbGllbnRUaW1lem9uZTolZCwgQ2xpZW50TENJRDoweCUwOFgnLFxuICAgICAgICAgICAgICB0aGlzLmJ1aWxkT3B0aW9uRmxhZ3MxKCksIHRoaXMuYnVpbGRPcHRpb25GbGFnczIoKSwgdGhpcy5idWlsZFR5cGVGbGFncygpLCB0aGlzLmJ1aWxkT3B0aW9uRmxhZ3MzKCksIHRoaXMuY2xpZW50VGltZVpvbmUsIHRoaXMuY2xpZW50TGNpZFxuICAgICAgKSArICdcXG4nICsgaW5kZW50ICsgJyAgICAgICAgICcgK1xuICAgICAgc3ByaW50ZihcIkhvc3RuYW1lOiclcycsIFVzZXJuYW1lOiclcycsIFBhc3N3b3JkOiclcycsIEFwcE5hbWU6JyVzJywgU2VydmVyTmFtZTonJXMnLCBMaWJyYXJ5TmFtZTonJXMnXCIsXG4gICAgICAgICAgICAgIHRoaXMuaG9zdG5hbWUsIHRoaXMudXNlck5hbWUsIHRoaXMucGFzc3dvcmQsIHRoaXMuYXBwTmFtZSwgdGhpcy5zZXJ2ZXJOYW1lLCB0aGlzLmxpYnJhcnlOYW1lXG4gICAgICApICsgJ1xcbicgKyBpbmRlbnQgKyAnICAgICAgICAgJyArXG4gICAgICBzcHJpbnRmKFwiTGFuZ3VhZ2U6JyVzJywgRGF0YWJhc2U6JyVzJywgU1NQSTonJXMnLCBBdHRhY2hEYkZpbGU6JyVzJywgQ2hhbmdlUGFzc3dvcmQ6JyVzJ1wiLFxuICAgICAgICAgICAgICB0aGlzLmxhbmd1YWdlLCB0aGlzLmRhdGFiYXNlLCB0aGlzLnNzcGksIHRoaXMuYXR0YWNoRGJGaWxlLCB0aGlzLmNoYW5nZVBhc3N3b3JkXG4gICAgICApO1xuICB9XG59XG5cbmV4cG9ydCBkZWZhdWx0IExvZ2luN1BheWxvYWQ7XG5tb2R1bGUuZXhwb3J0cyA9IExvZ2luN1BheWxvYWQ7XG4iXX0=