"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _crypto = _interopRequireDefault(require("crypto"));

var _os = _interopRequireDefault(require("os"));

var _constants = _interopRequireDefault(require("constants"));

var _tls = require("tls");

var _stream = require("stream");

var _identity = require("@azure/identity");

var _bulkLoad = _interopRequireDefault(require("./bulk-load"));

var _debug = _interopRequireDefault(require("./debug"));

var _events = require("events");

var _instanceLookup = require("./instance-lookup");

var _transientErrorLookup = require("./transient-error-lookup");

var _packet = require("./packet");

var _preloginPayload = _interopRequireDefault(require("./prelogin-payload"));

var _login7Payload = _interopRequireDefault(require("./login7-payload"));

var _ntlmPayload = _interopRequireDefault(require("./ntlm-payload"));

var _request = _interopRequireDefault(require("./request"));

var _rpcrequestPayload = _interopRequireDefault(require("./rpcrequest-payload"));

var _sqlbatchPayload = _interopRequireDefault(require("./sqlbatch-payload"));

var _messageIo = _interopRequireDefault(require("./message-io"));

var _tokenStreamParser = require("./token/token-stream-parser");

var _transaction = require("./transaction");

var _errors = require("./errors");

var _connector = require("./connector");

var _library = require("./library");

var _tdsVersions = require("./tds-versions");

var _message = _interopRequireDefault(require("./message"));

var _ntlm = require("./ntlm");

var _nodeAbortController = require("node-abort-controller");

var _dataType = require("./data-type");

var _bulkLoadPayload = require("./bulk-load-payload");

var _package = require("../package.json");

var _url = require("url");

var _handler = require("./token/handler");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let trustServerWarningEmitted = false;

const emitTrustServerCertificateWarning = () => {
  if (!trustServerWarningEmitted) {
    trustServerWarningEmitted = true;
    process.emitWarning('`config.options.trustServerCertificate` will default to false in the future. To silence this message, specify a value explicitly in the config options');
  }
};

let domainRenameToTenantIdWarningEmitted = false;

const emitDomainRenameToTenantIdWarning = () => {
  if (!domainRenameToTenantIdWarningEmitted) {
    domainRenameToTenantIdWarningEmitted = true;
    process.emitWarning('`When using authentication type `azure-active-directory-password`,' + ' config.authentication.options.domain` will be renamed to config.authentications.options.tenantId`' + ' in the future. Rename `domain` to `tenantId` to silence this message.');
  }
};

/**
 * @private
 */
const KEEP_ALIVE_INITIAL_DELAY = 30 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CLIENT_REQUEST_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CANCEL_TIMEOUT = 5 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_RETRY_INTERVAL = 500;
/**
 * @private
 */

const DEFAULT_PACKET_SIZE = 4 * 1024;
/**
 * @private
 */

const DEFAULT_TEXTSIZE = 2147483647;
/**
 * @private
 */

const DEFAULT_DATEFIRST = 7;
/**
 * @private
 */

const DEFAULT_PORT = 1433;
/**
 * @private
 */

const DEFAULT_TDS_VERSION = '7_4';
/**
 * @private
 */

const DEFAULT_LANGUAGE = 'us_english';
/**
 * @private
 */

const DEFAULT_DATEFORMAT = 'mdy';

/**
 * @private
 */
const CLEANUP_TYPE = {
  NORMAL: 0,
  REDIRECT: 1,
  RETRY: 2
};

/**
 * A [[Connection]] instance represents a single connection to a database server.
 *
 * ```js
 * var Connection = require('tedious').Connection;
 * var config = {
 *  "authentication": {
 *    ...,
 *    "options": {...}
 *  },
 *  "options": {...}
 * };
 * var connection = new Connection(config);
 * ```
 *
 * Only one request at a time may be executed on a connection. Once a [[Request]]
 * has been initiated (with [[Connection.callProcedure]], [[Connection.execSql]],
 * or [[Connection.execSqlBatch]]), another should not be initiated until the
 * [[Request]]'s completion callback is called.
 */
class Connection extends _events.EventEmitter {
  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * Note: be aware of the different options field:
   * 1. config.authentication.options
   * 2. config.options
   *
   * ```js
   * const { Connection } = require('tedious');
   *
   * const config = {
   *  "authentication": {
   *    ...,
   *    "options": {...}
   *  },
   *  "options": {...}
   * };
   *
   * const connection = new Connection(config);
   * ```
   *
   * @param config
   */
  constructor(config) {
    super();
    this.fedAuthRequired = void 0;
    this.config = void 0;
    this.secureContext = void 0;
    this.inTransaction = void 0;
    this.transactionDescriptors = void 0;
    this.transactionDepth = void 0;
    this.isSqlBatch = void 0;
    this.curTransientRetryCount = void 0;
    this.transientErrorLookup = void 0;
    this.closed = void 0;
    this.loginError = void 0;
    this.debug = void 0;
    this.ntlmpacket = void 0;
    this.ntlmpacketBuffer = void 0;
    this.routingData = void 0;
    this.messageIo = void 0;
    this.state = void 0;
    this.resetConnectionOnNextRequest = void 0;
    this.request = void 0;
    this.procReturnStatusValue = void 0;
    this.socket = void 0;
    this.messageBuffer = void 0;
    this.connectTimer = void 0;
    this.cancelTimer = void 0;
    this.requestTimer = void 0;
    this.retryTimer = void 0;
    this._cancelAfterRequestSent = void 0;
    this.databaseCollation = void 0;

    if (typeof config !== 'object' || config === null) {
      throw new TypeError('The "config" argument is required and must be of type Object.');
    }

    if (typeof config.server !== 'string') {
      throw new TypeError('The "config.server" property is required and must be of type string.');
    }

    this.fedAuthRequired = false;
    let authentication;

    if (config.authentication !== undefined) {
      if (typeof config.authentication !== 'object' || config.authentication === null) {
        throw new TypeError('The "config.authentication" property must be of type Object.');
      }

      const type = config.authentication.type;
      const options = config.authentication.options === undefined ? {} : config.authentication.options;

      if (typeof type !== 'string') {
        throw new TypeError('The "config.authentication.type" property must be of type string.');
      }

      if (type !== 'default' && type !== 'ntlm' && type !== 'azure-active-directory-password' && type !== 'azure-active-directory-access-token' && type !== 'azure-active-directory-msi-vm' && type !== 'azure-active-directory-msi-app-service' && type !== 'azure-active-directory-service-principal-secret') {
        throw new TypeError('The "type" property must one of "default", "ntlm", "azure-active-directory-password", "azure-active-directory-access-token", "azure-active-directory-msi-vm" or "azure-active-directory-msi-app-service" or "azure-active-directory-service-principal-secret".');
      }

      if (typeof options !== 'object' || options === null) {
        throw new TypeError('The "config.authentication.options" property must be of type object.');
      }

      if (type === 'ntlm') {
        if (typeof options.domain !== 'string') {
          throw new TypeError('The "config.authentication.options.domain" property must be of type string.');
        }

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'ntlm',
          options: {
            userName: options.userName,
            password: options.password,
            domain: options.domain && options.domain.toUpperCase()
          }
        };
      } else if (type === 'azure-active-directory-password') {
        var _options$tenantId, _options$clientId;

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        } else if (options.clientId === undefined) {
          emitAzureADPasswordClientIdDeprecationWarning();
        }

        if (options.domain !== undefined && typeof options.domain !== 'string') {
          throw new TypeError('The "config.authentication.options.domain" property must be of type string.');
        } else if (options.domain !== undefined) {
          emitDomainRenameToTenantIdWarning();
        }

        if (options.tenantId !== undefined && typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-password',
          options: {
            userName: options.userName,
            password: options.password,
            tenantId: (_options$tenantId = options.tenantId) !== null && _options$tenantId !== void 0 ? _options$tenantId : options.domain,
            clientId: (_options$clientId = options.clientId) !== null && _options$clientId !== void 0 ? _options$clientId : '7f98cb04-cd1e-40df-9140-3bf7e2cea4db'
          }
        };
      } else if (type === 'azure-active-directory-access-token') {
        if (typeof options.token !== 'string') {
          throw new TypeError('The "config.authentication.options.token" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-access-token',
          options: {
            token: options.token
          }
        };
      } else if (type === 'azure-active-directory-msi-vm') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-vm',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-msi-app-service') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-app-service',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-service-principal-secret') {
        if (typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        if (typeof options.clientSecret !== 'string') {
          throw new TypeError('The "config.authentication.options.clientSecret" property must be of type string.');
        }

        if (typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-service-principal-secret',
          options: {
            clientId: options.clientId,
            clientSecret: options.clientSecret,
            tenantId: options.tenantId
          }
        };
      } else {
        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'default',
          options: {
            userName: options.userName,
            password: options.password
          }
        };
      }
    } else {
      authentication = {
        type: 'default',
        options: {
          userName: undefined,
          password: undefined
        }
      };
    }

    this.config = {
      server: config.server,
      authentication: authentication,
      options: {
        abortTransactionOnError: false,
        appName: undefined,
        camelCaseColumns: false,
        cancelTimeout: DEFAULT_CANCEL_TIMEOUT,
        columnEncryptionKeyCacheTTL: 2 * 60 * 60 * 1000,
        // Units: miliseconds
        columnEncryptionSetting: false,
        columnNameReplacer: undefined,
        connectionRetryInterval: DEFAULT_CONNECT_RETRY_INTERVAL,
        connectTimeout: DEFAULT_CONNECT_TIMEOUT,
        connectionIsolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        cryptoCredentialsDetails: {},
        database: undefined,
        datefirst: DEFAULT_DATEFIRST,
        dateFormat: DEFAULT_DATEFORMAT,
        debug: {
          data: false,
          packet: false,
          payload: false,
          token: false
        },
        enableAnsiNull: true,
        enableAnsiNullDefault: true,
        enableAnsiPadding: true,
        enableAnsiWarnings: true,
        enableArithAbort: true,
        enableConcatNullYieldsNull: true,
        enableCursorCloseOnCommit: null,
        enableImplicitTransactions: false,
        enableNumericRoundabort: false,
        enableQuotedIdentifier: true,
        encrypt: true,
        fallbackToDefaultDb: false,
        encryptionKeyStoreProviders: undefined,
        instanceName: undefined,
        isolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        language: DEFAULT_LANGUAGE,
        localAddress: undefined,
        maxRetriesOnTransientErrors: 3,
        multiSubnetFailover: false,
        packetSize: DEFAULT_PACKET_SIZE,
        port: DEFAULT_PORT,
        readOnlyIntent: false,
        requestTimeout: DEFAULT_CLIENT_REQUEST_TIMEOUT,
        rowCollectionOnDone: false,
        rowCollectionOnRequestCompletion: false,
        serverName: undefined,
        serverSupportsColumnEncryption: false,
        tdsVersion: DEFAULT_TDS_VERSION,
        textsize: DEFAULT_TEXTSIZE,
        trustedServerNameAE: undefined,
        trustServerCertificate: true,
        useColumnNames: false,
        useUTC: true,
        workstationId: undefined,
        lowerCaseGuids: false
      }
    };

    if (config.options) {
      if (config.options.port && config.options.instanceName) {
        throw new Error('Port and instanceName are mutually exclusive, but ' + config.options.port + ' and ' + config.options.instanceName + ' provided');
      }

      if (config.options.abortTransactionOnError !== undefined) {
        if (typeof config.options.abortTransactionOnError !== 'boolean' && config.options.abortTransactionOnError !== null) {
          throw new TypeError('The "config.options.abortTransactionOnError" property must be of type string or null.');
        }

        this.config.options.abortTransactionOnError = config.options.abortTransactionOnError;
      }

      if (config.options.appName !== undefined) {
        if (typeof config.options.appName !== 'string') {
          throw new TypeError('The "config.options.appName" property must be of type string.');
        }

        this.config.options.appName = config.options.appName;
      }

      if (config.options.camelCaseColumns !== undefined) {
        if (typeof config.options.camelCaseColumns !== 'boolean') {
          throw new TypeError('The "config.options.camelCaseColumns" property must be of type boolean.');
        }

        this.config.options.camelCaseColumns = config.options.camelCaseColumns;
      }

      if (config.options.cancelTimeout !== undefined) {
        if (typeof config.options.cancelTimeout !== 'number') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type number.');
        }

        this.config.options.cancelTimeout = config.options.cancelTimeout;
      }

      if (config.options.columnNameReplacer) {
        if (typeof config.options.columnNameReplacer !== 'function') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type function.');
        }

        this.config.options.columnNameReplacer = config.options.columnNameReplacer;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.connectionIsolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.connectionIsolationLevel, 'config.options.connectionIsolationLevel');
        this.config.options.connectionIsolationLevel = config.options.connectionIsolationLevel;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.cryptoCredentialsDetails !== undefined) {
        if (typeof config.options.cryptoCredentialsDetails !== 'object' || config.options.cryptoCredentialsDetails === null) {
          throw new TypeError('The "config.options.cryptoCredentialsDetails" property must be of type Object.');
        }

        this.config.options.cryptoCredentialsDetails = config.options.cryptoCredentialsDetails;
      }

      if (config.options.database !== undefined) {
        if (typeof config.options.database !== 'string') {
          throw new TypeError('The "config.options.database" property must be of type string.');
        }

        this.config.options.database = config.options.database;
      }

      if (config.options.datefirst !== undefined) {
        if (typeof config.options.datefirst !== 'number' && config.options.datefirst !== null) {
          throw new TypeError('The "config.options.datefirst" property must be of type number.');
        }

        if (config.options.datefirst !== null && (config.options.datefirst < 1 || config.options.datefirst > 7)) {
          throw new RangeError('The "config.options.datefirst" property must be >= 1 and <= 7');
        }

        this.config.options.datefirst = config.options.datefirst;
      }

      if (config.options.dateFormat !== undefined) {
        if (typeof config.options.dateFormat !== 'string' && config.options.dateFormat !== null) {
          throw new TypeError('The "config.options.dateFormat" property must be of type string or null.');
        }

        this.config.options.dateFormat = config.options.dateFormat;
      }

      if (config.options.debug) {
        if (config.options.debug.data !== undefined) {
          if (typeof config.options.debug.data !== 'boolean') {
            throw new TypeError('The "config.options.debug.data" property must be of type boolean.');
          }

          this.config.options.debug.data = config.options.debug.data;
        }

        if (config.options.debug.packet !== undefined) {
          if (typeof config.options.debug.packet !== 'boolean') {
            throw new TypeError('The "config.options.debug.packet" property must be of type boolean.');
          }

          this.config.options.debug.packet = config.options.debug.packet;
        }

        if (config.options.debug.payload !== undefined) {
          if (typeof config.options.debug.payload !== 'boolean') {
            throw new TypeError('The "config.options.debug.payload" property must be of type boolean.');
          }

          this.config.options.debug.payload = config.options.debug.payload;
        }

        if (config.options.debug.token !== undefined) {
          if (typeof config.options.debug.token !== 'boolean') {
            throw new TypeError('The "config.options.debug.token" property must be of type boolean.');
          }

          this.config.options.debug.token = config.options.debug.token;
        }
      }

      if (config.options.enableAnsiNull !== undefined) {
        if (typeof config.options.enableAnsiNull !== 'boolean' && config.options.enableAnsiNull !== null) {
          throw new TypeError('The "config.options.enableAnsiNull" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNull = config.options.enableAnsiNull;
      }

      if (config.options.enableAnsiNullDefault !== undefined) {
        if (typeof config.options.enableAnsiNullDefault !== 'boolean' && config.options.enableAnsiNullDefault !== null) {
          throw new TypeError('The "config.options.enableAnsiNullDefault" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNullDefault = config.options.enableAnsiNullDefault;
      }

      if (config.options.enableAnsiPadding !== undefined) {
        if (typeof config.options.enableAnsiPadding !== 'boolean' && config.options.enableAnsiPadding !== null) {
          throw new TypeError('The "config.options.enableAnsiPadding" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiPadding = config.options.enableAnsiPadding;
      }

      if (config.options.enableAnsiWarnings !== undefined) {
        if (typeof config.options.enableAnsiWarnings !== 'boolean' && config.options.enableAnsiWarnings !== null) {
          throw new TypeError('The "config.options.enableAnsiWarnings" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiWarnings = config.options.enableAnsiWarnings;
      }

      if (config.options.enableArithAbort !== undefined) {
        if (typeof config.options.enableArithAbort !== 'boolean' && config.options.enableArithAbort !== null) {
          throw new TypeError('The "config.options.enableArithAbort" property must be of type boolean or null.');
        }

        this.config.options.enableArithAbort = config.options.enableArithAbort;
      }

      if (config.options.enableConcatNullYieldsNull !== undefined) {
        if (typeof config.options.enableConcatNullYieldsNull !== 'boolean' && config.options.enableConcatNullYieldsNull !== null) {
          throw new TypeError('The "config.options.enableConcatNullYieldsNull" property must be of type boolean or null.');
        }

        this.config.options.enableConcatNullYieldsNull = config.options.enableConcatNullYieldsNull;
      }

      if (config.options.enableCursorCloseOnCommit !== undefined) {
        if (typeof config.options.enableCursorCloseOnCommit !== 'boolean' && config.options.enableCursorCloseOnCommit !== null) {
          throw new TypeError('The "config.options.enableCursorCloseOnCommit" property must be of type boolean or null.');
        }

        this.config.options.enableCursorCloseOnCommit = config.options.enableCursorCloseOnCommit;
      }

      if (config.options.enableImplicitTransactions !== undefined) {
        if (typeof config.options.enableImplicitTransactions !== 'boolean' && config.options.enableImplicitTransactions !== null) {
          throw new TypeError('The "config.options.enableImplicitTransactions" property must be of type boolean or null.');
        }

        this.config.options.enableImplicitTransactions = config.options.enableImplicitTransactions;
      }

      if (config.options.enableNumericRoundabort !== undefined) {
        if (typeof config.options.enableNumericRoundabort !== 'boolean' && config.options.enableNumericRoundabort !== null) {
          throw new TypeError('The "config.options.enableNumericRoundabort" property must be of type boolean or null.');
        }

        this.config.options.enableNumericRoundabort = config.options.enableNumericRoundabort;
      }

      if (config.options.enableQuotedIdentifier !== undefined) {
        if (typeof config.options.enableQuotedIdentifier !== 'boolean' && config.options.enableQuotedIdentifier !== null) {
          throw new TypeError('The "config.options.enableQuotedIdentifier" property must be of type boolean or null.');
        }

        this.config.options.enableQuotedIdentifier = config.options.enableQuotedIdentifier;
      }

      if (config.options.encrypt !== undefined) {
        if (typeof config.options.encrypt !== 'boolean') {
          throw new TypeError('The "config.options.encrypt" property must be of type boolean.');
        }

        this.config.options.encrypt = config.options.encrypt;
      }

      if (config.options.fallbackToDefaultDb !== undefined) {
        if (typeof config.options.fallbackToDefaultDb !== 'boolean') {
          throw new TypeError('The "config.options.fallbackToDefaultDb" property must be of type boolean.');
        }

        this.config.options.fallbackToDefaultDb = config.options.fallbackToDefaultDb;
      }

      if (config.options.instanceName !== undefined) {
        if (typeof config.options.instanceName !== 'string') {
          throw new TypeError('The "config.options.instanceName" property must be of type string.');
        }

        this.config.options.instanceName = config.options.instanceName;
        this.config.options.port = undefined;
      }

      if (config.options.isolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.isolationLevel, 'config.options.isolationLevel');
        this.config.options.isolationLevel = config.options.isolationLevel;
      }

      if (config.options.language !== undefined) {
        if (typeof config.options.language !== 'string' && config.options.language !== null) {
          throw new TypeError('The "config.options.language" property must be of type string or null.');
        }

        this.config.options.language = config.options.language;
      }

      if (config.options.localAddress !== undefined) {
        if (typeof config.options.localAddress !== 'string') {
          throw new TypeError('The "config.options.localAddress" property must be of type string.');
        }

        this.config.options.localAddress = config.options.localAddress;
      }

      if (config.options.multiSubnetFailover !== undefined) {
        if (typeof config.options.multiSubnetFailover !== 'boolean') {
          throw new TypeError('The "config.options.multiSubnetFailover" property must be of type boolean.');
        }

        this.config.options.multiSubnetFailover = config.options.multiSubnetFailover;
      }

      if (config.options.packetSize !== undefined) {
        if (typeof config.options.packetSize !== 'number') {
          throw new TypeError('The "config.options.packetSize" property must be of type number.');
        }

        this.config.options.packetSize = config.options.packetSize;
      }

      if (config.options.port !== undefined) {
        if (typeof config.options.port !== 'number') {
          throw new TypeError('The "config.options.port" property must be of type number.');
        }

        if (config.options.port <= 0 || config.options.port >= 65536) {
          throw new RangeError('The "config.options.port" property must be > 0 and < 65536');
        }

        this.config.options.port = config.options.port;
        this.config.options.instanceName = undefined;
      }

      if (config.options.readOnlyIntent !== undefined) {
        if (typeof config.options.readOnlyIntent !== 'boolean') {
          throw new TypeError('The "config.options.readOnlyIntent" property must be of type boolean.');
        }

        this.config.options.readOnlyIntent = config.options.readOnlyIntent;
      }

      if (config.options.requestTimeout !== undefined) {
        if (typeof config.options.requestTimeout !== 'number') {
          throw new TypeError('The "config.options.requestTimeout" property must be of type number.');
        }

        this.config.options.requestTimeout = config.options.requestTimeout;
      }

      if (config.options.maxRetriesOnTransientErrors !== undefined) {
        if (typeof config.options.maxRetriesOnTransientErrors !== 'number') {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be of type number.');
        }

        if (config.options.maxRetriesOnTransientErrors < 0) {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be equal or greater than 0.');
        }

        this.config.options.maxRetriesOnTransientErrors = config.options.maxRetriesOnTransientErrors;
      }

      if (config.options.connectionRetryInterval !== undefined) {
        if (typeof config.options.connectionRetryInterval !== 'number') {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be of type number.');
        }

        if (config.options.connectionRetryInterval <= 0) {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be greater than 0.');
        }

        this.config.options.connectionRetryInterval = config.options.connectionRetryInterval;
      }

      if (config.options.rowCollectionOnDone !== undefined) {
        if (typeof config.options.rowCollectionOnDone !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnDone" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnDone = config.options.rowCollectionOnDone;
      }

      if (config.options.rowCollectionOnRequestCompletion !== undefined) {
        if (typeof config.options.rowCollectionOnRequestCompletion !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnRequestCompletion" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnRequestCompletion = config.options.rowCollectionOnRequestCompletion;
      }

      if (config.options.tdsVersion !== undefined) {
        if (typeof config.options.tdsVersion !== 'string') {
          throw new TypeError('The "config.options.tdsVersion" property must be of type string.');
        }

        this.config.options.tdsVersion = config.options.tdsVersion;
      }

      if (config.options.textsize !== undefined) {
        if (typeof config.options.textsize !== 'number' && config.options.textsize !== null) {
          throw new TypeError('The "config.options.textsize" property must be of type number or null.');
        }

        if (config.options.textsize > 2147483647) {
          throw new TypeError('The "config.options.textsize" can\'t be greater than 2147483647.');
        } else if (config.options.textsize < -1) {
          throw new TypeError('The "config.options.textsize" can\'t be smaller than -1.');
        }

        this.config.options.textsize = config.options.textsize | 0;
      }

      if (config.options.trustServerCertificate !== undefined) {
        if (typeof config.options.trustServerCertificate !== 'boolean') {
          throw new TypeError('The "config.options.trustServerCertificate" property must be of type boolean.');
        }

        this.config.options.trustServerCertificate = config.options.trustServerCertificate;
      } else {
        emitTrustServerCertificateWarning();
      }

      if (config.options.useColumnNames !== undefined) {
        if (typeof config.options.useColumnNames !== 'boolean') {
          throw new TypeError('The "config.options.useColumnNames" property must be of type boolean.');
        }

        this.config.options.useColumnNames = config.options.useColumnNames;
      }

      if (config.options.useUTC !== undefined) {
        if (typeof config.options.useUTC !== 'boolean') {
          throw new TypeError('The "config.options.useUTC" property must be of type boolean.');
        }

        this.config.options.useUTC = config.options.useUTC;
      }

      if (config.options.workstationId !== undefined) {
        if (typeof config.options.workstationId !== 'string') {
          throw new TypeError('The "config.options.workstationId" property must be of type string.');
        }

        this.config.options.workstationId = config.options.workstationId;
      }

      if (config.options.lowerCaseGuids !== undefined) {
        if (typeof config.options.lowerCaseGuids !== 'boolean') {
          throw new TypeError('The "config.options.lowerCaseGuids" property must be of type boolean.');
        }

        this.config.options.lowerCaseGuids = config.options.lowerCaseGuids;
      }
    }

    let credentialsDetails = this.config.options.cryptoCredentialsDetails;

    if (credentialsDetails.secureOptions === undefined) {
      // If the caller has not specified their own `secureOptions`,
      // we set `SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS` here.
      // Older SQL Server instances running on older Windows versions have
      // trouble with the BEAST workaround in OpenSSL.
      // As BEAST is a browser specific exploit, we can just disable this option here.
      credentialsDetails = Object.create(credentialsDetails, {
        secureOptions: {
          value: _constants.default.SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS
        }
      });
    }

    this.secureContext = (0, _tls.createSecureContext)(credentialsDetails);
    this.debug = this.createDebug();
    this.inTransaction = false;
    this.transactionDescriptors = [Buffer.from([0, 0, 0, 0, 0, 0, 0, 0])]; // 'beginTransaction', 'commitTransaction' and 'rollbackTransaction'
    // events are utilized to maintain inTransaction property state which in
    // turn is used in managing transactions. These events are only fired for
    // TDS version 7.2 and beyond. The properties below are used to emulate
    // equivalent behavior for TDS versions before 7.2.

    this.transactionDepth = 0;
    this.isSqlBatch = false;
    this.closed = false;
    this.messageBuffer = Buffer.alloc(0);
    this.curTransientRetryCount = 0;
    this.transientErrorLookup = new _transientErrorLookup.TransientErrorLookup();
    this.state = this.STATE.INITIALIZED;

    this._cancelAfterRequestSent = () => {
      this.messageIo.sendMessage(_packet.TYPE.ATTENTION);
      this.createCancelTimer();
    };
  }

  connect(connectListener) {
    if (this.state !== this.STATE.INITIALIZED) {
      throw new _errors.ConnectionError('`.connect` can not be called on a Connection in `' + this.state.name + '` state.');
    }

    if (connectListener) {
      const onConnect = err => {
        this.removeListener('error', onError);
        connectListener(err);
      };

      const onError = err => {
        this.removeListener('connect', onConnect);
        connectListener(err);
      };

      this.once('connect', onConnect);
      this.once('error', onError);
    }

    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * The server has reported that the charset has changed.
   */


  on(event, listener) {
    return super.on(event, listener);
  }
  /**
   * @private
   */


  emit(event, ...args) {
    return super.emit(event, ...args);
  }
  /**
   * Closes the connection to the database.
   *
   * The [[Event_end]] will be emitted once the connection has been closed.
   */


  close() {
    this.transitionTo(this.STATE.FINAL);
  }
  /**
   * @private
   */


  initialiseConnection() {
    const signal = this.createConnectTimer();

    if (this.config.options.port) {
      return this.connectOnPort(this.config.options.port, this.config.options.multiSubnetFailover, signal);
    } else {
      return new _instanceLookup.InstanceLookup().instanceLookup({
        server: this.config.server,
        instanceName: this.config.options.instanceName,
        timeout: this.config.options.connectTimeout,
        signal: signal
      }, (err, port) => {
        if (err) {
          if (err.name === 'AbortError') {
            return;
          }

          this.emit('connect', new _errors.ConnectionError(err.message, 'EINSTLOOKUP'));
        } else {
          this.connectOnPort(port, this.config.options.multiSubnetFailover, signal);
        }
      });
    }
  }
  /**
   * @private
   */


  cleanupConnection(cleanupType) {
    if (!this.closed) {
      this.clearConnectTimer();
      this.clearRequestTimer();
      this.clearRetryTimer();
      this.closeConnection();

      if (cleanupType === CLEANUP_TYPE.REDIRECT) {
        this.emit('rerouting');
      } else if (cleanupType !== CLEANUP_TYPE.RETRY) {
        process.nextTick(() => {
          this.emit('end');
        });
      }

      const request = this.request;

      if (request) {
        const err = new _errors.RequestError('Connection closed before request completed.', 'ECLOSE');
        request.callback(err);
        this.request = undefined;
      }

      this.closed = true;
      this.loginError = undefined;
    }
  }
  /**
   * @private
   */


  createDebug() {
    const debug = new _debug.default(this.config.options.debug);
    debug.on('debug', message => {
      this.emit('debug', message);
    });
    return debug;
  }
  /**
   * @private
   */


  createTokenStreamParser(message, handler) {
    return new _tokenStreamParser.Parser(message, this.debug, handler, this.config.options);
  }

  connectOnPort(port, multiSubnetFailover, signal) {
    const connectOpts = {
      host: this.routingData ? this.routingData.server : this.config.server,
      port: this.routingData ? this.routingData.port : port,
      localAddress: this.config.options.localAddress
    };
    new _connector.Connector(connectOpts, signal, multiSubnetFailover).execute((err, socket) => {
      if (err) {
        if (err.name === 'AbortError') {
          return;
        }

        return this.socketError(err);
      }

      socket = socket;
      socket.on('error', error => {
        this.socketError(error);
      });
      socket.on('close', () => {
        this.socketClose();
      });
      socket.on('end', () => {
        this.socketEnd();
      });
      socket.setKeepAlive(true, KEEP_ALIVE_INITIAL_DELAY);
      this.messageIo = new _messageIo.default(socket, this.config.options.packetSize, this.debug);
      this.messageIo.on('secure', cleartext => {
        this.emit('secure', cleartext);
      });
      this.socket = socket;
      this.closed = false;
      this.debug.log('connected to ' + this.config.server + ':' + this.config.options.port);
      this.sendPreLogin();
      this.transitionTo(this.STATE.SENT_PRELOGIN);
    });
  }
  /**
   * @private
   */


  closeConnection() {
    if (this.socket) {
      this.socket.destroy();
    }
  }
  /**
   * @private
   */


  createConnectTimer() {
    const controller = new _nodeAbortController.AbortController();
    this.connectTimer = setTimeout(() => {
      controller.abort();
      this.connectTimeout();
    }, this.config.options.connectTimeout);
    return controller.signal;
  }
  /**
   * @private
   */


  createCancelTimer() {
    this.clearCancelTimer();
    const timeout = this.config.options.cancelTimeout;

    if (timeout > 0) {
      this.cancelTimer = setTimeout(() => {
        this.cancelTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRequestTimer() {
    this.clearRequestTimer(); // release old timer, just to be safe

    const request = this.request;
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;

    if (timeout) {
      this.requestTimer = setTimeout(() => {
        this.requestTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRetryTimer() {
    this.clearRetryTimer();
    this.retryTimer = setTimeout(() => {
      this.retryTimeout();
    }, this.config.options.connectionRetryInterval);
  }
  /**
   * @private
   */


  connectTimeout() {
    const message = `Failed to connect to ${this.config.server}${this.config.options.port ? `:${this.config.options.port}` : `\\${this.config.options.instanceName}`} in ${this.config.options.connectTimeout}ms`;
    this.debug.log(message);
    this.emit('connect', new _errors.ConnectionError(message, 'ETIMEOUT'));
    this.connectTimer = undefined;
    this.dispatchEvent('connectTimeout');
  }
  /**
   * @private
   */


  cancelTimeout() {
    const message = `Failed to cancel request in ${this.config.options.cancelTimeout}ms`;
    this.debug.log(message);
    this.dispatchEvent('socketError', new _errors.ConnectionError(message, 'ETIMEOUT'));
  }
  /**
   * @private
   */


  requestTimeout() {
    this.requestTimer = undefined;
    const request = this.request;
    request.cancel();
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;
    const message = 'Timeout: Request failed to complete in ' + timeout + 'ms';
    request.error = new _errors.RequestError(message, 'ETIMEOUT');
  }
  /**
   * @private
   */


  retryTimeout() {
    this.retryTimer = undefined;
    this.emit('retry');
    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * @private
   */


  clearConnectTimer() {
    if (this.connectTimer) {
      clearTimeout(this.connectTimer);
      this.connectTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearCancelTimer() {
    if (this.cancelTimer) {
      clearTimeout(this.cancelTimer);
      this.cancelTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRequestTimer() {
    if (this.requestTimer) {
      clearTimeout(this.requestTimer);
      this.requestTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRetryTimer() {
    if (this.retryTimer) {
      clearTimeout(this.retryTimer);
      this.retryTimer = undefined;
    }
  }
  /**
   * @private
   */


  transitionTo(newState) {
    if (this.state === newState) {
      this.debug.log('State is already ' + newState.name);
      return;
    }

    if (this.state && this.state.exit) {
      this.state.exit.call(this, newState);
    }

    this.debug.log('State change: ' + (this.state ? this.state.name : 'undefined') + ' -> ' + newState.name);
    this.state = newState;

    if (this.state.enter) {
      this.state.enter.apply(this);
    }
  }
  /**
   * @private
   */


  getEventHandler(eventName) {
    const handler = this.state.events[eventName];

    if (!handler) {
      throw new Error(`No event '${eventName}' in state '${this.state.name}'`);
    }

    return handler;
  }
  /**
   * @private
   */


  dispatchEvent(eventName, ...args) {
    const handler = this.state.events[eventName];

    if (handler) {
      handler.apply(this, args);
    } else {
      this.emit('error', new Error(`No event '${eventName}' in state '${this.state.name}'`));
      this.close();
    }
  }
  /**
   * @private
   */


  socketError(error) {
    if (this.state === this.STATE.CONNECTING || this.state === this.STATE.SENT_TLSSSLNEGOTIATION) {
      const message = `Failed to connect to ${this.config.server}:${this.config.options.port} - ${error.message}`;
      this.debug.log(message);
      this.emit('connect', new _errors.ConnectionError(message, 'ESOCKET'));
    } else {
      const message = `Connection lost - ${error.message}`;
      this.debug.log(message);
      this.emit('error', new _errors.ConnectionError(message, 'ESOCKET'));
    }

    this.dispatchEvent('socketError', error);
  }
  /**
   * @private
   */


  socketEnd() {
    this.debug.log('socket ended');

    if (this.state !== this.STATE.FINAL) {
      const error = new Error('socket hang up');
      error.code = 'ECONNRESET';
      this.socketError(error);
    }
  }
  /**
   * @private
   */


  socketClose() {
    this.debug.log('connection to ' + this.config.server + ':' + this.config.options.port + ' closed');

    if (this.state === this.STATE.REROUTING) {
      this.debug.log('Rerouting to ' + this.routingData.server + ':' + this.routingData.port);
      this.dispatchEvent('reconnect');
    } else if (this.state === this.STATE.TRANSIENT_FAILURE_RETRY) {
      const server = this.routingData ? this.routingData.server : this.config.server;
      const port = this.routingData ? this.routingData.port : this.config.options.port;
      this.debug.log('Retry after transient failure connecting to ' + server + ':' + port);
      this.dispatchEvent('retry');
    } else {
      this.transitionTo(this.STATE.FINAL);
    }
  }
  /**
   * @private
   */


  sendPreLogin() {
    var _$exec;

    const [, major, minor, build] = (_$exec = /^(\d+)\.(\d+)\.(\d+)/.exec(_package.version)) !== null && _$exec !== void 0 ? _$exec : ['0.0.0', '0', '0', '0'];
    const payload = new _preloginPayload.default({
      encrypt: this.config.options.encrypt,
      version: {
        major: Number(major),
        minor: Number(minor),
        build: Number(build),
        subbuild: 0
      }
    });
    this.messageIo.sendMessage(_packet.TYPE.PRELOGIN, payload.data);
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  emptyMessageBuffer() {
    this.messageBuffer = Buffer.alloc(0);
  }
  /**
   * @private
   */


  addToMessageBuffer(data) {
    this.messageBuffer = Buffer.concat([this.messageBuffer, data]);
  }
  /**
   * @private
   */


  sendLogin7Packet() {
    const payload = new _login7Payload.default({
      tdsVersion: _tdsVersions.versions[this.config.options.tdsVersion],
      packetSize: this.config.options.packetSize,
      clientProgVer: 0,
      clientPid: process.pid,
      connectionId: 0,
      clientTimeZone: new Date().getTimezoneOffset(),
      clientLcid: 0x00000409
    });
    const {
      authentication
    } = this.config;

    switch (authentication.type) {
      case 'azure-active-directory-password':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'default'
        };
        break;

      case 'azure-active-directory-access-token':
        payload.fedAuth = {
          type: 'SECURITYTOKEN',
          echo: this.fedAuthRequired,
          fedAuthToken: authentication.options.token
        };
        break;

      case 'azure-active-directory-msi-vm':
      case 'azure-active-directory-msi-app-service':
      case 'azure-active-directory-service-principal-secret':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'integrated'
        };
        break;

      case 'ntlm':
        payload.sspi = (0, _ntlm.createNTLMRequest)({
          domain: authentication.options.domain
        });
        break;

      default:
        payload.userName = authentication.options.userName;
        payload.password = authentication.options.password;
    }

    payload.hostname = this.config.options.workstationId || _os.default.hostname();
    payload.serverName = this.routingData ? this.routingData.server : this.config.server;
    payload.appName = this.config.options.appName || 'Tedious';
    payload.libraryName = _library.name;
    payload.language = this.config.options.language;
    payload.database = this.config.options.database;
    payload.clientId = Buffer.from([1, 2, 3, 4, 5, 6]);
    payload.readOnlyIntent = this.config.options.readOnlyIntent;
    payload.initDbFatal = !this.config.options.fallbackToDefaultDb;
    this.routingData = undefined;
    this.messageIo.sendMessage(_packet.TYPE.LOGIN7, payload.toBuffer());
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  sendFedAuthTokenMessage(token) {
    const accessTokenLen = Buffer.byteLength(token, 'ucs2');
    const data = Buffer.alloc(8 + accessTokenLen);
    let offset = 0;
    offset = data.writeUInt32LE(accessTokenLen + 4, offset);
    offset = data.writeUInt32LE(accessTokenLen, offset);
    data.write(token, offset, 'ucs2');
    this.messageIo.sendMessage(_packet.TYPE.FEDAUTH_TOKEN, data); // sent the fedAuth token message, the rest is similar to standard login 7

    this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
  }
  /**
   * @private
   */


  sendInitialSql() {
    const payload = new _sqlbatchPayload.default(this.getInitialSql(), this.currentTransactionDescriptor(), this.config.options);
    const message = new _message.default({
      type: _packet.TYPE.SQL_BATCH
    });
    this.messageIo.outgoingMessageStream.write(message);

    _stream.Readable.from(payload).pipe(message);
  }
  /**
   * @private
   */


  getInitialSql() {
    const options = [];

    if (this.config.options.enableAnsiNull === true) {
      options.push('set ansi_nulls on');
    } else if (this.config.options.enableAnsiNull === false) {
      options.push('set ansi_nulls off');
    }

    if (this.config.options.enableAnsiNullDefault === true) {
      options.push('set ansi_null_dflt_on on');
    } else if (this.config.options.enableAnsiNullDefault === false) {
      options.push('set ansi_null_dflt_on off');
    }

    if (this.config.options.enableAnsiPadding === true) {
      options.push('set ansi_padding on');
    } else if (this.config.options.enableAnsiPadding === false) {
      options.push('set ansi_padding off');
    }

    if (this.config.options.enableAnsiWarnings === true) {
      options.push('set ansi_warnings on');
    } else if (this.config.options.enableAnsiWarnings === false) {
      options.push('set ansi_warnings off');
    }

    if (this.config.options.enableArithAbort === true) {
      options.push('set arithabort on');
    } else if (this.config.options.enableArithAbort === false) {
      options.push('set arithabort off');
    }

    if (this.config.options.enableConcatNullYieldsNull === true) {
      options.push('set concat_null_yields_null on');
    } else if (this.config.options.enableConcatNullYieldsNull === false) {
      options.push('set concat_null_yields_null off');
    }

    if (this.config.options.enableCursorCloseOnCommit === true) {
      options.push('set cursor_close_on_commit on');
    } else if (this.config.options.enableCursorCloseOnCommit === false) {
      options.push('set cursor_close_on_commit off');
    }

    if (this.config.options.datefirst !== null) {
      options.push(`set datefirst ${this.config.options.datefirst}`);
    }

    if (this.config.options.dateFormat !== null) {
      options.push(`set dateformat ${this.config.options.dateFormat}`);
    }

    if (this.config.options.enableImplicitTransactions === true) {
      options.push('set implicit_transactions on');
    } else if (this.config.options.enableImplicitTransactions === false) {
      options.push('set implicit_transactions off');
    }

    if (this.config.options.language !== null) {
      options.push(`set language ${this.config.options.language}`);
    }

    if (this.config.options.enableNumericRoundabort === true) {
      options.push('set numeric_roundabort on');
    } else if (this.config.options.enableNumericRoundabort === false) {
      options.push('set numeric_roundabort off');
    }

    if (this.config.options.enableQuotedIdentifier === true) {
      options.push('set quoted_identifier on');
    } else if (this.config.options.enableQuotedIdentifier === false) {
      options.push('set quoted_identifier off');
    }

    if (this.config.options.textsize !== null) {
      options.push(`set textsize ${this.config.options.textsize}`);
    }

    if (this.config.options.connectionIsolationLevel !== null) {
      options.push(`set transaction isolation level ${this.getIsolationLevelText(this.config.options.connectionIsolationLevel)}`);
    }

    if (this.config.options.abortTransactionOnError === true) {
      options.push('set xact_abort on');
    } else if (this.config.options.abortTransactionOnError === false) {
      options.push('set xact_abort off');
    }

    return options.join('\n');
  }
  /**
   * @private
   */


  processedInitialSql() {
    this.clearConnectTimer();
    this.emit('connect');
  }
  /**
   * Execute the SQL batch represented by [[Request]].
   * There is no param support, and unlike [[Request.execSql]],
   * it is not likely that SQL Server will reuse the execution plan it generates for the SQL.
   *
   * In almost all cases, [[Request.execSql]] will be a better choice.
   *
   * @param request A [[Request]] object representing the request.
   */


  execSqlBatch(request) {
    this.makeRequest(request, _packet.TYPE.SQL_BATCH, new _sqlbatchPayload.default(request.sqlTextOrProcedure, this.currentTransactionDescriptor(), this.config.options));
  }
  /**
   *  Execute the SQL represented by [[Request]].
   *
   * As `sp_executesql` is used to execute the SQL, if the same SQL is executed multiples times
   * using this function, the SQL Server query optimizer is likely to reuse the execution plan it generates
   * for the first execution. This may also result in SQL server treating the request like a stored procedure
   * which can result in the [[Event_doneInProc]] or [[Event_doneProc]] events being emitted instead of the
   * [[Event_done]] event you might expect. Using [[execSqlBatch]] will prevent this from occurring but may have a negative performance impact.
   *
   * Beware of the way that scoping rules apply, and how they may [affect local temp tables](http://weblogs.sqlteam.com/mladenp/archive/2006/11/03/17197.aspx)
   * If you're running in to scoping issues, then [[execSqlBatch]] may be a better choice.
   * See also [issue #24](https://github.com/pekim/tedious/issues/24)
   *
   * @param request A [[Request]] object representing the request.
   */


  execSql(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'statement',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    if (request.parameters.length) {
      parameters.push({
        type: _dataType.TYPES.NVarChar,
        name: 'params',
        value: request.makeParamsParameter(request.parameters),
        output: false,
        length: undefined,
        precision: undefined,
        scale: undefined
      });
      parameters.push(...request.parameters);
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_executesql', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Creates a new BulkLoad instance.
   *
   * @param table The name of the table to bulk-insert into.
   * @param options A set of bulk load options.
   */


  newBulkLoad(table, callbackOrOptions, callback) {
    let options;

    if (callback === undefined) {
      callback = callbackOrOptions;
      options = {};
    } else {
      options = callbackOrOptions;
    }

    if (typeof options !== 'object') {
      throw new TypeError('"options" argument must be an object');
    }

    return new _bulkLoad.default(table, this.databaseCollation, this.config.options, options, callback);
  }
  /**
   * Execute a [[BulkLoad]].
   *
   * ```js
   * // We want to perform a bulk load into a table with the following format:
   * // CREATE TABLE employees (first_name nvarchar(255), last_name nvarchar(255), day_of_birth date);
   *
   * const bulkLoad = connection.newBulkLoad('employees', (err, rowCount) => {
   *   // ...
   * });
   *
   * // First, we need to specify the columns that we want to write to,
   * // and their definitions. These definitions must match the actual table,
   * // otherwise the bulk load will fail.
   * bulkLoad.addColumn('first_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('last_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('date_of_birth', TYPES.Date, { nullable: false });
   *
   * // Execute a bulk load with a predefined list of rows.
   * //
   * // Note that these rows are held in memory until the
   * // bulk load was performed, so if you need to write a large
   * // number of rows (e.g. by reading from a CSV file),
   * // passing an `AsyncIterable` is advisable to keep memory usage low.
   * connection.execBulkLoad(bulkLoad, [
   *   { 'first_name': 'Steve', 'last_name': 'Jobs', 'day_of_birth': new Date('02-24-1955') },
   *   { 'first_name': 'Bill', 'last_name': 'Gates', 'day_of_birth': new Date('10-28-1955') }
   * ]);
   * ```
   *
   * @param bulkLoad A previously created [[BulkLoad]].
   * @param rows A [[Iterable]] or [[AsyncIterable]] that contains the rows that should be bulk loaded.
   */


  execBulkLoad(bulkLoad, rows) {
    bulkLoad.executionStarted = true;

    if (rows) {
      if (bulkLoad.streamingMode) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that was put in streaming mode.");
      }

      if (bulkLoad.firstRowWritten) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that already has rows written to it.");
      }

      const rowStream = _stream.Readable.from(rows); // Destroy the packet transform if an error happens in the row stream,
      // e.g. if an error is thrown from within a generator or stream.


      rowStream.on('error', err => {
        bulkLoad.rowToPacketTransform.destroy(err);
      }); // Destroy the row stream if an error happens in the packet transform,
      // e.g. if the bulk load is cancelled.

      bulkLoad.rowToPacketTransform.on('error', err => {
        rowStream.destroy(err);
      });
      rowStream.pipe(bulkLoad.rowToPacketTransform);
    } else if (!bulkLoad.streamingMode) {
      // If the bulkload was not put into streaming mode by the user,
      // we end the rowToPacketTransform here for them.
      //
      // If it was put into streaming mode, it's the user's responsibility
      // to end the stream.
      bulkLoad.rowToPacketTransform.end();
    }

    const onCancel = () => {
      request.cancel();
    };

    const payload = new _bulkLoadPayload.BulkLoadPayload(bulkLoad);
    const request = new _request.default(bulkLoad.getBulkInsertSql(), error => {
      bulkLoad.removeListener('cancel', onCancel);

      if (error) {
        if (error.code === 'UNKNOWN') {
          error.message += ' This is likely because the schema of the BulkLoad does not match the schema of the table you are attempting to insert into.';
        }

        bulkLoad.error = error;
        bulkLoad.callback(error);
        return;
      }

      this.makeRequest(bulkLoad, _packet.TYPE.BULK_LOAD, payload);
    });
    bulkLoad.once('cancel', onCancel);
    this.execSqlBatch(request);
  }
  /**
   * Prepare the SQL represented by the request.
   *
   * The request can then be used in subsequent calls to
   * [[execute]] and [[unprepare]]
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type. Parameter values are ignored.
   */


  prepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      value: undefined,
      output: true,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'params',
      value: request.parameters.length ? request.makeParamsParameter(request.parameters) : null,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'stmt',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    request.preparing = true; // TODO: We need to clean up this event handler, otherwise this leaks memory

    request.on('returnValue', (name, value) => {
      if (name === 'handle') {
        request.handle = value;
      } else {
        request.error = new _errors.RequestError(`Tedious > Unexpected output parameter ${name} from sp_prepare`);
      }
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_prepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Release the SQL Server resources associated with a previously prepared request.
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type.
   *   Parameter values are ignored.
   */


  unprepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_unprepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Execute previously prepared SQL, using the supplied parameters.
   *
   * @param request A previously prepared [[Request]].
   * @param parameters  An object whose names correspond to the names of
   *   parameters that were added to the [[Request]] before it was prepared.
   *   The object's values are passed as the parameters' values when the
   *   request is executed.
   */


  execute(request, parameters) {
    const executeParameters = [];
    executeParameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    try {
      for (let i = 0, len = request.parameters.length; i < len; i++) {
        const parameter = request.parameters[i];
        executeParameters.push({ ...parameter,
          value: parameter.type.validate(parameters ? parameters[parameter.name] : null, this.databaseCollation)
        });
      }
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_execute', executeParameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Call a stored procedure represented by [[Request]].
   *
   * @param request A [[Request]] object representing the request.
   */


  callProcedure(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default(request.sqlTextOrProcedure, request.parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Start a transaction.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`
   *   is present.
   * @param isolationLevel The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  beginTransaction(callback, name = '', isolationLevel = this.config.options.isolationLevel) {
    (0, _transaction.assertValidIsolationLevel)(isolationLevel, 'isolationLevel');
    const transaction = new _transaction.Transaction(name, isolationLevel);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SET TRANSACTION ISOLATION LEVEL ' + transaction.isolationLevelToTSQL() + ';BEGIN TRAN ' + transaction.name, err => {
        this.transactionDepth++;

        if (this.transactionDepth === 1) {
          this.inTransaction = true;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, err => {
      return callback(err, this.currentTransactionDescriptor());
    });
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.beginPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Commit a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`is present.
   */


  commitTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('COMMIT TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.commitPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Rollback a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  rollbackTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('ROLLBACK TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.rollbackPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Set a savepoint within a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.\
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  saveTransaction(callback, name) {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SAVE TRAN ' + transaction.name, err => {
        this.transactionDepth++;
        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.savePayload(this.currentTransactionDescriptor()));
  }
  /**
   * Run the given callback after starting a transaction, and commit or
   * rollback the transaction afterwards.
   *
   * This is a helper that employs [[beginTransaction]], [[commitTransaction]],
   * [[rollbackTransaction]], and [[saveTransaction]] to greatly simplify the
   * use of database transactions and automatically handle transaction nesting.
   *
   * @param cb
   * @param isolationLevel
   *   The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  transaction(cb, isolationLevel) {
    if (typeof cb !== 'function') {
      throw new TypeError('`cb` must be a function');
    }

    const useSavepoint = this.inTransaction;

    const name = '_tedious_' + _crypto.default.randomBytes(10).toString('hex');

    const txDone = (err, done, ...args) => {
      if (err) {
        if (this.inTransaction && this.state === this.STATE.LOGGED_IN) {
          this.rollbackTransaction(txErr => {
            done(txErr || err, ...args);
          }, name);
        } else {
          done(err, ...args);
        }
      } else if (useSavepoint) {
        if (this.config.options.tdsVersion < '7_2') {
          this.transactionDepth--;
        }

        done(null, ...args);
      } else {
        this.commitTransaction(txErr => {
          done(txErr, ...args);
        }, name);
      }
    };

    if (useSavepoint) {
      return this.saveTransaction(err => {
        if (err) {
          return cb(err);
        }

        if (isolationLevel) {
          return this.execSqlBatch(new _request.default('SET transaction isolation level ' + this.getIsolationLevelText(isolationLevel), err => {
            return cb(err, txDone);
          }));
        } else {
          return cb(null, txDone);
        }
      }, name);
    } else {
      return this.beginTransaction(err => {
        if (err) {
          return cb(err);
        }

        return cb(null, txDone);
      }, name, isolationLevel);
    }
  }
  /**
   * @private
   */


  makeRequest(request, packetType, payload) {
    if (this.state !== this.STATE.LOGGED_IN) {
      const message = 'Requests can only be made in the ' + this.STATE.LOGGED_IN.name + ' state, not the ' + this.state.name + ' state';
      this.debug.log(message);
      request.callback(new _errors.RequestError(message, 'EINVALIDSTATE'));
    } else if (request.canceled) {
      process.nextTick(() => {
        request.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
      });
    } else {
      if (packetType === _packet.TYPE.SQL_BATCH) {
        this.isSqlBatch = true;
      } else {
        this.isSqlBatch = false;
      }

      this.request = request;
      request.connection = this;
      request.rowCount = 0;
      request.rows = [];
      request.rst = [];

      const onCancel = () => {
        payloadStream.unpipe(message);
        payloadStream.destroy(new _errors.RequestError('Canceled.', 'ECANCEL')); // set the ignore bit and end the message.

        message.ignore = true;
        message.end();

        if (request instanceof _request.default && request.paused) {
          // resume the request if it was paused so we can read the remaining tokens
          request.resume();
        }
      };

      request.once('cancel', onCancel);
      this.createRequestTimer();
      const message = new _message.default({
        type: packetType,
        resetConnection: this.resetConnectionOnNextRequest
      });
      this.messageIo.outgoingMessageStream.write(message);
      this.transitionTo(this.STATE.SENT_CLIENT_REQUEST);
      message.once('finish', () => {
        request.removeListener('cancel', onCancel);
        request.once('cancel', this._cancelAfterRequestSent);
        this.resetConnectionOnNextRequest = false;
        this.debug.payload(function () {
          return payload.toString('  ');
        });
      });

      const payloadStream = _stream.Readable.from(payload);

      payloadStream.once('error', error => {
        var _request$error;

        payloadStream.unpipe(message); // Only set a request error if no error was set yet.

        (_request$error = request.error) !== null && _request$error !== void 0 ? _request$error : request.error = error;
        message.ignore = true;
        message.end();
      });
      payloadStream.pipe(message);
    }
  }
  /**
   * Cancel currently executed request.
   */


  cancel() {
    if (!this.request) {
      return false;
    }

    if (this.request.canceled) {
      return false;
    }

    this.request.cancel();
    return true;
  }
  /**
   * Reset the connection to its initial state.
   * Can be useful for connection pool implementations.
   *
   * @param callback
   */


  reset(callback) {
    const request = new _request.default(this.getInitialSql(), err => {
      if (this.config.options.tdsVersion < '7_2') {
        this.inTransaction = false;
      }

      callback(err);
    });
    this.resetConnectionOnNextRequest = true;
    this.execSqlBatch(request);
  }
  /**
   * @private
   */


  currentTransactionDescriptor() {
    return this.transactionDescriptors[this.transactionDescriptors.length - 1];
  }
  /**
   * @private
   */


  getIsolationLevelText(isolationLevel) {
    switch (isolationLevel) {
      case _transaction.ISOLATION_LEVEL.READ_UNCOMMITTED:
        return 'read uncommitted';

      case _transaction.ISOLATION_LEVEL.REPEATABLE_READ:
        return 'repeatable read';

      case _transaction.ISOLATION_LEVEL.SERIALIZABLE:
        return 'serializable';

      case _transaction.ISOLATION_LEVEL.SNAPSHOT:
        return 'snapshot';

      default:
        return 'read committed';
    }
  }

}

let azureADPasswordClientIdDeprecationWarningEmitted = false;

function emitAzureADPasswordClientIdDeprecationWarning() {
  if (azureADPasswordClientIdDeprecationWarningEmitted) {
    return;
  }

  azureADPasswordClientIdDeprecationWarningEmitted = true;
  process.emitWarning('When using the `azure-active-directory-password` authentication method, please provide a value for the `clientId` option. ' + 'This option will be required in a future release.', 'DeprecationWarning', Connection.prototype.on);
}

var _default = Connection;
exports.default = _default;
module.exports = Connection;
Connection.prototype.STATE = {
  INITIALIZED: {
    name: 'Initialized',
    events: {}
  },
  CONNECTING: {
    name: 'Connecting',
    enter: function () {
      this.initialiseConnection();
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_PRELOGIN: {
    name: 'SentPrelogin',
    enter: function () {
      this.emptyMessageBuffer();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        message.on('data', data => {
          this.addToMessageBuffer(data);
        });
        message.once('end', () => {
          const preloginPayload = new _preloginPayload.default(this.messageBuffer);
          this.debug.payload(function () {
            return preloginPayload.toString('  ');
          });

          if (preloginPayload.fedAuthRequired === 1) {
            this.fedAuthRequired = true;
          }

          if (preloginPayload.encryptionString === 'ON' || preloginPayload.encryptionString === 'REQ') {
            var _this$routingData$ser, _this$routingData;

            if (!this.config.options.encrypt) {
              this.emit('connect', new _errors.ConnectionError("Server requires encryption, set 'encrypt' config option to true.", 'EENCRYPT'));
              return this.close();
            }

            this.messageIo.startTls(this.secureContext, (_this$routingData$ser = (_this$routingData = this.routingData) === null || _this$routingData === void 0 ? void 0 : _this$routingData.server) !== null && _this$routingData$ser !== void 0 ? _this$routingData$ser : this.config.server, this.config.options.trustServerCertificate);
            this.transitionTo(this.STATE.SENT_TLSSSLNEGOTIATION);
          } else {
            this.sendLogin7Packet();
            const {
              authentication
            } = this.config;

            if (authentication.type === 'ntlm') {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_NTLM);
            } else {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
            }
          }
        });
      }
    }
  },
  REROUTING: {
    name: 'ReRouting',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.REDIRECT);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      reconnect: function () {
        this.transitionTo(this.STATE.CONNECTING);
      }
    }
  },
  TRANSIENT_FAILURE_RETRY: {
    name: 'TRANSIENT_FAILURE_RETRY',
    enter: function () {
      this.curTransientRetryCount++;
      this.cleanupConnection(CLEANUP_TYPE.RETRY);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      retry: function () {
        this.createRetryTimer();
      }
    }
  },
  SENT_TLSSSLNEGOTIATION: {
    name: 'SentTLSSSLNegotiation',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        message.on('data', data => {
          this.messageIo.tlsHandshakeData(data);
        });
        message.once('end', () => {
          if (this.messageIo.tlsNegotiationComplete) {
            this.sendLogin7Packet();
            const {
              authentication
            } = this.config;

            if (authentication.type === 'azure-active-directory-password' || authentication.type === 'azure-active-directory-msi-vm' || authentication.type === 'azure-active-directory-msi-app-service' || authentication.type === 'azure-active-directory-service-principal-secret') {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_FEDAUTH);
            } else if (authentication.type === 'ntlm') {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_NTLM);
            } else {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
            }
          } else {
            this.messageIo.readMessage().then(message => {
              this.dispatchEvent('message', message);
            }, err => {
              this.socketError(err);
            });
          }
        });
      }
    }
  },
  SENT_LOGIN7_WITH_STANDARD_LOGIN: {
    name: 'SentLogin7WithStandardLogin',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              this.transitionTo(this.STATE.REROUTING);
            } else {
              this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }
          } else if (this.loginError) {
            if (this.loginError.isTransient) {
              this.debug.log('Initiating retry on transient error');
              this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            this.transitionTo(this.STATE.FINAL);
          }
        });
      }
    }
  },
  SENT_LOGIN7_WITH_NTLM: {
    name: 'SentLogin7WithNTLMLogin',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              this.transitionTo(this.STATE.REROUTING);
            } else {
              this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }
          } else if (this.ntlmpacket) {
            const authentication = this.config.authentication;
            const payload = new _ntlmPayload.default({
              domain: authentication.options.domain,
              userName: authentication.options.userName,
              password: authentication.options.password,
              ntlmpacket: this.ntlmpacket
            });
            this.messageIo.sendMessage(_packet.TYPE.NTLMAUTH_PKT, payload.data);
            this.debug.payload(function () {
              return payload.toString('  ');
            });
            this.ntlmpacket = undefined;
            this.messageIo.readMessage().then(message => {
              this.dispatchEvent('message', message);
            }, err => {
              this.socketError(err);
            });
          } else if (this.loginError) {
            if (this.loginError.isTransient) {
              this.debug.log('Initiating retry on transient error');
              this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            this.transitionTo(this.STATE.FINAL);
          }
        });
      }
    }
  },
  SENT_LOGIN7_WITH_FEDAUTH: {
    name: 'SentLogin7Withfedauth',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              this.transitionTo(this.STATE.REROUTING);
            } else {
              this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }

            return;
          }

          const fedAuthInfoToken = handler.fedAuthInfoToken;

          if (fedAuthInfoToken && fedAuthInfoToken.stsurl && fedAuthInfoToken.spn) {
            const authentication = this.config.authentication;
            const tokenScope = new _url.URL('/.default', fedAuthInfoToken.spn).toString();

            const getToken = callback => {
              const getTokenFromCredentials = credentials => {
                credentials.getToken(tokenScope).then(tokenResponse => {
                  callback(null, tokenResponse === null || tokenResponse === void 0 ? void 0 : tokenResponse.token);
                }, callback);
              };

              if (authentication.type === 'azure-active-directory-password') {
                var _authentication$optio;

                const credentials = new _identity.UsernamePasswordCredential((_authentication$optio = authentication.options.tenantId) !== null && _authentication$optio !== void 0 ? _authentication$optio : 'common', authentication.options.clientId, authentication.options.userName, authentication.options.password);
                getTokenFromCredentials(credentials);
              } else if (authentication.type === 'azure-active-directory-msi-vm' || authentication.type === 'azure-active-directory-msi-app-service') {
                const msiArgs = authentication.options.clientId ? [authentication.options.clientId, {}] : [{}];
                const credentials = new _identity.ManagedIdentityCredential(...msiArgs);
                getTokenFromCredentials(credentials);
              } else if (authentication.type === 'azure-active-directory-service-principal-secret') {
                const credentials = new _identity.ClientSecretCredential(authentication.options.tenantId, authentication.options.clientId, authentication.options.clientSecret);
                getTokenFromCredentials(credentials);
              }
            };

            getToken((err, token) => {
              if (err) {
                this.loginError = new _errors.ConnectionError('Security token could not be authenticated or authorized.', 'EFEDAUTH');
                this.emit('connect', this.loginError);
                this.transitionTo(this.STATE.FINAL);
                return;
              }

              this.sendFedAuthTokenMessage(token);
            });
          } else if (this.loginError) {
            if (this.loginError.isTransient) {
              this.debug.log('Initiating retry on transient error');
              this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            this.transitionTo(this.STATE.FINAL);
          }
        });
      }
    }
  },
  LOGGED_IN_SENDING_INITIAL_SQL: {
    name: 'LoggedInSendingInitialSql',
    enter: function () {
      this.sendInitialSql();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function socketError() {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.InitialSqlTokenHandler(this));
        tokenStreamParser.once('end', () => {
          this.transitionTo(this.STATE.LOGGED_IN);
          this.processedInitialSql();
        });
      }
    }
  },
  LOGGED_IN: {
    name: 'LoggedIn',
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_CLIENT_REQUEST: {
    name: 'SentClientRequest',
    enter: function () {
      this.emptyMessageBuffer();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    exit: function (nextState) {
      this.clearRequestTimer();
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      },
      message: function (message) {
        var _this$request, _this$request3, _this$request10;

        // request timer is stopped on first data package
        this.clearRequestTimer();
        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.RequestTokenHandler(this, this.request)); // If the request was canceled and we have a `cancelTimer`
        // defined, we send a attention message after the
        // request message was fully sent off.
        //
        // We already started consuming the current message
        // (but all the token handlers should be no-ops), and
        // need to ensure the next message is handled by the
        // `SENT_ATTENTION` state.

        if ((_this$request = this.request) !== null && _this$request !== void 0 && _this$request.canceled && this.cancelTimer) {
          return this.transitionTo(this.STATE.SENT_ATTENTION);
        }

        const onResume = () => {
          tokenStreamParser.resume();
        };

        const onPause = () => {
          var _this$request2;

          tokenStreamParser.pause();
          (_this$request2 = this.request) === null || _this$request2 === void 0 ? void 0 : _this$request2.once('resume', onResume);
        };

        (_this$request3 = this.request) === null || _this$request3 === void 0 ? void 0 : _this$request3.on('pause', onPause);

        if (this.request instanceof _request.default && this.request.paused) {
          onPause();
        }

        const onCancel = () => {
          var _this$request4, _this$request5;

          tokenStreamParser.removeListener('end', onEndOfMessage);

          if (this.request instanceof _request.default && this.request.paused) {
            // resume the request if it was paused so we can read the remaining tokens
            this.request.resume();
          }

          (_this$request4 = this.request) === null || _this$request4 === void 0 ? void 0 : _this$request4.removeListener('pause', onPause);
          (_this$request5 = this.request) === null || _this$request5 === void 0 ? void 0 : _this$request5.removeListener('resume', onResume); // The `_cancelAfterRequestSent` callback will have sent a
          // attention message, so now we need to also switch to
          // the `SENT_ATTENTION` state to make sure the attention ack
          // message is processed correctly.

          this.transitionTo(this.STATE.SENT_ATTENTION);
        };

        const onEndOfMessage = () => {
          var _this$request6, _this$request7, _this$request8, _this$request9;

          (_this$request6 = this.request) === null || _this$request6 === void 0 ? void 0 : _this$request6.removeListener('cancel', this._cancelAfterRequestSent);
          (_this$request7 = this.request) === null || _this$request7 === void 0 ? void 0 : _this$request7.removeListener('cancel', onCancel);
          (_this$request8 = this.request) === null || _this$request8 === void 0 ? void 0 : _this$request8.removeListener('pause', onPause);
          (_this$request9 = this.request) === null || _this$request9 === void 0 ? void 0 : _this$request9.removeListener('resume', onResume);
          this.transitionTo(this.STATE.LOGGED_IN);
          const sqlRequest = this.request;
          this.request = undefined;

          if (this.config.options.tdsVersion < '7_2' && sqlRequest.error && this.isSqlBatch) {
            this.inTransaction = false;
          }

          sqlRequest.callback(sqlRequest.error, sqlRequest.rowCount, sqlRequest.rows);
        };

        tokenStreamParser.once('end', onEndOfMessage);
        (_this$request10 = this.request) === null || _this$request10 === void 0 ? void 0 : _this$request10.once('cancel', onCancel);
      }
    }
  },
  SENT_ATTENTION: {
    name: 'SentAttention',
    enter: function () {
      this.emptyMessageBuffer();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      },
      message: function (message) {
        const handler = new _handler.AttentionTokenHandler(this, this.request);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          // 3.2.5.7 Sent Attention State
          // Discard any data contained in the response, until we receive the attention response
          if (handler.attentionReceived) {
            this.clearCancelTimer();
            const sqlRequest = this.request;
            this.request = undefined;
            this.transitionTo(this.STATE.LOGGED_IN);

            if (sqlRequest.error && sqlRequest.error instanceof _errors.RequestError && sqlRequest.error.code === 'ETIMEOUT') {
              sqlRequest.callback(sqlRequest.error);
            } else {
              sqlRequest.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
            }
          }
        });
      }
    }
  },
  FINAL: {
    name: 'Final',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.NORMAL);
    },
    events: {
      connectTimeout: function () {// Do nothing, as the timer should be cleaned up.
      },
      message: function () {// Do nothing
      },
      socketError: function () {// Do nothing
      }
    }
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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