var url = require('url');
var fs = require('fs');
var path = require('path');
var util = require('util');
var querystring = require('querystring');
var utils = require('utils');
var zlib = require('zlib');

/**
 * 任讯前端页面处理器
 */
var WXFPHandler = function () {
  this.tfp = null;
  this.tfpRootPath = "/node_modules/tfp";
  this.pageSrcPath = "/app";
  this.pageDistPath = "/dist";
  this.jqueryPath = "/node_modules/jquery/dist/jquery.min.js";
  this.filePath = null;
  this.metaDatas = {};
  this.cssFiles = [];
  this.jsFiles = [];
  this.cptStyles = [];
  this.isDebug = false;
};

module.exports = WXFPHandler;

WXFPHandler.prototype.process = function (request, response) {
  //解析物理路径
  request.urlPath = url.parse(request.url).pathname;
  //去掉路径中的“../”，防止恶意访问
  request.urlPath = path.normalize(request.urlPath.replace(/\.\./g, ''));
  request.urlPath = utils.formatPath(request.urlPath);
  request.wxfpFilePath = utils.getPath(utils.appendPath('/miniprogram', request.urlPath));
  if (!fs.existsSync(request.wxfpFilePath)) {
    server.responseErrorByHTML(request, response, 404, '请求地址无效。');
    return;
  }
  var self = this;
  fs.stat(request.wxfpFilePath, function (err, stats) {
    if (err) {
      logger.log(err);
      server.responseErrorByHTML(request, response, 500, '读取文件失败。');
    } else {
      self.handle(request, response, stats);
    }
  });
};

WXFPHandler.prototype.handle = function (request, response, stats) {
  var lastModified = stats.mtime.toUTCString();
  var ifModifiedSince = 'If-Modified-Since'.toLowerCase();
  if (request.headers[ifModifiedSince] && lastModified == request.headers[ifModifiedSince]) {
    server.endResponse(request, response, 304, 'Not Modified');
    return;
  }

  let config = {
    "tfpRootPath": "/node_modules/tfp",
    "pageSrcPath": "/app",
    "pageDistPath": "/dist",
    "jqueryPath": "/node_modules/jquery/dist/jquery.min.js"
  };
  if (global.server.config.TFPHandler) {
    if (global.server.config.TFPHandler.tfpRootPath) config.tfpRootPath = global.server.config.TFPHandler.tfpRootPath;
    if (global.server.config.TFPHandler.pageSrcPath) config.pageSrcPath = global.server.config.TFPHandler.pageSrcPath;
    if (global.server.config.TFPHandler.pageDistPath) config.pageDistPath = global.server.config.TFPHandler.pageDistPath;
    if (global.server.config.TFPHandler.jqueryPath) config.jqueryPath = global.server.config.TFPHandler.jqueryPath;
  }
  let distFilePath = request.urlPath.replace(config.pageSrcPath, config.pageDistPath);
  distFilePath = utils.getPath(utils.appendPath('/web', distFilePath));

  var req = url.parse(request.url, true).query;

  let statsHtml;
  try {
    //如果Html文件不存在，则重新编译
    if (req._isDebug || !fs.existsSync(distFilePath + ".html")) {
      this.compile(request, response, req, config, distFilePath);
      return;
    }
    statsHtml = fs.statSync(distFilePath + ".html");
    //如果Html文件的修改时间早于tfp文件的修改时间，则重新编译
    if (stats.mtime > statsHtml.mtime) {
      this.compile(request, response, req, config, distFilePath);
      return;
    }
  } catch (err) {
    logger.log(err);
    server.responseErrorByHTML(request, response, 500, '读取html文件失败。');
    return;
  }
  request.physicalPath = distFilePath + ".html";
  server.staticHandle(request, response, statsHtml);
};

WXFPHandler.prototype.compile = function (request, response, req, config, distFilePath) {
  var self = this;
  fs.readFile(request.tfpFilePath, "utf-8", function (err, data) {
    if (err) {
      logger.log(err);
      server.responseErrorByHTML(request, response, 500, '读取TFP文件失败。');
      return;
    }
    let pageDataModel;
    try {
      pageDataModel = JSON.parse(data);
    } catch (e) {
      logger.log(e);
      server.responseErrorByHTML(request, response, 500, '解析TFP文件失败。');
      return;
    }
    let options = {
      urlPath: request.urlPath,
      pageDataModel: pageDataModel,
      createFile: true
    };
    if (req._isDebug) options.isDebug = true;
    self.compileTfp(options, function (err, html) {
      if (err) {
        logger.log(err);
        server.responseErrorByHTML(request, response, 500, '编译TFP文件失败。');
        return;
      }
      self.responseHtml(request, response, html);
    });
  });
};

WXFPHandler.prototype.responseHtml = function (request, response, html) {
  try {
    response.writeHead(200, { 'Content-Type': 'text/html; charset=utf-8' });
    response.write(html);
    //统计响应数据
    let statRes = server.stat.res["200"];
    if (!statRes) {
      server.stat.res["200"] = 1;
    } else {
      server.stat.res["200"]++;
    }
  } catch (err) { }
  try {
    response.end();
  } catch (err) { }
  //删除连接信息
  server.delNetCon(request._id);
  html = null;

  logger.logRequestToDBEnd(request, response, 200, 'OK');
};

WXFPHandler.prototype.compileTfp = function (options, cb) {
  this.filePath = options.urlPath;
  this.createFile = options.createFile;
  if (options.isDebug) this.isDebug = true;

  if (global.server && global.server.config && global.server.config.TFPHandler) {
    if (global.server.config.TFPHandler.tfpRootPath) this.tfpRootPath = global.server.config.TFPHandler.tfpRootPath;
    if (global.server.config.TFPHandler.pageSrcPath) this.pageSrcPath = global.server.config.TFPHandler.pageSrcPath;
    if (global.server.config.TFPHandler.pageDistPath) this.pageDistPath = global.server.config.TFPHandler.pageDistPath;
    if (global.server.config.TFPHandler.jqueryPath) this.jqueryPath = global.server.config.TFPHandler.jqueryPath;
    if (global.server.config.TFPHandler.isDebug) this.isDebug = true;
  }

  this.clientType = options.pageDataModel.client;
  if (this.clientType == "tb") this.clientType = "pc";

  // global.server.tfpLibFilesCache["mini"] = global.server.tfpLibFilesCache["phone"];


  this.tfp = new global.server.tfpLibFilesCache.TaskFrontPage();
  this.tfp.metaDatas = global.server.tfpLibFilesCache[this.clientType].metaDatas;
  this.tfp.controllers = global.server.tfpLibFilesCache[this.clientType].controllers;
  this.tfp.renders = global.server.tfpLibFilesCache[this.clientType].renders;
  this.tfp.isLoadingPage = true;
  this.parseCpt(options.pageDataModel);
  this.tfp.isLoadingPage = false;

  if (!this.createFile) {
    this.getHtml(null, cb);
    return;
  }

  this.setDistPath(cb);
};

WXFPHandler.prototype.setDistPath = function (cb) {
  this.fileName = path.basename(this.filePath);
  console.log("this.fileName:" + this.fileName);
  let arrDir = [];
  if (this.filePath.startWith("/node_modules")) {
    this.dirDist = path.dirname(this.filePath).replace("/node_modules", "");
    arrDir = this.dirDist.split("/");
    this.dirDist = "/web/node_modules";
  } else {
    this.dirDist = path.dirname(this.filePath).replace(this.pageSrcPath, "");
    arrDir = this.dirDist.split("/");
    this.dirDist = "/web/dist";
  }
  this.dirDist = utils.getPath(this.dirDist);
  let self = this;
  let tfpPath = ".." + this.tfpRootPath;
  try {
    for (let i = 0; i < arrDir.length; i++) {
      if (arrDir[i] == "") continue;
      this.dirDist += "/" + arrDir[i];
      tfpPath = "../" + tfpPath;
      if (!fs.existsSync(this.dirDist)) fs.mkdirSync(this.dirDist);
    }
  } catch (err) {
    logger.log(err);
    cb(err);
    return;
  }
  this.createModuleJs(tfpPath, cb);
};

WXFPHandler.prototype.createModuleJs = function (tfpPath, cb) {
  logger.log("正在生成" + this.filePath + "的js代码...");
  let pageDataModel = this.tfp.curPage.dataModel;
  let js = "import TaskFrontPage from '" + tfpPath + "/lib/tfp.js'\r\n";
  js += "import TFPMetaDatas from '" + tfpPath + "/lib/components." + this.clientType + ".js'\r\n";
  for (let cptType in this.metaDatas) {
    let cptClass = this.tfp.type(cptType);
    //js += "import "+cptType+"MetaData from '"+this.tfpRootPath+"/src/components/"+cptType.toLowerCase()+"/metadata.js'\r\n";
    if (global.server.tfpLibFilesCache.matchAllClientCpts.includes(cptClass.name)) {
      js += "import " + cptType + " from '" + tfpPath + "/lib/components/"
        + cptType.toLowerCase() + "/controller.js'\r\n";
    } else {
      js += "import " + cptType + " from '" + tfpPath + "/lib/components/" + cptType.toLowerCase()
        + "/controller." + this.clientType + ".js'\r\n";
    }
  }
  js += "\r\n";
  js += "var tfp = new TaskFrontPage();\r\n";
  js += "tfp.rootPath = \"" + tfpPath + "\";\r\n";
  js += "tfp.setComponentMetaDatas(TFPMetaDatas);\r\n";
  for (let cptType in this.metaDatas) {
    //js += "tfp.metaDatas['"+cptType+"'] = "+cptType+"MetaData;\r\n";
    js += "tfp.controllers['" + cptType + "'] = " + cptType + ";\r\n";
  }
  js += "\r\n";
  js += "window.tfp = tfp;\r\n";
  js += "tfp.extendSysObj();\r\n";

  //console.log(js);

  this.babel(this.dirDist + "/" + this.fileName, js, cb);
};

WXFPHandler.prototype.babel = function (fileName, js, cb) {
  logger.log("正在将" + this.filePath + "的js代码转换为ES5格式...");
  var self = this;
  var babel = server.loadModule("/node_modules/@babel/core/lib/index.js");
  let babelOptions = {
    "presets": [
      [
        "@babel/preset-env"
      ]
    ]
    /*,
    "plugins": [
      [
        "@babel/plugin-transform-runtime"
      ]
    ]*/
  };
  babel.transform(js, babelOptions, function (err, result) {
    if (err) {
      logger.log(err);
      cb(err);
      return;
    }

    //console.log(result.code); // => { code, map, ast }

    let tmpFilePath = fileName + ".tmp.js";
    fs.open(tmpFilePath, 'w+', (err, fd) => {
      if (err) {
        logger.log(err);
        fs.closeSync(fd);
        cb(err);
        return;
      }
      fs.write(fd, result.code, (err2) => {
        fs.closeSync(fd);
        if (err2) {
          logger.log(err2);
          cb(err2);
          return;
        }
        self.browserify(fileName, cb);
      });
    });
  });
};

WXFPHandler.prototype.browserify = function (fileName, cb) {
  logger.log("正在打包" + this.filePath + "的js代码...");
  var self = this;
  let tmpFilePath = fileName + ".tmp.js";
  var browserify = server.loadModule("/node_modules/browserify/index.js");
  var b = browserify();
  b.add(tmpFilePath);
  b.bundle(function (err, buf) {
    if (err) {
      logger.log(err);
      cb(err);
      return;
    }
    fs.unlinkSync(tmpFilePath);
    //开启调试模式时，不压缩代码
    if (self.isDebug) {
      self.getHtml(buf.toString("utf8"), cb);
    } else {
      self.uglify(fileName, buf.toString("utf8"), cb);
    }
  });
};

WXFPHandler.prototype.uglify = function (fileName, jsTmp, cb) {
  logger.log("正在压缩" + this.filePath + "的js代码...");
  var UglifyJS = server.loadModule("/node_modules/uglify-js/tools/node.js");
  var result = UglifyJS.minify(jsTmp);

  this.getHtml(result.code, cb);
};

WXFPHandler.prototype.getHtml = function (jsCode, cb) {
  logger.log("正在生成" + this.filePath + "的HTML代码...");
  let pageDataModel = this.tfp.curPage.dataModel;
  let htmlHead = "\t<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\" />\r\n";
  if (this.clientType == "phone") htmlHead += "\t<meta name=\"viewport\" content=\"width=device-width, initial-scale=1, maximum-scale=1\" />\r\n";
  if (pageDataModel.title) {
    htmlHead += "\t<title>" + pageDataModel.title + "</title>\r\n";
  } else {
    htmlHead += "\t<title></title>\r\n";
  }
  //添加组件引用的远程CSS文件
  for (var i = 0; i < this.cssFiles.length; i++) {
    let styleFile = this.cssFiles[i];
    if (styleFile.onlyRuntime) styleFile = styleFile.src;
    if (styleFile.startsWith("http://") || styleFile.startsWith("https://")) {
      htmlHead += "\t<link rel=\"stylesheet\" type=\"text/css\" href=\"" + styleFile + "\" />\r\n";
    }
  }
  //添加当前页面引用的外部CSS样式文件
  if (pageDataModel.cssFiles) {
    for (var i = 0; i < pageDataModel.cssFiles.length; i++) {
      let styleFile = pageDataModel.cssFiles[i];
      htmlHead += "\t<link rel=\"stylesheet\" type=\"text/css\" href=\"" + styleFile + "\" />\r\n";
    }
  }
  //添加公共JS文件
  htmlHead += "\t<script type=\"text/javascript\" src=\"" + this.jqueryPath + "\"></script>\r\n";
  if (pageDataModel.jsFiles) {
    //添加当前页面引用的JS文件
    for (var i = 0; i < pageDataModel.jsFiles.length; i++) {
      let jsFile = pageDataModel.jsFiles[i];
      if (jsFile.onlyRuntime) jsFile = jsFile.src;
      htmlHead += "\t<script type=\"text/javascript\" src=\"" + jsFile + "\"></script>\r\n";
    }
  }
  //添加组件扩展的JS文件
  for (var i = 0; i < this.jsFiles.length; i++) {
    let jsFile = this.jsFiles[i];
    if (jsFile.onlyRuntime) jsFile = jsFile.src;
    htmlHead += "\t<script type=\"text/javascript\" src=\"" + jsFile + "\"></script>\r\n";
  }

  if (pageDataModel.jsGlobalVars || pageDataModel.jsFuncs) {
    htmlHead += "\t<script type=\"text/javascript\">\r\n";
    if (pageDataModel.jsGlobalVars) {
      for (var i = 0; i < pageDataModel.jsGlobalVars.length; i++) {
        let v = pageDataModel.jsGlobalVars[i];
        htmlHead += "\t\tvar " + v.name;
        if (v.value) htmlHead += " = " + v.value;
        htmlHead += ";\r\n";
      }
    }
    if (pageDataModel.jsFuncs) {
      for (var i = 0; i < pageDataModel.jsFuncs.length; i++) {
        let func = pageDataModel.jsFuncs[i];
        htmlHead += "\t\tfunction " + func.name + "(";
        for (var j = 0; j < func.args.length; j++) {
          if (j > 0) htmlHead += " , ";
          htmlHead += func.args[j].name;
        }
        htmlHead += ") {\r\n";
        if (func.code) htmlHead += "\t\t" + func.code.replace(/\n/g, "\n\t\t");
        htmlHead += "}\r\n";
      }
    }
    htmlHead += "</script>\r\n";
  }

  //设置当前页面的CSS代码
  let cssCode = "";
  if (this.createFile) {
    let bgColorMode = "light";
    if (pageDataModel.bgColorMode == "dark") bgColorMode = "dark";
    cssCode = global.server.tfpLibFilesCache.cssFiles[this.tfpRootPath + "/src/tfp." + bgColorMode + ".css"] + "\r\n";
    //添加组件默认的CSS文件
    for (var i = 0; i < this.cssFiles.length; i++) {
      let styleFile = this.cssFiles[i];
      styleFile = styleFile.replace("{client}", this.clientType);
      styleFile = styleFile.replace("{bgColorMode}", bgColorMode);
      if (styleFile.startsWith("http://") || styleFile.startsWith("https://")) continue;
      cssCode += global.server.tfpLibFilesCache.cssFiles[styleFile] + "\r\n";
    }
  }
  //添加当前页面的CSS
  if (pageDataModel.styleSheets) {
    for (var i = 0; i < pageDataModel.styleSheets.length; i++) {
      let ss = pageDataModel.styleSheets[i];
      cssCode += "\t\t" + ss.id + " {\r\n";
      if (ss.styles) {
        for (var j = 0; j < ss.styles.length; j++) {
          let s = ss.styles[j];
          cssCode += "\t\t\t" + s.name + ": " + s.value + ";\r\n";
        }
      }
      cssCode += "\t\t}\r\n";
    }
  }
  //添加页面组件的CSS
  if (this.cptStyles.length > 0) {
    for (var i = 0; i < this.cptStyles.length; i++) {
      let ss = this.cptStyles[i];
      cssCode += "\t\t#" + ss.id + " {" + ss.style + "}\r\n";
    }
  }

  if (this.createFile) {
    htmlHead += "\t<script type=\"text/javascript\">\r\n" + jsCode + "\r\n\t</script>\r\n";
    //压缩CSS代码
    var uglifycss = server.loadModule("/node_modules/uglifycss/index.js");
    cssCode = uglifycss.processString(cssCode) + "\r\n";
  }

  htmlHead += "\t<style>\r\n" + cssCode + "\t</style>\r\n";

  var html = "<html>\r\n<head>\r\n" + htmlHead + "</head>\r\n<body class=\"tfp-page";
  if (pageDataModel.class) html += " " + pageDataModel.class;
  html += "\"";
  if (pageDataModel.style || pageDataModel.styles) {
    html += " style=\"";
    if (pageDataModel.styles) {
      for (style in pageDataModel.styles) {
        html += style + ": " + pageDataModel.styles[style] + "; ";
      }
    }
    if (pageDataModel.style) html += pageDataModel.style;
    html += "\"";
  }
  let pageMetaData = this.tfp.type("Page");
  for (var i = 0; i < pageMetaData.events.length; i++) {
    let eventName = pageMetaData.events[i].name;
    if (eventName.toLowerCase() == "onload") continue;
    if (pageDataModel[eventName])
      html += " " + eventName.toLowerCase() + "=\"" + pageDataModel[eventName] + "\"";
  }
  //获得页面主体部分的html代码
  let PageRender = this.tfp.renders["Page"];
  let pageRender = new PageRender(this.tfp, pageDataModel);
  html += ">\r\n" + pageRender.getHtml(true) + "</body>\r\n";

  //下述代码不能放在$(function(){})里，那样得等整个页面所有元素渲染完才能执行
  //会造成延迟请求后台初始数据，页面界面显示后卡几秒才请求后台服务
  //********************************************************************
  html += "<script type=\"text/javascript\">\r\n";
  html += "tfp.parsePage("
    + JSON.stringify(pageDataModel, function (key, value) {
      if (["cssFiles", "jsFiles", "jsGlobalVars", "jsFuncs",
        'styles', "styleSheets"].includes(key)) {
        return undefined;
      }
      return value;
    }, "") + ");\r\n";
  html += "</script>\r\n";
  //********************************************************************

  html += "\r\n</html>";

  //console.log(html);

  //如果只需要html代码，或正在调试，则不创建html文件
  if (!this.createFile || this.isDebug) {
    cb(null, html);
    return;
  }

  this.createHtmlFile(html, cb);
};

WXFPHandler.prototype.createHtmlFile = function (html, cb) {
  logger.log("正在保存" + this.filePath + "的HTML代码...");
  let htmlFilePath = this.dirDist + "/" + this.fileName + ".html";
  var self = this;
  fs.open(htmlFilePath, 'w+', (err, fd) => {
    if (err) {
      logger.log(err);
      fs.closeSync(fd);
      cb(err);
      return;
    }
    fs.write(fd, html, (err2) => {
      fs.closeSync(fd);
      if (err2) {
        logger.log(err2);
        cb(err2);
        return;
      }
      logger.log(self.filePath + "的HTML代码保存完毕。");
      cb(null, html);
    });
  });
};


WXFPHandler.prototype.parseCpt = function (dataModel, parent) {
  if (!dataModel) {
    console.error("请提供组件的数据模型！");
    return null;
  }
  if (!dataModel.type) {
    console.error("请提供组件类型！");
    return null;
  }
  var cptMetaData = this.tfp.type(dataModel.type);
  if (!cptMetaData) {
    throw new Error("未找到名为[" + dataModel.type + "]的组件类型定义信息！");
    return null;
  }
  if (!this.metaDatas[dataModel.type]) this.metaDatas[dataModel.type] = cptMetaData;
  var cptClass = this.tfp.controllers[dataModel.type];
  if (!cptClass) {
    throw new Error("未找到名为[" + dataModel.type + "]的组件类别！");
    return;
  }

  var cpt = new cptClass(this.tfp, dataModel, parent);
  var Render = this.tfp.renders[dataModel.type];
  var render = new Render(this.tfp, dataModel, cpt.level);
  if (dataModel.type == "Page") this.tfp.curPage = cpt;

  if (cptMetaData.cssFiles) {
    let cssFiles = [];
    if (typeof (cptMetaData.cssFiles) === "string") {
      cssFiles.push(cptMetaData.cssFiles);
    } else if (Array.isArray(cptMetaData.cssFiles)) {
      cssFiles = cptMetaData.cssFiles;
    }
    let bgColorMode = "light";
    if (this.tfp.curPage.dataModel.bgColorMode == "dark") bgColorMode = "dark";
    for (var i = 0; i < cssFiles.length; i++) {
      let cssFile = cssFiles[i];
      if (cssFile.onlyRuntime) {
        cssFile = cssFile.src;
      }
      cssFile = cssFile.replace("{client}", this.clientType);
      cssFile = cssFile.replace("{bgColorMode}", bgColorMode);
      cssFile = this.tfp.getCptIncludeFile(cssFile, dataModel.type);
      if (!this.cssFiles.includes(cssFile)) this.cssFiles.push(cssFile);
    }
  }
  if (cptMetaData.jsFiles) {
    let jsFiles = [];
    if (typeof (cptMetaData.jsFiles) === "string") {
      jsFiles.push(cptMetaData.jsFiles);
    } else if (Array.isArray(cptMetaData.jsFiles)) {
      jsFiles = cptMetaData.jsFiles;
    }
    for (var i = 0; i < jsFiles.length; i++) {
      let jsFile = jsFiles[i];
      if (jsFile.onlyRuntime) {
        jsFile = jsFile.src;
      }
      jsFile = this.tfp.getCptIncludeFile(jsFile, dataModel.type);
      if (!this.jsFiles.includes(jsFile)) this.jsFiles.push(jsFile);
    }
  }

  let cptStyle = render.getStyleHtml(true);
  if (cptStyle) this.cptStyles.push({ id: cpt.id, style: cptStyle });
  if (dataModel.components) {
    for (var i = 0; i < dataModel.components.length; i++) {
      let cdmChild = dataModel.components[i];
      let cptChild = this.parseCpt(cdmChild, cpt);
    }
  }
};



WXFPHandler.prototype.compileWXFP = function (options, cb) {
  this.filePath = options.urlPath;
  this.createFile = options.createFile;
  this.relativeDir = options.relativeDir;
  if (options.isDebug) this.isDebug = true;

  if (global.server && global.server.config && global.server.config.WXFPHandler) {
    if (global.server.config.TFPHandler.wxfpRootPath) this.wxfpRootPath = global.server.config.TFPHandler.wxfpRootPath;
    if (global.server.config.TFPHandler.pageSrcPath) this.pageSrcPath = global.server.config.TFPHandler.pageSrcPath;
    if (global.server.config.TFPHandler.pageDistPath) this.pageDistPath = global.server.config.TFPHandler.pageDistPath;
    // if (global.server.config.TFPHandler.jqueryPath) this.jqueryPath = global.server.config.TFPHandler.jqueryPath;
    if (global.server.config.TFPHandler.isDebug) this.isDebug = true;
  }

  this.clientType = options.pageDataModel.client;
  if (this.clientType == "tb") this.clientType = "pc";

  //global.server.tfpLibFilesCache["mini"] = global.server.tfpLibFilesCache["phone"]; //TODO

  this.tfp = new global.server.tfpLibFilesCache.TaskFrontPage();
  this.tfp.metaDatas = global.server.tfpLibFilesCache[this.clientType].metaDatas;
  this.tfp.controllers = global.server.tfpLibFilesCache[this.clientType].controllers;
  this.tfp.renders = global.server.tfpLibFilesCache[this.clientType].renders;
  this.tfp.isLoadingPage = true;
  this.parseCpt(options.pageDataModel);
  this.tfp.isLoadingPage = false;

  if (!this.createFile) {
    this.getHtmlWX(null, cb);
    return;
  }

  // this.setDistPath(cb);
};

WXFPHandler.prototype.getHtmlWX = function (jsCode, cb) {
  logger.log("正在生成" + this.filePath + "的WX代码...");
  let pageDataModel = this.tfp.curPage.dataModel;

  var wx = {
    wxjson: "",
    wxjs: "",
    wxml: "",
    wxss: ""
  };

  //设置当前页面的CSS代码
  //添加当前页面的CSS
  if (pageDataModel.styleSheets) {
    for (var i = 0; i < pageDataModel.styleSheets.length; i++) {
      let ss = pageDataModel.styleSheets[i];
      wx.wxss += "\t\t" + ss.id + " {\r\n";
      if (ss.styles) {
        for (var j = 0; j < ss.styles.length; j++) {
          let s = ss.styles[j];
          wx.wxss += "\t\t\t" + s.name + ": " + s.value + ";\r\n";
        }
      }
      wx.wxss += "\t\t}\r\n";
    }
  }

  //添加页面组件的CSS
  if (this.cptStyles.length > 0) {
    for (var i = 0; i < this.cptStyles.length; i++) {
      let ss = this.cptStyles[i];
      wx.wxss += "\t\t.w-" + escape(ss.id).replaceAll("%", "") + " {" + ss.style + "}\r\n";
    }
  }

  let pageMetaData = this.tfp.type("Page");

  let PageRender = this.tfp.renders["Page"];
  let pageRender = new PageRender(this.tfp, pageDataModel);
  var wx0 = pageRender.getWX(true);
  wx.wxml += wx0.wxml;

  wx.wxjson += "{\r\n";
  wx.wxjson += "  \"usingComponents\": {}\r\n";
  wx.wxjson += "}\r\n";

  wx.wxjs += "const app = getApp()\r\n";
  wx.wxjs += "const util = require('" + this.relativeDir + "utils/util.js')\r\n";
  // wx.wxjs += "const wxfp = require('" + this.relativeDir + "utils/wxfp.js')\r\n";
  wx.wxjs += "import TaskFrontPage from '" + this.relativeDir + "utils/wxfp.js'\r\n";
  wx.wxjs += "\r\n";
  wx.wxjs += "Page({\r\n\r\n";

  wx.wxjs += "  data: {\r\n";
  wx.wxjs += "  },\r\n\r\n";
  wx.wxjs += "  onLoad(options) {\r\n";
  wx.wxjs += "    this.setData({\r\n";
  wx.wxjs += "      urlArg: options\r\n";
  wx.wxjs += "    });\r\n";
  wx.wxjs += "    this.tfp = new TaskFrontPage();\r\n";
  wx.wxjs += "    this.tfp.parsePage(";
  wx.wxjs += "this,";
  wx.wxjs += JSON.stringify(pageDataModel, function (key, value) {
    if (["cssFiles", "jsFiles", "jsGlobalVars", "jsFuncs",
      'styles', "styleSheets"].includes(key)) {
      return undefined;
    }
    return value;
  }, "");

  wx.wxjs += ")\r\n\r\n";

  wx.wxjs += "  },\r\n\r\n";
  wx.wxjs += "  onShow(options) {\r\n";
  wx.wxjs += "    this.tfp.showPage();\r\n";
  wx.wxjs += "  },\r\n\r\n";
  wx.wxjs += "  onRefresh() {\r\n";
  wx.wxjs += "    this.tfp.refreshPage();\r\n";
  wx.wxjs += "  },\r\n\r\n";

  wx.wxjs += "  bindinput_controller(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.bindController(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";

  wx.wxjs += "  bindchange_controller(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.bindController(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";
  
  wx.wxjs += "  bindclick_uploadFile(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.uploadFile(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";
  
  wx.wxjs += "  bindclick_openFile(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.openFile(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";
  
  wx.wxjs += "  bindclick_seeBiggerImage(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.seeBiggerImage(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";
  wx.wxjs += "  bindclick_switchDel(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.switchDel(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";
  wx.wxjs += "  bindclick_delFile(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.delFile(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";
  
  
  wx.wxjs += "  bindclick_scanCode(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.scanCode(e,this);\r\n";
  wx.wxjs += "  },\r\n\r\n";
  
    
  wx.wxjs += "  bindclick_toNewPage(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  wx.wxjs += "    this.tfp.toNewPage(e);\r\n";
  wx.wxjs += "  },\r\n\r\n";

  // wx.wxjs += "  bindinput_text(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  // wx.wxjs += "  },\r\n\r\n";

  // wx.wxjs += "  bindchange_switch(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id+\"\"]: e.detail.value\r\n";
  // // wx.wxjs += "      [e.target.id+\"_checked\"]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  // wx.wxjs += "  },\r\n\r\n";

  // wx.wxjs += "  bindchange_datetime(e){\r\n";
  // wx.wxjs += "    this.setData({\r\n";
  // wx.wxjs += "      [e.target.id]: e.detail.value\r\n";
  // wx.wxjs += "    })\r\n";
  // wx.wxjs += "  },\r\n\r\n";

  wx.wxjs += "  bindsubmit_form(e){\r\n"; //catchsubmit_form
  wx.wxjs += "    this.tfp.submit(e.target.id, e.detail.value);\r\n";
  wx.wxjs += "  },\r\n\r\n";


  // service1_onResponse(req, res){

  // },

  wx.wxjs += wx0.wxjs;


  if (pageDataModel.jsFuncs) {
    for (var i = 0; i < pageDataModel.jsFuncs.length; i++) {
      let func = pageDataModel.jsFuncs[i];

      wx.wxjs += "  " + func.name + "(";
      for (var j = 0; j < func.args.length; j++) {
        if (j > 0) wx.wxjs += " , ";
        wx.wxjs += func.args[j].name;
      }
      wx.wxjs += ") {\r\n";
      if (func.code) wx.wxjs += "  " + func.code.replace(/\n/g, "\n  ");
      wx.wxjs += "},\r\n\r\n";
    }
  }

  wx.wxjs += "})\r\n";

  if (!this.createFile || this.isDebug) {
    cb(null, wx);
    return;
  }

  this.createWX(wx, cb);
};


WXFPHandler.prototype.createWX = function (wx, cb) {
  var self = this;
  self.createWXJSONFile(wx.wxjson, function () {
    self.createWXJSFile(wx.wxjs, function () {
      self.createWXMLFile(wx.wxml, function () {
        self.createWXSSFile(wx.wxss, function () {
          cb(null, wx);
        });
      });
    });
  });
};


WXFPHandler.prototype.createWXJSONFile = function (json, cb) {
  logger.log("正在保存" + this.filePath + "的json代码...");
  let wxjsonFilePath = this.dirDist + "/" + this.fileName + ".json";
  var self = this;
  fs.open(wxjsonFilePath, 'w+', (err, fd) => {
    if (err) {
      logger.log(err);
      fs.closeSync(fd);
      cb(err);
      return;
    }
    fs.write(fd, json, (err2) => {
      fs.closeSync(fd);
      if (err2) {
        logger.log(err2);
        cb(err2);
        return;
      }
      logger.log(self.filePath + "的json代码保存完毕。");
      cb(null, json);
    });
  });
};

WXFPHandler.prototype.createWXJSFile = function (js, cb) {
  logger.log("正在保存" + this.filePath + "的js代码...");





  let wxjsFilePath = this.dirDist + "/" + this.fileName + ".js";
  var self = this;
  fs.open(wxjsFilePath, 'w+', (err, fd) => {
    if (err) {
      logger.log(err);
      fs.closeSync(fd);
      cb(err);
      return;
    }
    fs.write(fd, js, (err2) => {
      fs.closeSync(fd);
      if (err2) {
        logger.log(err2);
        cb(err2);
        return;
      }
      logger.log(self.filePath + "的js代码保存完毕。");
      cb(null, js);
    });
  });
};

WXFPHandler.prototype.createWXMLFile = function (wxml, cb) {
  logger.log("正在保存" + this.filePath + "的WXML代码...");
  let wxmlFilePath = this.dirDist + "/" + this.fileName + ".wxml";
  var self = this;
  fs.open(wxmlFilePath, 'w+', (err, fd) => {
    if (err) {
      logger.log(err);
      fs.closeSync(fd);
      cb(err);
      return;
    }
    fs.write(fd, wxml, (err2) => {
      fs.closeSync(fd);
      if (err2) {
        logger.log(err2);
        cb(err2);
        return;
      }
      logger.log(self.filePath + "的WXML代码保存完毕。");
      cb(null, wxml);
    });
  });
};

WXFPHandler.prototype.createWXSSFile = function (wxss, cb) {
  logger.log("正在保存" + this.filePath + "的WXSS代码...");
  let wxssFilePath = this.dirDist + "/" + this.fileName + ".wxss";
  var self = this;
  fs.open(wxssFilePath, 'w+', (err, fd) => {
    if (err) {
      logger.log(err);
      fs.closeSync(fd);
      cb(err);
      return;
    }
    fs.write(fd, wxss, (err2) => {
      fs.closeSync(fd);
      if (err2) {
        logger.log(err2);
        cb(err2);
        return;
      }
      logger.log(self.filePath + "的WXSS代码保存完毕。");
      cb(null, wxss);
    });
  });
};