const Service = require('Service');
const util = require('util');
const utils = require('utils');
const Dao = require('Dao');
const { requestSync } = server.loadModule('/app/dev/service/requestUtils');

const SelectApps = function () {
    Service.call(this);
    this.checkLogin = false;
    this.checkAuthority = false;
};

util.inherits(SelectApps, Service);

module.exports = SelectApps;

/**
 * 根据订单内容创建应用
 * @param  {object} req 服务请求对象
 * @param  {String} req.orgCode 组织编码
 * @param  {String} req.orderNumber 订单编号
 * @param  {Boolean} [req.isRegister] 是否为注册组织时执行订单
 * @param  {object} res 服务响应对象
 */
SelectApps.prototype.process = async function (req, res) {
    try {
        const { orgCode, orderNumber, isRegister } = req;
        if (!orgCode) return this.onLogicError(1, '请提供组织编号！');
        if (!orderNumber) return this.onLogicError(2, '请提供组订单编号！');

        const dao = new Dao(this);
        const orgDao = new Dao(this);
        orgDao.model = server.loadModule('/app/sys/model/org.json');
        const vorgAppDao = new Dao(this);
        vorgAppDao.model = server.loadModule('/app/sys/model/vorg_app.json');

        // 查询组织信息
        const [orgInfo] = await orgDao.querySync({ where: 'code=?', params: [orgCode] });
        if (!orgInfo) return this.onLogicError(4, '当前组织异常，请联系管理员！');

        // 查询订单信息
        const getOrderRes = await requestSync(this, server.config.SaasSystemUrl + '/Service', {
            service: '/app/saas/service/platform/getOrder.js', orgCode, orderNumber
        });
        if (getOrderRes.code != 0) return this.onLogicError(3, getOrderRes.message);
        const orderInfo = getOrderRes.data;
        if (orderInfo.status != 1) return this.onLogicError(4, '当前订单异常，请查看订单状态！');

        // 商品中应用列表
        const appCodeList = [];
        const appDetailDict = {};
        for (const detail of orderInfo.details) {
            const { product_type: type, product_app_codes: appCodes } = detail;
            if (type != 'app') continue;
            const temp = appCodes.split(',');
            for (const item of temp) {
                appCodeList.push(item);
                appDetailDict[item] = detail;
            }
        }

        // 查询所在组织中商品应用
        const sysAppList = await vorgAppDao.querySync({
            where: 'org_id=? and app_code in (?)',
            params: [orgInfo.id, appCodeList]
        });
        const sysAppCodeList = sysAppList.map(app => app.app_code);

        // 当前组织添加应用
        const createAppCodeList = appCodeList.filter(appCode => !sysAppCodeList.includes(appCode));
        if (createAppCodeList.length) {
            // 添加应用分类关联
            await dao.executeSync(`
                insert into sys_org_app_category(code,name,parent_id,org_id,group_apps,icon_code,order_no)
                select code,name,parent_id,? as org_id,group_apps,icon_code,order_no from sys_org_app_category where 
                id in (select category_id from sys_org_app where app_id in (select id from sys_app where code in (?)) and org_id=1) and 
                code not in (select code from sys_org_app_category where org_id=?)`,
                [orgInfo.id, createAppCodeList, orgInfo.id]
            );

            // 添加应用关联
            await dao.executeSync(`
                insert into sys_org_app(org_id,app_id,category_id,status,installer_id,install_time,order_no)
                select ? as org_id,app_id,(select id from sys_org_app_category where code in (select code from sys_org_app_category where id=app.category_id) and org_id=? limit 1) as category_id,status,0 as installer_id,sysdate() install_time,order_no
                from sys_org_app as app where org_id=1 and app_id in (select id from sys_app where code in (?))`,
                [orgInfo.id, orgInfo.id, createAppCodeList]
            );

            // 添加应用权限
            await dao.executeSync(`
                insert into sys_app_auth(org_id,app_id,owner_type,owner_id,terminal)
                select org_id,app_id,owner_type,owner_id,terminal from (
                    select role.org_id,app.id as app_id,app.code as app_code,'role' as owner_type,role.id as owner_id,role.name as role_name,'web' as terminal 
                    from sys_app as app join sys_role as role where app.terminal like 'web'
                    union all 
                    select role.org_id,app.id as app_id,app.code as app_code,'role' as owner_type,role.id as owner_id,role.name as role_name,'phone' as terminal 
                    from sys_app as app join sys_role as role where app.terminal like 'phone'
                    union all 
                    select role.org_id,app.id as app_id,app.code as app_code,'role' as owner_type,role.id as owner_id,role.name as role_name,'pc' as terminal 
                    from sys_app as app join sys_role as role  where app.terminal like 'pc'
                ) t where org_id=? and app_code in (?) and role_name in ('系统管理员', '所有人')`,
                [orgInfo.id, createAppCodeList]
            );
        }

        // 修改应用过期时间和用户数
        for (const appCode of appCodeList) {
            const appDetail = appDetailDict[appCode];
            await dao.executeSync(
                `update sys_org_app set expiration_time=?,user_count=? where org_id=? and app_id=(select id from sys_app where code=? limit 1)`,
                [appDetail.expiration_time, appDetail.user_count, orgInfo.id, appCode]
            );
        }

        // 如果为注册组织时执行订单 创建数据字典和编码规则
        if (isRegister) {
            // 查询当前组织数据字典分类
            const dictypeRows = await dao.executeSync(
                `select * from sys_data_dictype where org_id=? and status=0`,
                [orgInfo.id]
            );

            // 如果没有数据字典则添加
            if (dictypeRows.length == 0) {
                // 查询数据字典分类
                const dataDicTypes = await dao.executeSync(`
                    select a.*,(
                        select group_concat(b.code) as code_full_path
                        from sys_data_dictype as b
                        where id in (
                            SELECT SUBSTRING_INDEX(SUBSTRING_INDEX(a.id_full_path, '/', help_topic_id + 1), '/', -1) AS num
                            FROM mysql.help_topic
                            WHERE help_topic_id < LENGTH(a.id_full_path) - LENGTH(REPLACE(a.id_full_path, '/', '')) + 1
                        )
                    ) as code_full_path,(select code from sys_data_dictype as b where a.parent_id=b.id) as parent_code 
                    from sys_data_dictype as a where org_id=1 and status=0`, []
                );

                // 添加数据字典分类
                await dao.executeSync(`
                    insert into sys_data_dictype(org_id,code,name,status)
                    select ? as org_id,code,name,status
                    from sys_data_dictype where org_id=1 and status=0`,
                    [orgInfo.id]
                );

                // 查询新添加的数据字典分类
                const dataDicTypesNew = await dao.executeSync(
                    `select id,code from sys_data_dictype where org_id=? and status=0`,
                    [orgInfo.id]
                );
                const dict = {};
                for (const dataDicType of dataDicTypesNew) dict[dataDicType.code] = dataDicType.id;

                for (const dataDicType of dataDicTypes) {
                    const parentId = dataDicType.parent_code ? dict[dataDicType.parent_code] : 0;
                    const idFullPath = (() => {
                        const temp = dataDicType.code_full_path.split(',');
                        return '/' + temp.map(item => dict[item]).join('/') + '/';
                    })();
                    await dao.executeSync(
                        `update sys_data_dictype set parent_id=?,id_full_path=? where code=? and org_id=? and status=0`,
                        [parentId, idFullPath, dataDicType.code, orgInfo.id]
                    );
                }

                // 添加数据字典
                await dao.executeSync(`
                    insert into sys_data_dic(org_id,code,name,type_id,type_code,status)
                    select ? as org_id,code,name,(select id from sys_data_dictype where org_id=? and code=type_code) as type_id,type_code,status
                    from sys_data_dic where org_id=1 and status=0 having type_id is not null`,
                    [orgInfo.id, orgInfo.id]
                );
            }

            // 查询当前组织数据字典分类
            const encodingruleRows = await dao.executeSync(
                `select * from sys_encodingrule where org_id=?`,
                [orgInfo.id]
            );

            // 如果没有编码规则则添加
            if (encodingruleRows.length == 0) {
                // 添加编码规则
                await dao.executeSync(`
                    insert into sys_encodingrule(org_id,pk_type,pk_code,clear_mode,update_mode,sn_type)
                    select ? as org_id,pk_type,pk_code,clear_mode,update_mode,sn_type
                    from sys_encodingrule where org_id=1`,
                    [orgInfo.id]
                );

                // 添加编码规则明细
                await dao.executeSync(`
                    insert into sys_encodingrule_item(org_id,encodingrule_id,order_number,encoding_type,encoding_count,encoding_content)
                    select ? as org_id,(select id from sys_encodingrule where org_id=? and pk_code=(select pk_code from sys_encodingrule where org_id=1 and id=encodingrule_id)) as encodingrule_id,order_number,encoding_type,encoding_count,encoding_content
                    from sys_encodingrule_item  where org_id=1 having encodingrule_id is not null`,
                    [orgInfo.id, orgInfo.id]
                );
            }
        }

        this.end(res);

        // 结束服务后重新加载当前组织缓存
        server.cache.loadOrgData(orgInfo.id, function (errCache) {
            if (errCache) return logger.log(errCache);
        });
    } catch (error) {
        this.onError(error);
    }
}