var util = require('util');
var Service = require('Service');
var utils = require('utils');
var Dao = require('Dao');
var Connection = require('Connection');
const fs = require('fs');
const path = require('path');
const { requestMarket } = server.loadModule("/app/dev/service/requestUtils");
const Downloader = require("../../../dev/service/product/downloader");
const Tzip = require("../../../dev/service/product/tzip");
const utilExt = require("../../../dev/service/product/utilExt");

var DownloadSourcecode = function () {
    Service.call(this);
    this.checkLogin = false;
};

util.inherits(DownloadSourcecode, Service);

module.exports = DownloadSourcecode;

/**
 * 更新源码
 * @param   {Object} req 服务请求对象
 * @param   {String} req.appCode 应用编码
 * @param   {String} req.systemCode 系统编码
 */
DownloadSourcecode.prototype.process = async function (req, res) {
    this.runInBackground(res);
    this.serviceInfo = server.tmpBGServices[this.id];
    this.serviceInfo.status = 1;
    this.serviceInfo.process = 0;
    await this.main(req, res);
};

DownloadSourcecode.prototype.main = async function (req, res) {
    try {
        const { appCode, systemCode } = req;
        const _root = process.cwd();

        if (!appCode) throw new Error('请提供应用编码！');
        if (!systemCode) throw new Error('请提供系统编码！');

        // 查询系统配置文件
        const appConfigFilePath = path.join(_root, 'app', systemCode, appCode, 'app.json');
        const appConfig = this.getAppConfig(appConfigFilePath);

        // 查询应用安装前版本信息
        const { versionList } = await this.getPrevInstall(req, appConfig);
        if (versionList.length == 0) throw new Error('未查询到应用源码信息！');

        // 根据相差版本逐一更新
        await this.installVersions(req, versionList);

        this.onImportEnd();
    } catch (error) {
        this.onImportError(error.message);
    }
};

/**
* 获取应用的配置
* @param {String} filePath 文件路径
* @returns {Object|null}
*/
DownloadSourcecode.prototype.getAppConfig = function (filePath) {
    if (!fs.existsSync(filePath)) return null;
    try {
        return JSON.parse(fs.readFileSync(filePath, 'utf8'));
    } catch (error) {
        throw new Error('读取配置文件失败！');
    }
};

/**
 * 获取应用安装前版本信息
 * @param {Object} req 服务请求对象
 * @param {String} req.appCode 应用代码
 * @param {String} req.systemCode 系统编码
 * @param {String} req.userId 用户编号
 * @param {Object} appConfig 应用配置
 * @param {String} appConfig.sourcecodeVersion 源码版本号
 * @returns {Promise} 
 */
DownloadSourcecode.prototype.getPrevInstall = function (req, appConfig) {
    const { appCode, systemCode, userId } = req;
    this.serviceInfo.message = '正在获取应用版本信息...';
    return new Promise((resolve, reject) => {
        requestMarket(this, {
            service: 'dp/service/market/install/getPrevInstallSourcecode.js',
            appCode, systemCode, userId, sourcecodeVersion: appConfig ? appConfig.sourcecodeVersion : null
        }, (_res) => {
            if (_res.code == 0) resolve(_res);
            else throw new Error(_res.message);
        });
    });
};

/**
 * 根据版本列表更新应用
 * @param {Object} req 服务请求对象
 * @param {[Object]} versionList 版本列表
 * @returns {Promise}
 */
DownloadSourcecode.prototype.installVersions = async function (req, versionList) {
    const { appCode, systemCode } = req;
    const self = this;
    const dao = new Dao(this);
    this.serviceInfo.message = `正在准备更新...`;
    dao.onError = function (err) {
        throw err;
    };
    const tzip = new Tzip();

    // 创建应用包存放位置 /tpm/package/[orgCode]/download/
    const downloadPath = (() => {
        const _root = process.cwd();
        let _appPath = path.join(_root, 'tpm');
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, 'package');
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, req.session.orgCode);
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, 'download');
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        return _appPath;
    })();
    // 创建后端文件存放位置 /app/[systemCode]/[appCode]/
    const appPath = (() => {
        const _root = process.cwd();
        let _appPath = path.join(_root, 'app');
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, systemCode);
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, appCode);
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        return _appPath;
    })();
    // 创建前端文件存放位置 /web/app/[systemCode]/[appCode]/
    const webPath = (() => {
        const _root = process.cwd();
        let _appPath = path.join(_root, 'web');
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, 'app');
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, systemCode);
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        _appPath = path.join(_appPath, appCode);
        if (!fs.existsSync(_appPath)) fs.mkdirSync(_appPath);
        return _appPath;
    })();

    // 遍历版本列表
    let versionIndex = 0;
    for (const version of versionList) {
        await download(version);
        const unzipPath = await install(version);
        await runSqlFile(version);
        utilExt.removeFilesSync(unzipPath, true);
        versionIndex++;
    }

    // 修改app.json的sourcecodeVersion
    const packagePath = path.join(appPath, 'app.json');
    const packageStr = fs.readFileSync(packagePath, 'utf8');
    const packageJson = JSON.parse(packageStr);
    packageJson.sourcecodeVersion = versionList[versionList.length - 1].version;
    fs.writeFileSync(packagePath, JSON.stringify(packageJson, null, '\t'));


    // 与市场服务器建立socket连接，拉取应用包到存放位置
    async function download(version) {
        const { version: versionCode, fileCode } = version;
        self.serviceInfo.message = `正在下载${versionCode}版本更新包...`;
        const downloader = new Downloader();
        const downloadFilePath = path.join(downloadPath, fileCode + '.file');
        return new Promise((resolve, reject) => {
            downloader.start({ fileCode, savePath: downloadFilePath });
            downloader.on('downloadProgress', (progress) => {
                self.serviceInfo.process = Math.round((100 / versionList.length) * versionIndex + (100 / versionList.length) * progress * 0.01);
            });
            downloader.on('downloadError', (err) => {
                throw new Error(err);
            });
            downloader.on('downloadFinish', () => {
                downloader.stop();
                resolve();
            });
        });
    }

    // 安装更新包
    async function install(version) {
        const { version: versionCode, fileCode } = version;
        self.serviceInfo.message = `正在安装${versionCode}版本更新包...`;
        const downloadFilePath = path.join(downloadPath, fileCode + '.file');
        return new Promise((resolve, reject) => {
            // 解压应用包到临时目录中
            const unzipPath = path.resolve(downloadFilePath, '../', fileCode);
            tzip.unzip(downloadFilePath, unzipPath, (error) => {
                if (error) throw new Error('文件解压异常！');
                // 将临时目录中的文件按需移动到应用目录中
                utilExt.copyFilesSync(path.join(unzipPath, appCode, 'app'), appPath);
                utilExt.copyFilesSync(path.join(unzipPath, appCode, 'web'), webPath);
                resolve(unzipPath);
            });
        });
    }

    // 执行sql tdm更新sql、上传的sql
    async function runSqlFile(version) {
        const { version: versionCode, fileCode } = version;
        self.serviceInfo.message = `正在执行${versionCode}版本数据库升级语句...`;
        try {
            const dbFilePath = path.join(downloadPath, fileCode, appCode, 'app', 'db');
            const database = JSON.parse(JSON.stringify(server.config.Database));
            database.multipleStatements = true;
            const connection = new Connection(database);
            await connection.openSync();
            for (const filePath of getAllSqlFile(dbFilePath)) {
                const sqlContent = fs.readFileSync(filePath, 'utf8');
                await connection.executeSync(sqlContent, null);
            }
            const dbSqlFilePath = path.join(downloadPath, fileCode, appCode, 'db');
            for (const filePath of getAllSqlFile(dbSqlFilePath)) {
                const sqlContent = fs.readFileSync(filePath, 'utf8');
                await connection.executeSync(sqlContent, null);
            }
            await connection.closeSync();
        } catch (error) {
            console.log(error);
            throw new Error('执行sql文件错误！');
        }
    }

    function getAllSqlFile(dir) {
        let res = [];
        traverse(dir);
        return res;

        function traverse(dir) {
            if(!fs.existsSync(dir)) return;
            fs.readdirSync(dir).forEach((file) => {
                const pathname = path.join(dir, file)
                if (fs.statSync(pathname).isDirectory()) {
                    traverse(pathname);
                } else {
                    if (path.extname(pathname) == '.sql') res.push(pathname);
                }
            })
        }
    }
};

DownloadSourcecode.prototype.pipeSync = function (servicePath, req, res = { code: 0 }) {
    return new Promise((resolve, reject) => {
        this.pipe(require(servicePath), req, res, async (reqU, resU) => {
            if (resU.code == 0) resolve({ reqU, resU });
            else throw new Error(resU.message);
        });
    });
};

DownloadSourcecode.prototype.onImportEnd = async function () {
    this.serviceInfo.status = 2;
    this.serviceInfo.message = '';
    await this.commitDB();
    await this.closeDB();
    this.end({ code: 0 });
};

DownloadSourcecode.prototype.onImportError = function (message) {
    this._importFields = null;
    this._pkFields = null;
    this._keys = null;
    const res = { code: 500, message: message };
    const tmpBGService = server.tmpBGServices[this.id];
    tmpBGService.status = -1;
    tmpBGService.endTime = new Date();
    tmpBGService.error = res;
    this.rollbackDB(res);
    this.end(res);
};

DownloadSourcecode.prototype.commitDB = function () {
    if (!this.connection) return Promise.resolve();
    const pid = this.connection.threadId;
    return new Promise((resolve, reject) => {
        if (!this.connection.isBeginTransaction) return resolve();
        logger.log('[pid:' + pid + ']正在提交数据库事务...');
        this.connection.commit((err) => {
            if (err) throw err;
            resolve();
        });
    });
};

DownloadSourcecode.prototype.rollbackDB = function () {
    if (!this.connection) return Promise.resolve();
    const pid = this.connection.threadId;
    return new Promise((resolve, reject) => {
        if (!this.connection.isBeginTransaction) return resolve();
        this.connection.rollback(async (err) => {
            if (err) logger.log(err);
            else logger.log('[pid:' + pid + ']数据库事务已回滚！');
            await this.closeDB();
            resolve();
        });
    });
};

DownloadSourcecode.prototype.closeDB = function () {
    if (!this.connection) return Promise.resolve();
    const pid = this.connection.threadId;
    return new Promise((resolve, reject) => {
        logger.log('[pid:' + pid + ']关闭数据库连接。');
        this.connection.end();
        this.connection = null;
        resolve();
    });
};