var Stream = require('stream');
var util = require('util');

/**
 * 工作状态
 */
var ST_HEAD = 1;        // 等待解析包头
var ST_BODY = 2;        // 等待解析包体
var ST_CLOSED = 3;      // 已关闭
var ST_STOPLISTEN = 4;  // 停止在本类中监听，由其他代码接管

/**
 * TCP网络连接
 *
 * @param {Object} socket 原生的Socket对象
 */
var TcpSocket = function(socket) {
  if(!(this instanceof TcpSocket)) {
    return new TcpSocket(socket);
  }

  if(!socket) {
    throw new Error('无效的socket。');
  }

  // stream style interfaces.
  Stream.call(this);
  this.readable = true;
  this.writeable = true;
  this.sessionId = null;
  this.remoteAddress = socket.remoteAddress;
  this.remotePort = socket.remotePort;

  this._socket = socket;
  this.headSize = 4;
  this.headBuffer = new Buffer(this.headSize);

  this.headOffset = 0;
  this.packageOffset = 0;
  this.packageSize = 0;
  this.packageBuffer = null;

  this._socket.on('data', ondata.bind(null, this));
  this._socket.on('end', onend.bind(null, this));
  this._socket.on('error', onerror.bind(null, this));
  this._socket.on('timeout', ontimeout.bind(null, this));
  this._socket.on('close', this.emit.bind(this, 'close'));

  this.state = ST_HEAD;

  this.reqMsgIndex = 0;               //请求消息索引
  this.reqCbs = {};                   //请求回调集合
};

util.inherits(TcpSocket, Stream);

module.exports = TcpSocket;

var PKG_HEAD_BYTES = 4;

var Package = TcpSocket.Package = {};
Package.TYPE_HEARTBEAT = 1;  //心跳
Package.TYPE_PUSH      = 2;  //推送消息，不需要返回结果
Package.TYPE_REQUEST   = 3;  //请求消息，需要返回结果
Package.TYPE_RESPONSE  = 4;  //响应消息

TcpSocket.prototype.stopListen = function() {
  this.state = ST_STOPLISTEN;
};

/**
 * 编码数据包
 * @param  {Number} type 包类型
 * @param  {Object} body 包体
 * @return {Buffer}      编码后的数据包
 */
TcpSocket.encode = function(type, body){
  var length = body ? body.length : 0;
  var buffer = new Buffer(PKG_HEAD_BYTES + length);
  var index = 0;
  buffer[index++] = type & 0xff;
  buffer[index++] = (length >> 16) & 0xff;
  buffer[index++] = (length >> 8) & 0xff;
  buffer[index++] = length & 0xff;
  if(body) {
    TcpSocket.copyArray(buffer, index, body, 0, length);
  }
  return buffer;
};

/**
 * 复制数组
 * @param  {Array}  dest    目标数组
 * @param  {Number} doffset 目标数组起始位置
 * @param  {Array}  src     源数组
 * @param  {Number} soffset 源数组起始位置
 * @param  {Number} length  长度
 */
TcpSocket.copyArray = function(dest, doffset, src, soffset, length) {
  if('function' === typeof src.copy) {
    // Buffer
    src.copy(dest, doffset, soffset, soffset + length);
  } else {
    // Uint8Array
    for(var index=0; index<length; index++){
      dest[doffset++] = src[soffset++];
    }
  }
};

/**
 * 发送数据包
 * @param  {Object}   type 包类型
 * @param  {Object}   body 包体
 * @param  {Function} cb   回调函数
 */
TcpSocket.prototype.sendPackage = function(type, body, cb) {
  if(this.state === ST_CLOSED) return;
  var buffer = null;
  var str = '';
  if(body) {
    if(typeof body === 'string') {
      str = body;
    }
    //如果响应对象定义了格式化函数，则调用该函数
    else if(body.formatFunc) {
      str = JSON.stringify(body, body.formatFunc);
    } else {
      str = JSON.stringify(body);
    }
    buffer = TcpSocket.encode(type, new Buffer(str));
  } else {
    buffer = TcpSocket.encode(type);
  }
  //if(type!=Package.TYPE_HEARTBEAT) 
  //logger.log('发送数据：'+type+' '+buffer.length+' '+str, this._id);
  var self = this;
  if(this._socket.destroyed) return;
  try {
    this._socket.write(buffer, function(err){
      buffer = null;
      if(err) {
        if(err.message.indexOf('ECONNRESET')<0 
          && err.message.indexOf('ECONNABORTED')<0) 
          logger.log(err, self._id);
        self.close();
      } else {
        if(str!='') {
          self.emit('send', str);
        }
      }
      str = null;
      if(cb) {
        if(err && err.message.indexOf('ECONNRESET')<0 
          && err.message.indexOf('ECONNABORTED')<0) {
          cb(err);
        } else {
          cb();
        }
      }
    });
  } catch(err) {
    if(err.message.indexOf('ECONNRESET')<0 
      && err.message.indexOf('ECONNABORTED')<0) 
      logger.log(err, this._id);
  }
};

/**
 * 发送推送消息
 * @param  {Object}   msg 消息
 * @param  {Function} cb  回调函数
 */
TcpSocket.prototype.push = function(msg, cb) {
  this.sendPackage(Package.TYPE_PUSH, msg, cb);
};

/**
 * 发送请求消息
 * @param  {Object}   msg 消息
 * @param  {Function} cb  回调函数
 */
TcpSocket.prototype.request = function(service, msg, cb) {
  this.reqMsgIndex++;
  var reqId = this.reqMsgIndex;
  this.reqCbs[reqId] = cb;
  this.sendPackage(Package.TYPE_REQUEST, [reqId, service, msg]);
};

/**
 * 发送响应消息
 * @param  {Object}   msg 消息
 * @param  {Function} cb  回调函数
 */
TcpSocket.prototype.response = function(reqId, code, message, cb) {
  var res = [reqId, code];
  if(message) res.push(message);
  this.sendPackage(Package.TYPE_RESPONSE, res, cb);
};

/**
 * 发送心跳包
 */
TcpSocket.prototype.sendHeartbeat = function(cb) {
  var self = this;
  setTimeout(function() {
    //logger.log('发送心跳包。');
    if(self.state !== ST_CLOSED) {
      self.sendPackage(Package.TYPE_HEARTBEAT, null, cb);
    }
  }, 1000);
};

/**
 * 发送消息
 * @param  {Object}   msg 消息
 * @param  {Function} cb  回调函数
 */
TcpSocket.prototype.write = function(msg, cb) {
  var self = this;
  this._socket.write(new Buffer(msg), function(err){
    if(err) {
      if(err.message.indexOf('ECONNRESET')<0) {
        logger.log(err, self._id);
        if(cb) cb(err);
      } else {
        if(cb) cb();
      }
      self.close();
      return;
    }
    if(cb) cb();
  });
};

/**
 * 发送响应信息，并关闭网络连接
 * @param  {Object}   msg 消息
 * @param  {Function} cb  回调函数
 */
TcpSocket.prototype.end = function(reqId, msg, cb) {
  if(this.state == ST_CLOSED) return;
  var self = this;
  if(msg) {
    this.sendPackage(Package.TYPE_RESPONSE, [reqId, 0, msg], function(err) {
      self.close();
      msg = null;
      if(cb) cb(err);
    });
  } else {
    this.close();
    if(cb) cb();
  }
};

/**
 * 发生错误时，将错误信息发送给客户端，然后关闭网络连接
 * @param  {Object}   msg 消息
 */
TcpSocket.prototype.endOnError = function(reqId, err, code) {
  if(this.state === ST_CLOSED) return;
  var res = [reqId, 500];
  if(code) res[1] = code;
  if(typeof err !== 'string' && err.message) {
    res.push(err.message);
    logger.log(err, this._id);
  } else {
    res.push(err);
    logger.log(err, this._id);
  }
  var self = this;
  this.sendPackage(Package.TYPE_RESPONSE, res, function(err2) {
    res = null;
    self.close();
  });
};

/**
 * 关闭连接
 */
TcpSocket.prototype.close = function() {
  if(this.state === ST_CLOSED) return;
  this.state = ST_CLOSED;
  try { 
    logger.log('关闭TCP连接。', this._id);
    this._socket.destroy(); 
    this._socket = null;
  } catch (e) {
  }
  this._socket = null;
};

TcpSocket.prototype.isClose = function() {
  if(this.state == ST_CLOSED) return true;
  return false;
};

/**
 * 解析数据
 * @param  {Object} tcpSocket 网络连接对象
 * @param  {Object} chunk  要解析的数据
 * @return {Boolean}       处理结果
 */
var ondata = function(tcpSocket, chunk) {
  if(tcpSocket.state === ST_CLOSED || tcpSocket.state === ST_STOPLISTEN) {
    return;
  }

  if(typeof chunk !== 'string' && !Buffer.isBuffer(chunk)) {
    logger.log('非法访问，数据包格式无效。', tcpSocket._id);
    tcpSocket.close();
    return;
  }

  if(typeof chunk === 'string') {
    chunk = new Buffer(chunk, 'utf8');
  }

  var offset = 0, end = chunk.length;

  if(tcpSocket.state === ST_HEAD && tcpSocket.packageBuffer === null && !checkTypeData(chunk[0])) {
    logger.log('非法访问，数据包类型无效。', tcpSocket._id);
    tcpSocket.close();
    return;
  }

  tcpSocket.lastReceiveDataTime = new Date();

  while(offset < end) {
    if(tcpSocket.state === ST_HEAD) {
      offset = readHead(tcpSocket, chunk, offset);
    }

    if(tcpSocket.state === ST_BODY) {
      offset = readBody(tcpSocket, chunk, offset);
    }
  }

  return true;
};

/**
 * 解析结束
 * @param  {Object} tcpSocket 网络连接对象
 * @param  {Object} chunk    数据
 */
var onend = function(tcpSocket, chunk) {
  if(chunk) {
    tcpSocket._socket.write(chunk);
  }
  reset(tcpSocket);
  tcpSocket.state = ST_CLOSED;
  tcpSocket = null;
};

/**
 * 发生错误时
 * @param  {Object} tcpSocket 网络连接对象
 * @param  {Object} err       错误对象
 */
var onerror = function(tcpSocket, err) {
  if(err.message.indexOf('ECONNRESET')<0) logger.log(err, tcpSocket._id);
  tcpSocket.close();
};

/**
 * 连接超时
 * @param  {Object} tcpSocket 网络连接对象
 */
var ontimeout = function(tcpSocket) {
  logger.log('连接超时。', tcpSocket._id);
  tcpSocket.close();
};

/**
 * 解析包头
 *
 * @param  {Object} tcpSocket 网络连接对象
 * @param  {Object} data      数据缓存
 * @param  {Number} offset    读取包头数据的起始位置
 * @return {Number}           新的起始位置
 */
var readHead = function(tcpSocket, data, offset) {
  var hlen = tcpSocket.headSize - tcpSocket.headOffset;
  var dlen = data.length - offset;
  var len = Math.min(hlen, dlen);
  var dend = offset + len;

  data.copy(tcpSocket.headBuffer, tcpSocket.headOffset, offset, dend);
  tcpSocket.headOffset += len;

  if(tcpSocket.headOffset === tcpSocket.headSize) {
    var size = getBodySize(tcpSocket.headBuffer);
    if(size < 0) {
      logger.log('非法访问，数据包长度无效。', tcpSocket._id);
      tcpSocket.close();
      return;
    }
    tcpSocket.packageSize = size + tcpSocket.headSize;
    tcpSocket.packageBuffer = new Buffer(tcpSocket.packageSize);
    tcpSocket.headBuffer.copy(tcpSocket.packageBuffer, 0, 0, tcpSocket.headSize);
    tcpSocket.packageOffset = tcpSocket.headSize;
    tcpSocket.state = ST_BODY;
  }

  return dend;
};

/**
 * 解析包体
 *
 * @param  {Object} tcpSocket 网络连接对象
 * @param  {Object} data      数据缓存
 * @param  {Number} offset    读取包体数据的起始位置
 * @return {Number}           新的起始位置
 */
var readBody = function(tcpSocket, data, offset) {
  var blen = tcpSocket.packageSize - tcpSocket.packageOffset;
  var dlen = data.length - offset;
  var len = Math.min(blen, dlen);
  var dend = offset + len;

  data.copy(tcpSocket.packageBuffer, tcpSocket.packageOffset, offset, dend);

  tcpSocket.packageOffset += len;

  if(tcpSocket.packageOffset === tcpSocket.packageSize) {
    var buffer = tcpSocket.packageBuffer;
    var package = {};
    package.type = buffer[0];
    package.bodySize = buffer.length - tcpSocket.headSize;
    var str = '';
    if(package.bodySize>0) {
      var body = new Buffer(package.bodySize);
      tcpSocket.packageBuffer.copy(body, 0, tcpSocket.headSize, tcpSocket.packageSize);
      str = body.toString('utf-8');
      try {
        package.body = JSON.parse(str);
      } catch(e) {
        logger.log('非法访问，数据包内容格式无效：'+str, tcpSocket._id);
        tcpSocket.close();
        return;
      }
    } else {
      package.body = {};
    }
    if(package.type!=Package.TYPE_HEARTBEAT) tcpSocket.emit('receive', package.body);
    if(package.type==Package.TYPE_HEARTBEAT) {
      tcpSocket.emit('heartbeat');
    } else if(package.type==Package.TYPE_PUSH) {
      tcpSocket.emit('push', package.body);
    } else if(package.type==Package.TYPE_REQUEST) {
      if((package.body instanceof Array) && package.body.length>=2) {
        var req = {};
        if(package.body.length>=3 && package.body[2]) {
          if(package.body[2] instanceof Array) {
            req.args = package.body[2];
          } else if(package.body[2] instanceof Object) {
            req = package.body[2];
          } else {
            req.args = [package.body[2]];
          }
        }
        req._id = package.body[0];
        req._type = package.body[1];
        tcpSocket.emit('request', req);
      } else {
        logger.log('收到无效的请求消息：');
        logger.log(package.body);
      }
    } else if(package.type==Package.TYPE_RESPONSE) {
      if((package.body instanceof Array) && package.body.length>=2) {
        var cb = tcpSocket.reqCbs[package.body[0]];
        if(cb) {
          var res = {};
          if(package.body[1]==0) {
            if(package.body.length>=3 && package.body[2]) {
              if(package.body[2] instanceof Array) {
                res.args = package.body[2];
              } else if(package.body[2] instanceof Object) {
                res = package.body[2];
              } else {
                res.args = [package.body[2]];
              }
            }
          } else {
            if(package.body.length>=3) res.message = package.body[2];
          }
          res.code = package.body[1];
          try {
            cb(res);
          } catch(err) {
            logger.log('执行处理响应消息的回调函数时出错：'+err.message);
          }
        }
      } else {
        logger.log('收到无效的响应消息：');
        logger.log(package.body);
      }
    }
    reset(tcpSocket);
  }

  return dend;
};

/**
 * 重置网络连接参数
 * @param  {Connecter} tcpSocket 网络连接对象
 */
var reset = function(tcpSocket) {
  tcpSocket.headOffset = 0;
  tcpSocket.packageOffset = 0;
  tcpSocket.packageSize = 0;
  tcpSocket.packageBuffer = null;
  //发现在某些服务里，调用stopListen方法后，还在解析包头
  //查找原因，发现是因为服务里的代码执行得太快，还没来得及reset就执行完了
  //导致最后连接状态又变成ST_HEAD了，所以需要在此加上状态判断
  if(tcpSocket.state === ST_CLOSED || tcpSocket.state === ST_STOPLISTEN) return;
  tcpSocket.state = ST_HEAD;
};

/**
 * 校验数据包类型
 * @param  {Number} data 数据类型
 */
var checkTypeData = function(dataType) {
  return dataType === Package.TYPE_HEARTBEAT 
  || dataType === Package.TYPE_PUSH 
  || dataType === Package.TYPE_REQUEST
  || dataType === Package.TYPE_RESPONSE;
};

/**
 * 从包头信息中获得包体大小
 * @param  {Object} headBuffer 包头信息
 * @return {Number}            包体大小
 */
var getBodySize = function(headBuffer) {
  var len = 0;
  for(var i=1; i<4; i++) {
    if(i > 1) {
      len <<= 8;
    }
    len += headBuffer.readUInt8(i);
  }
  return len;
};