var util = require('util');
var Service = require('Service');
var utils = require('utils');
var Dao = require('Dao');
var Connection = require('Connection');
const fs = require('fs');
const path = require('path');
const { requestMarket } = server.loadModule("/app/dev/service/requestUtils");
const Downloader = require("../../../../dev/service/product/downloader");
const Tzip = require("../../../../dev/service/product/tzip");
const utilExt = require("../../../../dev/service/product/utilExt");

var DownloadApp = function () {
    Service.call(this);
    this.checkLogin = false;
};

util.inherits(DownloadApp, Service);

module.exports = DownloadApp;

/**
 * 下载应用
 * @param   {Object} req 服务请求对象
 * @param   {String} req.appCode 应用编码
 * @param   {String} req.categoryId 应用所属类别
 */
DownloadApp.prototype.process = async function (req, res) {
    this.runInBackground(res);
    this.serviceInfo = server.tmpBGServices[this.id];
    this.serviceInfo.status = 1;
    this.serviceInfo.process = 0;
    await this.main(req, res);
};

DownloadApp.prototype.main = async function (req, res) {
    try {
        const { appCode } = req;
        const _root = process.cwd();

        if (!appCode) return this.onImportError('请提供应用编码！');

        // 查询应用安装前应用信息
        const { appInfo } = await this.getAppInfo(req);
        this.appInfo = appInfo;
        this.compiler = appInfo.compiler == 2;

        // 查询系统是否存在此应用
        const appConfigFilePath = path.join(_root, this.compiler ? 'dist' : 'app', this.appInfo.system_code, appCode, 'app.json');
        const appConfig = this.getAppConfig(appConfigFilePath);

        // 查询应用安装前版本信息
        const { versionList } = await this.getPrevInstall(req, appConfig);
        if (versionList.length == 0) return this.onImportError('未查询到应用安装信息！');

        // 根据相差版本逐一更新
        await this.installVersions(req, versionList);

        // 更新系统应用表
        await this.installApps(req);

        // 更新Tasgine.json的ProjectConfig，同时更新缓存
        if (this.compiler) this.updateProjectConfig(this.appInfo.system_code, appCode);

        // 更新市场应用信息
        requestMarket(this, {
            service: 'dp/service/market/detail/downloadApp.js',
            appId: appInfo.id
        }, (_res) => {
            this.onImportEnd();
        });
    } catch (error) {
        console.log(error);
        return this.onImportError(error.message);
    }
};

/**
 * 获取应用的配置
 * @param {String} filePath 文件路径
 * @returns {Object|null}
 */
DownloadApp.prototype.getAppConfig = function (filePath) {
    if (!fs.existsSync(filePath)) return null;
    try {
        return JSON.parse(fs.readFileSync(filePath, 'utf8'));
    } catch (error) {
        this.onImportError('读取配置文件失败！');
    }
};

/**
 * 获取应用安装前应用信息
 * @param {Object} req 服务请求对象
 * @param {String} req.appCode 应用代码
 * @returns {Promise} 
 */
DownloadApp.prototype.getAppInfo = function (req) {
    const { appCode } = req;
    return new Promise((resolve, reject) => {
        requestMarket(this, {
            service: 'dp/service/market/install/getAppInfo.js',
            appCode
        }, (_res) => {
            if (_res.code == 0) resolve(_res);
            else return this.onImportError(_res.message);
        });
    });
};

/**
 * 获取应用安装前版本信息
 * @param {Object} req 服务请求对象
 * @param {String} req.appCode 应用代码
 * @param {Object} appConfig 应用配置
 * @param {String} appConfig.version 版本号
 * @returns {Promise} 
 */
DownloadApp.prototype.getPrevInstall = function (req, appConfig) {
    const { appCode } = req;
    this.serviceInfo.message = '正在获取应用版本信息...';
    return new Promise((resolve, reject) => {
        requestMarket(this, {
            service: 'dp/service/market/install/getPrevInstall.js',
            appCode, version: appConfig ? appConfig.version : null
        }, (_res) => {
            if (_res.code == 0) resolve(_res);
            else return this.onImportError(_res.message);
        });
    });
};

/**
 * 根据版本列表更新应用
 * @param {Object} req 服务请求对象
 * @param {[Object]} versionList 版本列表
 * @returns {Promise}
 */
DownloadApp.prototype.installVersions = async function (req, versionList) {
    const self = this;
    const dao = new Dao(this);
    this.serviceInfo.message = `正在准备更新...`;
    dao.onError = function (err) {
        self.onImportError(err.message);
    };
    const { appCode } = req;
    const { system_code: systemCode, system_name: systemName } = this.appInfo;
    const tzip = new Tzip();
    const _root = process.cwd();
    // 创建应用包存放位置 /tpm/package/[orgCode]/download/
    const downloadPath = path.join(_root, 'tpm', 'package', req.session.orgCode, 'download');
    utilExt.mkdirsSync(downloadPath);
    // 创建后端文件存放位置 /dist|app/[systemCode]/[appCode]/
    const appPath = path.join(_root, this.compiler ? 'dist' : 'app', this.appInfo.system_code, appCode);
    utilExt.mkdirsSync(appPath);
    // 创建前端文件存放位置 /web/dist|app/[systemCode]/[appCode]/
    const webPath = path.join(_root, 'web', this.compiler ? 'dist' : 'app', this.appInfo.system_code, appCode);
    utilExt.mkdirsSync(appPath);

    // 遍历版本列表
    let unzipPath = '';
    let versionIndex = 0;
    for (const version of versionList) {
        await download(version);
        unzipPath = await install(version);
        await runSqlFile(version);
        if (versionList.length - 1 == versionIndex) {
            // 复制最后一个版本的app.json
            fs.copyFileSync(path.join(unzipPath, appCode, 'app.json'), path.join(appPath, 'app.json'));
        }
        utilExt.removeFilesSync(unzipPath, true);
        versionIndex++;
    }

    // 创建org.json文件
    const orgJsonPath = path.join(path.resolve(appPath, '../'), 'org.json');
    fs.writeFileSync(orgJsonPath, JSON.stringify({ code: systemCode, name: systemName }));

    // 与市场服务器建立socket连接，拉取应用包到存放位置
    function download(version) {
        const { version: versionCode, fileCode } = version;
        self.serviceInfo.message = `正在下载${versionCode}版本更新包...`;
        const downloader = new Downloader();
        const downloadFilePath = path.join(downloadPath, fileCode + '.file');
        return new Promise((resolve, reject) => {
            downloader.start({ fileCode, savePath: downloadFilePath });
            downloader.on('downloadProgress', (progress) => {
                self.serviceInfo.process = Math.round((100 / versionList.length) * versionIndex + (100 / versionList.length) * progress * 0.01);
            });
            downloader.on('downloadError', (err) => {
                return self.onImportError(err);
            });
            downloader.on('downloadFinish', () => {
                downloader.stop();
                resolve();
            });
        });
    }

    // 安装更新包
    function install(version) {
        const { version: versionCode, fileCode } = version;
        self.serviceInfo.message = `正在安装${versionCode}版本更新包...`;
        const downloadFilePath = path.join(downloadPath, fileCode + '.file');
        return new Promise((resolve, reject) => {
            // 解压应用包到临时目录中
            const unzipPath = path.resolve(downloadFilePath, '../', fileCode)
            tzip.unzip(downloadFilePath, unzipPath, (error) => {
                if (error) {
                    console.log(error);
                    return self.onImportError('文件解压异常！');
                }
                // 将临时目录中的文件按需移动到应用目录中
                utilExt.copyFilesSync(path.join(unzipPath, appCode, 'app'), appPath);
                utilExt.copyFilesSync(path.join(unzipPath, appCode, 'web'), webPath);
                resolve(unzipPath);
            });
        });
    }

    // 执行sql tdm更新sql、上传的sql(update_[versionId].sql)
    async function runSqlFile(version) {
        const { version: versionCode, fileCode, sqlFileList } = version;
        self.serviceInfo.message = `正在执行${versionCode}版本数据库升级语句...`;
        try {
            const appFilePath = path.join(downloadPath, fileCode, appCode);
            const dbSqlFilePath = path.join(appFilePath, 'db');
            const database = JSON.parse(JSON.stringify(server.config.Database));
            database.multipleStatements = true;
            const connection = new Connection(database);
            await connection.openSync();
            // for (const fileName of sqlFileList) {
            //     const filePath = path.join(appFilePath, 'db', fileName);
            //     if (!fs.existsSync(filePath)) continue;
            //     const sqlContent = fs.readFileSync(filePath, 'utf8');
            //     await connection.executeSync(sqlContent, null);
            // }
            for (const filePath of getAllSqlFile(dbSqlFilePath)) {
                const sqlContent = fs.readFileSync(filePath, 'utf8');
                await connection.executeSync(sqlContent, null);
            }
            await connection.closeSync();
        } catch (error) {
            console.log(error);
            return self.onImportError('执行sql文件错误！');
        }
    }

    function getAllSqlFile(dir) {
        let res = [];
        traverse(dir);
        return res;

        function traverse(dir) {
            if (!fs.existsSync(dir)) return;
            fs.readdirSync(dir).forEach((file) => {
                const pathname = path.join(dir, file)
                if (fs.statSync(pathname).isDirectory()) {
                    traverse(pathname);
                } else {
                    if (path.extname(pathname) == '.sql') res.push(pathname);
                }
            })
        }
    }
};

/**
 * 更新系统应用表
 * @param {Object} req 服务请求对象
 * @param {Number} [req.categoryId] 应用分类
 */
DownloadApp.prototype.installApps = async function (req) {
    const { categoryId } = req;
    const { appInfo } = this;
    this.serviceInfo.message = `正在更新应用信息...`;
    const self = this;
    const dao = new Dao(this);
    dao.onError = function (err) {
        self.onImportError(err.message);
    };
    const appCode = appInfo.code;
    const sysAppList = await dao.executeSync(`select *,(select category_id from sys_org_app where app_id=sys_app.id) as category from sys_app where code like ?`, [appCode + '/%']);
    const marketAppList = appInfo.apps_config ? JSON.parse(appInfo.apps_config) : [];
    const sysAppCodeDict = {};
    const sysAppNameDict = {};
    sysAppList.forEach(app => { sysAppCodeDict[app.code] = app; sysAppNameDict[app.name] = app; });
    const marketAppDict = {};
    
    // 添加应用
    for (const marketAppInfo of marketAppList) {
        const appCode = appInfo.code + '/' + marketAppInfo.code;
        const appName = marketAppInfo.name;
        const appKey = appCode + '_' + appName;
        marketAppDict[appKey] = marketAppInfo;
        if (sysAppCodeDict[appCode] || sysAppNameDict[appName]) continue;
        await this.pipeSync('/app/sys/service/app/add', changeAppInfo(marketAppInfo));
    }

    // 修改应用
    for (const sysAppInfo of sysAppList) {
        const appKey = sysAppInfo.code + '_' + sysAppInfo.name;
        const marketAppInfo = marketAppDict[appKey];
        if (!marketAppInfo) continue;
        await this.pipeSync('/app/sys/service/app/update', changeAppInfo(marketAppInfo, sysAppInfo));
        delete sysAppCodeDict[sysAppInfo.code];
        delete sysAppNameDict[sysAppInfo.name];
    }

    // 禁用应用
    for (const appCode of Object.keys(sysAppCodeDict)) {
        const sysAppInfo = sysAppCodeDict[appCode];
        await this.pipeSync('/app/sys/service/app/disable', { session: req.session, id: sysAppInfo.id });
    }

    function changeAppInfo(marketAppInfo, sysAppInfo = {}) {
        return {
            id: sysAppInfo.id,
            order_no: sysAppInfo.order_no || sysAppInfo.id,
            code: appInfo.code + '/' + marketAppInfo.code,
            name: marketAppInfo.name,
            categoryId: categoryId || sysAppList[0].category,
            icon_path: marketAppInfo.icon,
            deploy: "url",
            terminal: marketAppInfo.terminal,
            version: appInfo.cur_version,
            developer_id: req.session.userId,
            url_web: marketAppInfo.webIndexUrl,
            url_web_detail: marketAppInfo.webDetailUrl,
            url_phone: marketAppInfo.phoneIndexUrl,
            url_phone_detail: marketAppInfo.phoneDetailUrl,
            titlebar_color: marketAppInfo.phoneDetailBarColor,
            // "pc_open_type": "1",
            // "pc_show_on_sysbar": "1",
            // "msg_groupc": "1",
            session: req.session
        }
    }
}

/**
 * 更新Tasgine.json的ProjectConfig，同时更新缓存
 * @param {String} systemCode 系统编码
 * @param {String} appCode 应用编码
 */
DownloadApp.prototype.updateProjectConfig = function (systemCode, appCode) {
    const _root = process.cwd();
    const configPath = path.join(_root, 'config', 'Tasgine.json');
    const content = fs.readFileSync(configPath, { encoding: 'utf8' });
    const config = JSON.parse(content);
    const filePath = `${systemCode}/${appCode}`;
    const item = { compile: true, files: ['/app/' + filePath], encrypts: [] }
    if (!config['ProjectConfig']) config['ProjectConfig'] = {};
    config['ProjectConfig'][filePath] = item;
    if (!server.config['ProjectConfig']) server.config['ProjectConfig'] = {};
    server.config['ProjectConfig'][filePath] = item;
    fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
}

DownloadApp.prototype.pipeSync = function (servicePath, req, res = { code: 0 }) {
    console.log(servicePath);
    return new Promise((resolve, reject) => {
        this.pipe(server.loadModule(servicePath), req, res, async (reqU, resU) => {
            if (resU.code == 0) resolve({ reqU, resU });
            else this.onImportError(resU.message);
        });
    });
};

DownloadApp.prototype.onImportEnd = async function () {
    this.serviceInfo.status = 2;
    await this.commitDB();
    await this.closeDB();
    this.end({ code: 0 });
};

DownloadApp.prototype.onImportError = function (message) {
    this._importFields = null;
    this._pkFields = null;
    this._keys = null;
    const res = { code: 500, message: message };
    const tmpBGService = server.tmpBGServices[this.id];
    tmpBGService.status = -1;
    tmpBGService.endTime = new Date();
    tmpBGService.error = res;
    this.rollbackDB(res);
    this.end(res);
};

DownloadApp.prototype.commitDB = function () {
    if (!this.connection) return Promise.resolve();
    const pid = this.connection.threadId;
    return new Promise((resolve, reject) => {
        if (!this.connection.isBeginTransaction) return resolve();
        logger.log('[pid:' + pid + ']正在提交数据库事务...');
        this.connection.commit((err) => {
            if (err) {
                logger.log(err);
                this.onImportError(err.message);
            }
            resolve();
        });
    });
};

DownloadApp.prototype.rollbackDB = function () {
    if (!this.connection) return Promise.resolve();
    const pid = this.connection.threadId;
    return new Promise((resolve, reject) => {
        if (!this.connection.isBeginTransaction) return resolve();
        this.connection.rollback(async (err) => {
            if (err) logger.log(err);
            else logger.log('[pid:' + pid + ']数据库事务已回滚！');
            await this.closeDB();
            resolve();
        });
    });
};

DownloadApp.prototype.closeDB = function () {
    if (!this.connection) return Promise.resolve();
    const pid = this.connection.threadId;
    return new Promise((resolve, reject) => {
        logger.log('[pid:' + pid + ']关闭数据库连接。');
        this.connection.end();
        this.connection = null;
        resolve();
    });
};